{*
 * Outliner Lighto
 * Copyright (C) 2011 Kostas Michalopoulos
 *
 * This software is provided 'as-is', without any express or implied
 * warranty.  In no event will the authors be held liable for any damages
 * arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not
 *    claim that you wrote the original software. If you use this software
 *    in a product, an acknowledgment in the product documentation would be
 *    appreciated but is not required.
 * 2. Altered source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 3. This notice may not be removed or altered from any source distribution.
 *
 * Kostas Michalopoulos <badsector@runtimeterror.com>
 *}
unit UI;
interface
{$MODE OBJFPC}{$H+}
uses Video, Keyboard, Defines;
var
  FColor, BColor, CAttr: Integer;
  CurX, CurY: Integer;
  DoWrites: Boolean;

procedure GotoXY(X, Y: Integer);
procedure InvertCharAt(X, Y: Integer);
procedure Color(F, B: Integer);
procedure WriteChar(Ch: Char);
procedure WriteStr(Str: string);
procedure Top(s: string);
procedure DefaultTop;
procedure Status(s: string);
procedure ClearEOL;
procedure ClrTo(x1, x2, y: Integer);
procedure ClearBackScreen;
function Input(x, y, Max: Integer; var Value: string): Boolean;
function Select(Title: string; Choices: array of string; Index: Integer): Integer;
procedure FullScreenText(Title, Text: string; QuitChar: Char='Q');

implementation

procedure GotoXY(X, Y: Integer);
begin
  if X < 0 then X:=0 else if X > ScreenWidth - 1 then X:=ScreenWidth - 1;
  if Y < 0 then Y:=0 else if Y > ScreenHeight - 1 then Y:=ScreenHeight - 1;
  CurX:=X;
  CurY:=Y;
end;

procedure InvertCharAt(X, Y: Integer);
var
  A: Integer;
begin
  if (X < 0) or (Y < 0) or (X >= ScreenWidth) or (Y >= ScreenHeight) then Exit;
  A:=Y*ScreenWidth + X;
  if DoWrites then begin
    VideoBuf^[A]:=(VideoBuf^[A] and $FF) or ((not (VideoBuf^[A] shr 8)) shl 8);
    UpdateScreen(False);
  end;
end;

procedure Color(F, B: Integer);
begin
  FColor:=F;
  BColor:=B;
  CAttr:=F + (B shl 4);
end;

procedure WriteChar(Ch: Char);
var
  A: Integer;
begin
  A:=CurY*ScreenWidth + CurX;
  if DoWrites then
    VideoBuf^[A]:=Byte(Ch) or (CAttr shl 8);
  if CurX < ScreenWidth - 1 then Inc(CurX);
end;

procedure WriteStr(Str: string);
var
  i: Integer;
begin
  for i:=1 to Length(Str) do WriteChar(Str[i]);
end;

procedure Top(s: string);
begin
  Color(0, 7);
  GotoXY(0, 0);
  WriteStr(s);
  ClearEOL;
end;

procedure DefaultTop;
begin
  Top('Outliner Lighto ' + VERSION + '    Copyright (C) 2011-2019 Kostas "Bad Sector" Michalopoulos');
end;

procedure Status(s: string);
var
  i: Integer;
  Hi: Boolean;
begin
  Hi:=False;
  Color(0, 7);
  GotoXY(0, ScreenHeight - 1);
  for i:=1 to Length(s) do begin
    if s[i]='~' then begin
      Hi:=not Hi;
      if Hi then Color(4, 7) else Color(0, 7);
    end else WriteChar(s[i]);
  end;
  Color(0, 7);
  ClearEOL;
end;

procedure ClearEOL;
var
  x: Integer;
begin
  if CurX=ScreenWidth - 1 then Exit;
  for x:=CurX to ScreenWidth - 1 do WriteChar(' ');
end;

procedure ClrTo(x1, x2, y: Integer);
var
  x: Integer;
begin
  GotoXY(x1, y);
  for x:=x1 to x2 do WriteChar(' ');
end;

procedure ClearBackScreen;
var
  i: Integer;
begin
  for i:=0 to ScreenWidth*ScreenHeight - 1 do VideoBuf^[i]:=32 or (7 shl 8);
  GotoXY(0, 0);
end;

function Input(x, y, Max: Integer; var Value: string): Boolean;
var
  Key: TKeyEvent;
  Ch: Char;
begin
  while True do begin
    GotoXY(x, y);WriteStr(Value + ' ');
    SetCursorPos(x + Length(Value), y);
    UpdateScreen(False);
    Key:=TranslateKeyEvent(GetKeyEvent);
    case GetKeyEventFlags(Key) of
      kbASCII: begin
        Ch:=GetKeyEventChar(Key);
        case Ch of
          #8: Value:=Copy(Value, 1, Length(Value) - 1);
          #27: Exit(False);
          #13: Exit(True);
          else if (Ch in [#32..#127]) and
                  (Length(Value) < Max) then Value:=Value + Ch;
        end;
      end;
    end;
  end;
end;

function Select(Title: string; Choices: array of string; Index: Integer): Integer;
var
  X, Y, MaxW, VL, I, J, Scroll: Integer;
  Key: TKeyEvent;
  Ch: Char;

  procedure MoveDown;
  begin
    if Index < High(Choices) then Inc(Index);
    if Index - Scroll >= VL then Inc(Scroll);
  end;

  procedure MoveUp;
  begin
    if Index > 0 then Dec(Index);
    if Index - Scroll < 0 then Dec(Scroll);
  end;

begin
  MaxW:=Length(Title);
  for I:=0 to High(Choices) do
    if MaxW < Length(Choices[I]) then MaxW:=Length(Choices[I]);
  if MaxW > ScreenWidth - 6 then MaxW:=ScreenWidth - 6;
  VL:=Length(Choices);
  if VL > ScreenHeight - 6 then VL:=ScreenHeight - 6;
  X:=(ScreenWidth - MaxW - 4) div 2;
  Y:=(ScreenHeight - VL - 2) div 2;
  if X < 1 then X:=1;
  if Y < 1 then Y:=1;
  Color(15, 3);
  for I:=0 to MaxW + 3 do begin
    GotoXY(X + I, Y);WriteStr(' ');
    GotoXY(X + I, Y + VL + 1);WriteStr(' ');
  end;
  for I:=0 to VL + 1 do begin
    GotoXY(X, Y + I);WriteStr(' ');
    GotoXY(X + MaxW + 3, Y + I);WriteStr(' ');
  end;
  if MaxW < Length(Title) then GotoXY(X + 2, Y) else GotoXY(X + 2 + (MaxW - Length(Title)) div 2, Y);
  WriteStr(Copy(Title, 1, MaxW));
  Scroll:=0;
  while True do begin
    for I:=0 to VL - 1 do begin
      if I + Scroll=Index then Color(15, 4) else Color(14, 0);
      GotoXY(X + 1, Y + I + 1);
      WriteStr(' ' + Copy(Choices[I + Scroll], 1, MaxW));
      for J:=Length(Choices[I + Scroll]) + 1 to MaxW do WriteStr(' ');
      WriteStr(' ');
    end;
    GotoXY(X + 2, Y + Index + 1);
    UpdateScreen(False);
    Key:=TranslateKeyEvent(GetKeyEvent);
    case GetKeyEventFlags(Key) of
      kbASCII: begin
        Ch:=GetKeyEventChar(Key);
        case Ch of
          'j', 'J': MoveDown;
          'k', 'K': MoveUp;
          'h': for I:=1 to Length(Choices) do MoveUp;
          'l': for I:=1 to Length(Choices) do MoveDown;
          #27: Exit(-1);
          #13: Exit(Index);
        end;
      end;
      else case GetKeyEventCode(Key) of
        kbdDown: MoveDown;
        kbdUp: MoveUp;
        kbdLeft, kbdHome: for I:=1 to Length(Choices) do MoveUp;
        kbdRight, kbdEnd: for I:=1 to Length(Choices) do MoveDown;
      end;
    end;
  end;
end;

procedure FullScreenText(Title, Text: string; QuitChar: Char);
var
  Words: array of string;
  Word: string;
  Scroll, X, Y, I: Integer;
  SkipWord: Boolean;
  AnyWordOut: Boolean;
  Key: TKeyEvent;
  Ch: Char;

  procedure MoveUp;
  begin
    if Scroll > 0 then Dec(Scroll);
  end;

  procedure MoveDown;
  begin
    if AnyWordOut then Inc(Scroll);
  end;

begin
  Words:=nil;
  Word:='';
  for I:=1 to Length(Text) do begin
    if Text[I]=' ' then begin
      SetLength(Words, Length(Words) + 1);
      Words[High(Words)]:=Word + ' ';
      Word:='';
    end else if Text[I] in [#1, #2, #3, #4, #5, #6, #7, #10] then begin
      SetLength(Words, Length(Words) + 2);
      Words[High(Words) - 1]:=Word;
      Words[High(Words)]:=Text[I];
      Word:='';
    end else if Text[I] <> #13 then Word += Text[I];
  end;
  if Word <> '' then begin
    SetLength(Words, Length(Words) + 1);
    Words[High(Words)]:=Word;
  end;
  Color(0, 7);
  GotoXY(0, ScreenHeight - 1);
  Scroll:=0;
  QuitChar:=UpCase(QuitChar);
  while True do begin
    ClearBackScreen;
    if Title='' then DefaultTop else Top(Title);
    Status(' ~Arrows~ Scroll Line ~Page Up/Down~ Scroll Page ~' + QuitChar + ' or Escape~ Return ');
    X:=0;
    Y:=1-Scroll;
    AnyWordOut:=False;
    Color(7, 0);
    GotoXY(0, 1);
    for I:=0 to High(Words) do begin
      SkipWord:=False;
      if (Words[I]=#10) or (X + Length(Words[I]) >= ScreenWidth) then begin
        if Words[I]=#10 then SkipWord:=True;
        X:=0;
        Inc(Y);
        if (Y > 0) and (Y < ScreenHeight - 1) then GotoXY(0, Y);
      end
      else if Words[I]=#1 then begin Color(9, 0); SkipWord:=True; end
      else if Words[I]=#2 then begin Color(10, 0); SkipWord:=True; end
      else if Words[I]=#3 then begin Color(11, 0); SkipWord:=True; end
      else if Words[I]=#4 then begin Color(12, 0); SkipWord:=True; end
      else if Words[I]=#5 then begin Color(13, 0); SkipWord:=True; end
      else if Words[I]=#6 then begin Color(14, 0); SkipWord:=True; end
      else if Words[I]=#7 then begin Color(7, 0); SkipWord:=True; end;
      if Y > ScreenHeight - 2 then Break;
      if Y > 0 then begin
        if not SkipWord then WriteStr(Words[I]);
        AnyWordOut:=True;
      end;
      if not SkipWord then Inc(X, Length(Words[I]));
    end;
    GotoXY(ScreenWidth - 1, ScreenHeight - 1);
    UpdateScreen(False);
    Key:=TranslateKeyEvent(GetKeyEvent);
    case GetKeyEventFlags(Key) of
      kbASCII: begin
        Ch:=GetKeyEventChar(Key);
        case Ch of
          ' ', 'j', 'J': MoveDown;
          'k', 'K': MoveUp;
          'p', 'P': for I:=1 to ScreenHeight - 4 do MoveUp;
          'n', 'N': for I:=1 to ScreenHeight + 4 do MoveDown;
          #27: Exit;
          else if UpCase(Ch)=QuitChar then Exit;
        end;
      end;
      else case GetKeyEventCode(Key) of
        kbdDown: MoveDown;
        kbdUp: MoveUp;
        kbdPgUp: for I:=1 to ScreenHeight - 4 do MoveUp;
        kbdPgDn: for I:=1 to ScreenHeight + 4 do MoveDown;
      end;
    end;
  end;
end;

initialization
  DoWrites:=True;
end.
