{*
 * hnb2olol - utility that converts hnb files to olol files
 * Copyright (C) 2011 Kostas Michalopoulos
 *
 * This software is provided 'as-is', without any express or implied
 * warranty.  In no event will the authors be held liable for any damages
 * arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not
 *    claim that you wrote the original software. If you use this software
 *    in a product, an acknowledgment in the product documentation would be
 *    appreciated but is not required.
 * 2. Altered source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 3. This notice may not be removed or altered from any source distribution.
 *
 * Kostas Michalopoulos <badsector@runtimeterror.com>
 *}
program hnb2olol;
{$MODE OBJFPC}{$H+}
uses SysUtils;
const
  Spaces = [#8, #9, #10, #13, ' '];
type
  TAttribute = record
    Name: string;
    Value: string;
  end;

  TAttributes = array of TAttribute;
var
  f: File;
  Ch: Char;
  LastOpenTag: string = '';
  AddTickable, AddTick: Boolean;

procedure ParseCode; forward;

procedure HandleOpenTag(Name: string; Attr: TAttributes);
var
  i: Integer;
begin
  if Name='node' then begin
    Write('NODE ');
    AddTickable:=False;
    AddTick:=False;
    for i:=0 to Length(Attr)-1 do begin
      if (Attr[i].Name='type') and (Attr[i].Value='todo') then begin
        AddTickable:=True;
        continue;
      end;
      if (Attr[i].Name='done') and (Attr[i].Value='yes') then begin
        AddTick:=True;
        continue;
      end;
    end;
  end;
  LastOpenTag:=Name;
end;

procedure HandleCloseTag(Name: string);
begin
  if Name='node' then WriteLn('DONE');
end;

procedure HandleContent(Content: string);
begin
  if LastOpenTag='data' then begin
    WriteLn(Content);
    if AddTickable then
      WriteLn('TYPE TICKABLE')
    else
      WriteLn('TYPE NORMAL');
    if AddTick then WriteLn('TICK');
  end;
end;

procedure Next;
begin
  if not Eof(f) then BlockRead(f, Ch, 1) else Ch:=#0;
end;

procedure SkipSpaces;
begin
  while Ch in Spaces do Next;
end;

procedure SkipSpecialTag;
var
  Counter: Integer;
  InString: Boolean;
begin
  Counter:=1;
  InString:=False;
  while not Eof(f) do begin
    if Ch='"' then InString:=not InString;
    if not InString then begin
      if Ch='>' then begin
        Dec(Counter);
        if Counter=0 then begin
          Next;
          break;
        end;
      end;
      if Ch='<' then Inc(Counter);
    end;
    Next;
  end;
end;

function ParseName: string;
begin
  Result:='';
  SkipSpaces;
  while Ch in ['a'..'z', 'A'..'Z', '-', '_', '0'..'9'] do begin
    Result:=Result + Ch;
    Next;
  end;
end;

function NextXMLChar: Char;
var
  Name: string;
begin
  if Ch='&' then begin
    Next;
    Name:=ParseName;
    if Name='amp' then Result:='&'
    else if Name='quot' then Result:='"'
    else if Name='lt' then Result:='<'
    else if Name='gt' then Result:='>'
    else Result:='?';
    if Ch=';' then Next;
  end else begin
    Result:=Ch;
    Next;
  end;
end;

function ParseValue: string;
begin
  Result:='';
  SkipSpaces;
  if Ch='"' then begin
    Next;
    while not Eof(f) do begin
      if Ch='"' then begin
        Next;
        SkipSpaces;
        break;
      end;
      Result:=Result + NextXMLChar;
    end;
  end else exit(ParseName);
end;

procedure ParseOpenTag;
var
  Name: string;
  Attr: TAttributes;
  Attrs: Integer;
begin
  Name:=ParseName;
  SkipSpaces;
  SetLength(Attr, 0);
  Attrs:=0;
  while not Eof(f) do begin
    SkipSpaces;
    if Eof(f) or (Ch='>') then begin
      if Ch='>' then Next;
      break;
    end;
    SetLength(Attr, Attrs + 1);
    Attr[Attrs].Name:=ParseName;
    SkipSpaces;
    if Ch <> '>' then begin
      if Ch='=' then begin
        Next;
        Attr[Attrs].Value:=ParseValue;
      end;
      Inc(Attrs);
      SkipSpaces;
    end;
  end;
  HandleOpenTag(Name, Attr);
  ParseCode;
end;

procedure ParseCloseTag;
var
  Name: string;
begin
  Next;
  Name:=ParseName;
  SkipSpaces;
  if Ch='>' then begin
    Next;
    SkipSpaces;
  end;
  HandleCloseTag(Name);
end;

procedure ParseTagBracket;
begin
  Next;
  if Eof(f) then exit;
  if Ch in ['?', '!'] then
    SkipSpecialTag
  else if Ch='/' then
    ParseCloseTag
  else
    ParseOpenTag;
end;

procedure ParseCode;
var
  Content: string;
begin
  SkipSpaces;
  Content:='';
  while not Eof(f) do begin
    if Ch='<' then begin
      Content:=Trim(Content);
      if Content <> '' then HandleContent(Content);
      Content:='';
      ParseTagBracket;
    end else if Ch in Spaces then begin
      SkipSpaces;
      Content:=Content + ' ';
    end else begin
      Content:=Content + NextXMLChar;
    end;
  end;
  Content:=Trim(Content);
  if Content <> '' then HandleContent(Content);
end;

begin
  if ParamCount < 1 then begin
    WriteLn('Usage: hnb2olol <filename.hnb>');
    exit;
  end;
  Assign(f, ParamStr(1));
  {$I-}
  Reset(f, 1);
  {$I+}
  if IOResult <> 0 then begin
    WriteLn('Failed to open ', ParamStr(1), ' for input');
    exit;
  end;
  WriteLn('NOTE');
  WriteLn('NOTE Converted from ', ParamStr(1), ' using hnb2olol');
  WriteLn('NOTE');
  WriteLn('NODE');
  WriteLn('TYPE NORMAL');
  Next;
  ParseCode;
  WriteLn('DONE');
  WriteLn('STOP');
  Close(f);
end.
