unit Scripting;

{$mode objfpc}{$H+}

interface

uses
  Classes, SysUtils, FPLIL, ChunkIO;

const
  EV_CARD_VISIT = 'Card Visit';
  EV_CARD_LEAVE = 'Card Leave';
  EV_ACTION = 'Action';
  EV_BUTTON_PUSH = 'Button Push';
  EV_BUTTON_RELEASE = 'Button Release';
  EV_DOUBLE_CLICK = 'Double Click';
  EV_POINTER_HOVER = 'Pointer Hover';
  EV_POINTER_LEAVE = 'Pointer Leave';
  EV_POINTER_MOTION = 'Pointer Motion';
  EV_CHARACTER = 'Character';
  EV_ARROW = 'Arrow';
  EV_KEY_PRESS = 'Key Press';
  EV_KEY_RELEASE = 'Key Release';
  EV_STACK_LOAD = 'Stack Load';
  EV_STACK_SAVE = 'Stack Save';
  EV_BROWSE = 'Browse';
  EV_TIMER = 'Timer';

  ScriptHandlerSetChunkID = 'SHSC';
  ScriptPropertySetChunkID = 'SPSC';

type

  // TODO: both TScriptHandlerSet and TScriptPropertySet are basically Key/Value
  // stores for strings, with one being a class and the other object... figure
  // out what (and if) i was thinking when i made that decision and if needed
  // modify the code to not do this...

  { TScriptHandlerInfo }

  PScriptHandlerInfo = ^TScriptHandlerInfo;
  TScriptHandlerInfo = object
    Name, Code: string;
  end;

  { TScriptHandlerSet }

  TScriptHandlerSet = class
  private
    FHandlers: array of TScriptHandlerInfo;
    function FindHandlerIndex(AName: string; MakeNew: Boolean): Integer;
  public
    procedure Assign(const Source: TScriptHandlerSet);
    procedure SetHandlerCode(AName, ACode: string);
    function GetHandlerCode(AName: string): string;
    procedure RemoveAll;
    function RunHandler(AName: string): Boolean;
    procedure WriteToChunkIO(ChunkIO: TChunkIO);
    procedure ReadFromChunkIO(ChunkIO: TChunkIO);
  end;

  { TScriptPropertySet }

  TScriptPropertySet = object
    Name, Value: array of string;
    procedure Assign(const Source: TScriptPropertySet);
    procedure RemoveAll;
    procedure Put(AName, AValue: string);
    function Get(AName: string): string;
    procedure WriteToChunkIO(ChunkIO: TChunkIO);
    procedure ReadFromChunkIO(ChunkIO: TChunkIO);
  end;

var
  LILRuntime: TLIL;
  PassCalled: Boolean;

procedure InitializeScripting;
procedure SetScriptVariable(Name, Value: string);
procedure SetScriptVariable(Name: string; Value: Int64);
procedure SetScriptVariable(Name: string; Value: Boolean);
function EvaluateScriptCode(Code: string): Boolean;

implementation

uses
  MainUnit, {CommanderUnit, CardInspectorUnit, }Cards, {CardBrowserUnit,
  PaintSettingsUnit, PartInspectorUnit, }Controls, Forms, Dialogs, LCLIntf,
  Menus;

var
  OriginalSet: TLILFunctionProc;

function ParseObjectReferenceArguments(FncName: string; AllowCard, AllowPart: Boolean; LIL: TLIL; Args: TLILFunctionProcArgs; var Base: Integer; out Obj: TObject): Boolean;
type
  TTargetType = (ttUnknown, ttCard, ttPart);
var
  Target: string;
  TargetType: TTargetType = ttUnknown;
  FoundObj: TObject;
  ID: Cardinal;
begin
  Result:=False;
  Obj:=nil;
  if Length(Args)=Base then begin
    LIL.SetError('Not enough arguments given in "' + FncName + '"');
    Exit;
  end;
  Target:=TLIL.ToString(Args[Base]);
  if Target='card' then begin
    if not AllowCard then begin
      LIL.SetError('Cards cannot be used with "' + FncName + '" here');
      Exit;
    end;
    Inc(Base);
    if Length(Args)=Base then begin
      LIL.SetError('No card name or card ID was given in "' + FncName + '"');
      Exit;
    end;
    TargetType:=ttCard;
  end else if Target='part' then begin
    if not AllowPart then begin
      LIL.SetError('Parts cannot be used with "' + FncName + '" here');
      Exit;
    end;
    Inc(Base);
    if Length(Args)=Base then begin
      LIL.SetError('No part name or part ID was given in "' + FncName + '"');
      Exit;
    end;
    TargetType:=ttPart;
  end;
  Target:=TLIL.ToString(Args[Base]);
  if Target='' then begin
    LIL.SetError('An empty value was given as the object name in "' + FncName + '"');
    Exit;
  end;
  Inc(Base);
  if Target[1]='&' then begin
    try
      ID:=StrToInt64(Copy(Target, 2, Length(Target)));
    except
      LIL.SetError('Invalid numeric value "' + Copy(Target, 2, Length(Target)) + '" was given as the ID in "' + FncName + '"');
      Exit;
    end;
    if ID=0 then begin
      LIL.SetError('Zero is not a valid ID in "' + FncName + '"');
      Exit;
    end;
    FoundObj:=Main.Stack.GetObjectByID(ID);
    if not Assigned(FoundObj) then begin
      LIL.SetError('The ID number ' + IntToStr(ID) + ' given in "' + FncName + '" does not belong to a known object');
      Exit;
    end;
    if (FoundObj is TCard) and (TargetType=ttPart) then begin
      LIL.SetError('&' + IntToStr(ID) + ' is a card ID but a part ID was expected after "part" in "' + FncName + '" here');
      Exit;
    end;
    if (FoundObj is TPart) and (TargetType=ttCard) then begin
      LIL.SetError('&' + IntToStr(ID) + ' is a part ID but a card ID was expected after "card" in "' + FncName + '" here');
      Exit;
    end;
    if (FoundObj is TCard) and not AllowCard then begin
      LIL.SetError('&' + IntToStr(ID) + ' is a card and cards cannot be used with "' + FncName + '" here');
      Exit;
    end;
    if (FoundObj is TPart) and not AllowPart then begin
      LIL.SetError('&' + IntToStr(ID) + ' is a part and parts cannot be used with "' + FncName + '" here');
      Exit;
    end;
    Result:=True;
    Obj:=FoundObj;
  end else begin
    if TargetType=ttUnknown then
      FoundObj:=Main.Stack.GetObjectByName(Target)
    else if TargetType=ttCard then
      FoundObj:=Main.Stack.GetCardByName(Target)
    else begin
      // when a part is explicitly asked, check the current card first
      FoundObj:=Main.Card.GetSubObjectByName(Target);
      if not Assigned(FoundObj) then FoundObj:=Main.Stack.GetObjectByName(Target);
    end;
    if not Assigned(FoundObj) then begin
      LIL.SetError('The name "' + Target + '" given in "' + FncName + '" does not match any known name');
      Exit;
    end;
    if (FoundObj is TCard) and (TargetType=ttPart) then begin
      LIL.SetError('"' + Target + '" is a card but a part was expected after "part" in "' + FncName + '" here');
      Exit;
    end;
    if (FoundObj is TPart) and (TargetType=ttCard) then begin
      LIL.SetError('"' + Target + '" is a part but a card was expected after "card" in "' + FncName + '" here');
      Exit;
    end;
    if (FoundObj is TCard) and not AllowCard then begin
      LIL.SetError('"' + Target + ' is a card and cards cannot be used with "' + FncName + '" here');
      Exit;
    end;
    if (FoundObj is TPart) and not AllowPart then begin
      LIL.SetError('"' + Target + ' is a part and parts cannot be used with "' + FncName + '" here');
      Exit;
    end;
    Result:=True;
    Obj:=FoundObj;
  end;
end;

function FncSay(LIL: TLIL; Args: TLILFunctionProcArgs): TLILValue;
var
  Msg: string = '';
  i: Integer;
begin
  for i:=0 to Length(Args) - 1 do
    Msg:=Msg + TLIL.ToString(Args[i]) + ' ';
  Msg:=Copy(Msg, 1, Length(Msg) - 1);
  MessageDlg(' ', Msg, mtInformation, [mbOK], 0);
  Result:=nil;
end;

function FncConfirm(LIL: TLIL; Args: TLILFunctionProcArgs): TLILValue;
var
  Msg: string = '';
  i: Integer;
begin
  for i:=0 to Length(Args) - 1 do
    Msg:=Msg + TLIL.ToString(Args[i]) + ' ';
  Msg:=Copy(Msg, 1, Length(Msg) - 1);
  Result:=TLIL.AllocBoolean(MessageDlg(' ', Msg, mtConfirmation, mbYesNo, 0)=mrYes);
  LIL.SetVar('it', Result, lsvlGlobal);
end;

function FncWarn(LIL: TLIL; Args: TLILFunctionProcArgs): TLILValue;
var
  Msg: string = '';
  i: Integer;
begin
  for i:=0 to Length(Args) - 1 do
    Msg:=Msg + TLIL.ToString(Args[i]) + ' ';
  Msg:=Copy(Msg, 1, Length(Msg) - 1);
  MessageDlg(' ', Msg, mtWarning, [mbOK], 0);
  Result:=nil;
end;

function FncAsk(LIL: TLIL; Args: TLILFunctionProcArgs): TLILValue;
var
  Msg: string = '';
  i: Integer;
begin
  for i:=0 to Length(Args) - 1 do
    Msg:=Msg + TLIL.ToString(Args[i]) + ' ';
  Msg:=InputBox('', Copy(Msg, 1, Length(Msg) - 1), '');
  Result:=TLIL.AllocString(Msg);
  LIL.SetVar('it', Result, lsvlGlobal);
end;

function FncDisplay(LIL: TLIL; Args: TLILFunctionProcArgs): TLILValue;
begin
  if Length(Args)=0 then begin
    LIL.SetError('Display type was not given in "display"');
    Exit;
  end;
  if TLIL.ToString(Args[0])='windowed' then
    Main.FullscreenMode:=False
  else if TLIL.ToString(Args[0])='fullscreen' then
    Main.FullscreenMode:=True
  else if TLIL.ToString(Args[0])='toggle' then
    Main.mToolFullscreen.Click
  else
    LIL.SetError('Unknown display type, it must be either fullscreen or windowed');
  Result:=nil;
end;

function FncShow(LIL: TLIL; Args: TLILFunctionProcArgs): TLILValue;
var
  Obj: TObject;
  Base: Integer = 0;
begin
  if Length(Args)=0 then begin
    LIL.SetError('No object given in "show"');
    Exit;
  end;
  if TLIL.ToString(Args[0])='menu' then
    Main.Menu:=Main.MainMenu1
  else if TLIL.ToString(Args[0])='pointer' then
    Main.ForcedCursor:=crDefault
  else begin
    if not ParseObjectReferenceArguments('show', False, True, LIL, Args, Base, Obj) then Exit(nil);
    if Obj is TPart then
      TPart(Obj).Visible:=True
    else begin
      LIL.SetError('Invalid object given in "show"');
    end;
  end;
  Result:=nil;
end;

function FncHide(LIL: TLIL; Args: TLILFunctionProcArgs): TLILValue;
var
  Obj: TObject;
  Base: Integer = 0;
begin
  if Length(Args)=0 then begin
    LIL.SetError('No object given in "hide"');
    Exit;
  end;
  if TLIL.ToString(Args[0])='menu' then begin
    Main.Menu:=TMainMenu.Create(Main);
    Main.Menu.Free;
    Main.Menu:=nil;
    Main.InvalidateClientRectCache(True);
  end else if TLIL.ToString(Args[0])='pointer' then
    Main.ForcedCursor:=crNone
  else begin
    if not ParseObjectReferenceArguments('hide', False, True, LIL, Args, Base, Obj) then Exit(nil);
    if Obj is TPart then
      TPart(Obj).Visible:=False
    else begin
      LIL.SetError('Invalid object given in "hide"');
    end;
  end;
  Result:=nil;
end;

function FncGo(LIL: TLIL; Args: TLILFunctionProcArgs): TLILValue;
var
  Target: string = '';
  Base: Integer = 0;
  Card: TCard;
begin
  Result:=nil;
  if Length(Args)=0 then begin
    LIL.SetError('No destination given in "go"');
    Exit;
  end;
  if TLIL.ToString(Args[0])='to' then begin
    Base:=1;
    if Length(Args)=1 then begin
      LIL.SetError('No destination given in "go"');
      Exit;
    end;
  end;
  Target:=TLIL.ToString(Args[Base]);
  if (Target='back') or (Target='recent') then
    Main.mGoLastVisited.Click
  else if Target='previous' then
    Main.mGoPreviousCardClick(nil)
  else if Target='next' then
    Main.mGoNextClick(nil)
  else if Target='first' then
    Main.mGoFirstCardClick(nil)
  else if Target='last' then
    Main.mGoLastCardClick(nil)
  else if Target='browser' then
    Main.mGoToClick(nil)
  else begin
    if ParseObjectReferenceArguments('go [to]', True, False, LIL, Args, Base, TObject(Card)) then
      Main.Card:=Card
    else
      Exit;
  end;
end;

function FncNew(LIL: TLIL; Args: TLILFunctionProcArgs): TLILValue;

  function NewCard: TLILValue;
  var
    Card: TCard;
  begin
    Card:=Main.Stack.Add(Main.ScreenArea.BgCard, Main.Stack.IndexOf(Main.Card) + 1);
    Main.ScreenArea.Card:=Card;
    Result:=TLIL.AllocString('&' + IntToStr(Card.ID));
  end;

  function NewBackground: TLILValue;
  var
    Card: TCard;
  begin
    Card:=Main.Stack.AddBackground;
    Main.Card.Background:=Card;
    Result:=TLIL.AllocString('&' + IntToStr(Card.ID));
  end;

begin
  Result:=nil;
  if (Length(Args)=0) or (TLIL.ToString(Args[0])='card') then begin
    Result:=NewCard;
  end else if TLIL.ToString(Args[0])='background' then begin
    Result:=NewBackground;
  end else begin
    LIL.SetError('I don''t know how to create a new "' + TLIL.ToString(Args[0]) + '"');
  end;
  LIL.SetVar('it', Result, lsvlGlobal);
end;

function FncDelete(LIL: TLIL; Args: TLILFunctionProcArgs): TLILValue;
begin
  Result:=nil;
  if Length(Args) <> 0 then begin
    LIL.SetError('"delete" can currently only be used to delete the current card');
    Exit;
  end;
  if Main.Stack.Count=1 then begin
    LIL.SetError('You cannot delete the last card');
    Exit;
  end;
  Main.DeleteThisCard;
end;

function FncSet(LIL: TLIL; Args: TLILFunctionProcArgs): TLILValue;
var
  PropName: string;
  Obj: TObject;
  Base: Integer;
begin
  if (Length(Args) > 0) and (TLIL.ToString(Args[0])='property') then begin
    Result:=nil;
    if Length(Args) < 3 then begin
      LIL.SetError('Not enough arguments to "set property" - property name and object are missing');
      Exit;
    end;
    PropName:=TLIL.ToString(Args[1]);
    if PropName='' then begin
      LIL.SetError('Cannot use an empty property name with "set property"');
      Exit;
    end;
    if TLIL.ToString(Args[2])='of' then Base:=3 else Base:=2;
    if not ParseObjectReferenceArguments('set property', True, True, LIL, Args, Base, Obj) then Exit;
    if Length(Args)=Base then begin
      LIL.SetError('New property value is missing in "set property" after object');
      Exit;
    end;
    if TLIL.ToString(Args[Base])='to' then Inc(Base);
    if Length(Args)=Base then begin
      LIL.SetError('New property value is missing in "set property" after "to"');
      Exit;
    end;
    if Obj is TCard then
      TCard(Obj).SetScriptProperty(PropName, TLIL.ToString(Args[Base]))
    else
      TPart(Obj).SetScriptProperty(PropName, TLIL.ToString(Args[Base]));
  end else Result:=OriginalSet(LIL, Args);
end;

function FncGet(LIL: TLIL; Args: TLILFunctionProcArgs): TLILValue;
var
  PropName: string;
  Obj: TObject;
  Base: Integer;
begin
  Result:=nil;
  if Length(Args) < 2 then begin
    LIL.SetError('Not enough arguments to "get property" - property name and object are missing');
    Exit;
  end;
  if TLIL.ToString(Args[0])='property' then Base:=1 else Base:=0;
  PropName:=TLIL.ToString(Args[Base]);
  if PropName='' then begin
    LIL.SetError('Cannot use an empty property name with "get property"');
    Exit;
  end;
  Inc(Base);
  if Length(Args)=Base then begin
    LIL.SetError('Object is missing from "get property" after the property name');
    Exit;
  end;
  if TLIL.ToString(Args[Base])='of' then Inc(Base);
  if Length(Args)=Base then begin
    LIL.SetError('Object is missing from "get property" after "of"');
    Exit;
  end;
  if not ParseObjectReferenceArguments('get property', True, True, LIL, Args, Base, Obj) then Exit;
  if Obj is TCard then
    Result:=TLIL.AllocString(TCard(Obj).GetScriptProperty(PropName))
  else
    Result:=TLIL.AllocString(TPart(Obj).GetScriptProperty(PropName));
  LIL.SetVar('it', Result, lsvlGlobal);
end;

function FncBeep(LIL: TLIL; Args: TLILFunctionProcArgs): TLILValue;
begin
  Beep;
  Result:=nil;
end;

function FncSleep(LIL: TLIL; Args: TLILFunctionProcArgs): TLILValue;
begin
  if Length(Args)=0 then begin
    LIL.SetError('Time to sleep (in milliseconds) is missing in call to "sleep"');
    Exit(nil);
  end;
  Sleep(TLIL.ToInteger(Args[0]));
  Result:=nil;
end;

function FncPass(LIL: TLIL; Args: TLILFunctionProcArgs): TLILValue;
begin
  PassCalled:=True;
  Result:=nil;
end;

function FncTimer(LIL: TLIL; Args: TLILFunctionProcArgs): TLILValue;
begin
  if Length(Args)=0 then Exit(TLIL.AllocBoolean(Main.ScriptTimer.Enabled));
  Result:=nil;
  if TLIL.ToString(Args[0])='on' then begin
    Main.ScriptTimer.Enabled:=True;
  end else if TLIL.ToString(Args[0])='off' then begin
    Main.ScriptTimer.Enabled:=False;
  end else begin
    LIL.SetError('Unknown timer command');
  end;
end;

procedure InitializeScripting;
begin
  with LILRuntime do begin
    OriginalSet:=GetFunction('set').FunctionProc;

    // Messages, etc
    Register('say', @FncSay);
    Register('confirm', @FncConfirm);
    Register('warn', @FncWarn);
    Register('ask', @FncAsk);

    // User Interface
    Register('display', @FncDisplay);
    Register('show', @FncShow);
    Register('hide', @FncHide);

    // Navigation
    Register('go', @FncGo);

    // Making/deleting stuff
    Register('new', @FncNew);
    Register('delete', @FncDelete);

    // Property access
    Register('set', @FncSet);
    Register('get', @FncGet);

    // Misc
    Register('beep', @FncBeep);
    Register('sleep', @FncSleep);
    Register('pass', @FncPass);
    Register('timer', @FncTimer);
  end;
end;

procedure SetScriptVariable(Name, Value: string);
var
  LILValue: TLILValue;
begin
  LILValue:=TLIL.AllocString(Value);
  LILRuntime.SetVar(Name, LILValue, lsvlGlobal);
  FreeAndNil(LILValue);
end;

procedure SetScriptVariable(Name: string; Value: Int64);
var
  LILValue: TLILValue;
begin
  LILValue:=TLIL.AllocInteger(Value);
  LILRuntime.SetVar(Name, LILValue, lsvlGlobal);
  FreeAndNil(LILValue);
end;

procedure SetScriptVariable(Name: string; Value: Boolean);
var
  LILValue: TLILValue;
begin
  LILValue:=TLIL.AllocBoolean(Value);
  LILRuntime.SetVar(Name, LILValue, lsvlGlobal);
  FreeAndNil(LILValue);
end;

function EvaluateScriptCode(Code: string): Boolean;
var
  LILResult: TLILValue;
begin
  LILResult:=LILRuntime.Parse(Code, True);
  FreeAndNil(LILResult);
  Result:=not LILRuntime.Error;
end;

{ TScriptPropertySet }

procedure TScriptPropertySet.Assign(const Source: TScriptPropertySet);
var
  i: Integer;
begin
  SetLength(Name, Length(Source.Name));
  SetLength(Value, Length(Source.Name));
  for i:=0 to Length(Name) - 1 do begin
    Name[i]:=Source.Name[i];
    Value[i]:=Source.Value[i];
  end;
end;

procedure TScriptPropertySet.RemoveAll;
begin
  SetLength(Name, 0);
  SetLength(Value, 0);
end;

procedure TScriptPropertySet.Put(AName, AValue: string);
var
  i: Integer;
begin
  for i:=0 to Length(Name) - 1 do
    if Name[i]=AName then begin
      Value[i]:=AValue;
      Exit;
    end;
  SetLength(Name, Length(Name) + 1);
  SetLength(Value, Length(Name));
  Name[Length(Name) - 1]:=AName;
  Value[Length(Name) - 1]:=AValue;
end;

function TScriptPropertySet.Get(AName: string): string;
var
  i: Integer;
begin
  for i:=0 to Length(Name) - 1 do
    if Name[i]=AName then Exit(Value[i]);
  Result:='';
end;

procedure TScriptPropertySet.WriteToChunkIO(ChunkIO: TChunkIO);
var
  i: Integer;
begin
  with ChunkIO do begin
    BeginChunk(ChunkID(ScriptPropertySetChunkID));
    for i:=0 to Length(Name) - 1 do begin
      BeginChunk(ChunkID('PROP'));
      WriteString(ChunkID('NAME'), Name[i]);
      WriteString(ChunkID('VALU'), Value[i]);
      EndChunk();
    end;
    EndChunk();
  end;
end;

procedure TScriptPropertySet.ReadFromChunkIO(ChunkIO: TChunkIO);
var
  NName, NValue: string;
  ID, Size: Cardinal;
begin
  RemoveAll;
  with ChunkIO do begin
    while NextChunk(ID, Size) do begin
      if ID=ChunkID('PROP') then begin
        NName:='';
        NValue:='';
        while NextChunk(ID, Size) do begin
          if ID=ChunkID('NAME') then
            NName:=ReadString
          else if ID=ChunkID('VALU') then
            NValue:=ReadString
          else SkipChunk(Size);
        end;
        Put(NName, NValue);
      end else SkipChunk(Size);
    end;
  end;
end;

{ TScriptHandlerSet }

function TScriptHandlerSet.FindHandlerIndex(AName: string; MakeNew: Boolean): Integer;
var
  i: Integer;
begin
  for i:=0 to Length(FHandlers) - 1 do
    if FHandlers[i].Name=AName then Exit(i);
  if MakeNew then begin
    i:=Length(FHandlers);
    SetLength(FHandlers, i + 1);
    FHandlers[i].Name:=AName;
    Exit(i);
  end;
  Result:=-1;
end;

procedure TScriptHandlerSet.Assign(const Source: TScriptHandlerSet);
var
  i: Integer;
begin
  SetLength(FHandlers, Length(Source.FHandlers));
  for i:=0 to Length(FHandlers) - 1 do
    FHandlers[i]:=Source.FHandlers[i];
end;

procedure TScriptHandlerSet.SetHandlerCode(AName, ACode: string);
var
  i, Index: Integer;
  DoDelete: Boolean;
begin
  DoDelete:=Trim(ACode)='';
  // Note: if DoDelete is true and the handler is not set, this code will
  // cause it to be created and then immediatelly deleted.  While this isn't
  // a good idea in general, in this case since there will only be a handful
  // of handlers it won't make any practical difference
  Index:=FindHandlerIndex(AName, True);
  if DoDelete then begin
    for i:=Index to Length(FHandlers) - 2 do
      FHandlers[i]:=FHandlers[i + 1];
    SetLength(FHandlers, Length(FHandlers) - 1);
  end else begin
    FHandlers[Index].Code:=ACode;
  end;
end;

function TScriptHandlerSet.GetHandlerCode(AName: string): string;
var
  Index: Integer;
begin
  Index:=FindHandlerIndex(AName, False);
  if Index > -1 then Result:=FHandlers[Index].Code else Result:='';
end;

procedure TScriptHandlerSet.RemoveAll;
begin
  SetLength(FHandlers, 0);
end;

function TScriptHandlerSet.RunHandler(AName: string): Boolean;
var
  Code: string;
begin
  Code:=GetHandlerCode(AName);
  if Code <> '' then begin
    PassCalled:=False;
    if EvaluateScriptCode(Code) then
      Result:=not PassCalled
    else
      Result:=True;
  end else Result:=False;
end;

procedure TScriptHandlerSet.WriteToChunkIO(ChunkIO: TChunkIO);
var
  i: Integer;
begin
  with ChunkIO do begin
    BeginChunk(ChunkID(ScriptHandlerSetChunkID));
    for i:=0 to Length(FHandlers) - 1 do begin
      BeginChunk(ChunkID('HDLR'));
      WriteString(ChunkID('NAME'), FHandlers[i].Name);
      WriteString(ChunkID('CODE'), FHandlers[i].Code);
      EndChunk();
    end;
    EndChunk();
  end;
end;

procedure TScriptHandlerSet.ReadFromChunkIO(ChunkIO: TChunkIO);
var
  ID, Size: Cardinal;
  Name, Code: string;
begin
  RemoveAll;
  with ChunkIO do begin
    while NextChunk(ID, Size) do begin
      if ID=ChunkID('HDLR') then begin
        Name:='';
        Code:='';
        while NextChunk(ID, Size) do begin
          if ID=ChunkID('NAME') then
            Name:=ReadString
          else if ID=ChunkID('CODE') then
            Code:=ReadString
          else SkipChunk(Size);
        end;
        if (Trim(Name) <> '') and (Trim(Code) <> '') then SetHandlerCode(Name, Code);
      end else SkipChunk(Size);
    end;
  end;
end;

end.

