unit MainUnit;

{$mode objfpc}{$H+}

interface

uses
  {$IFDEF WINDOWS}
  Windows,
  {$ENDIF}
  Classes, SysUtils, FileUtil, Forms, Controls, Graphics, Dialogs, Menus,
  ExtCtrls, Cards, FPLIL, LazHelp, IntfGraphics, GraphType, LCLType, LCLIntf,
  ExtDlgs;

type

  { TScreenArea }

  TScreenArea = class(TCustomControl)
  private
    FBgMode: Boolean;
    FCard: TCard;
    FBackgroundBitmap: TBitmap;
    FHistory: array of TCard;
    FHistHeader: Integer;
    function GetBgCard: TCard; inline;
    function GetTargetCard: TCard; inline;
    procedure PushToHistory(ACard: TCard);
    procedure SetBgMode(const AValue: Boolean); inline;
    procedure SetCard(const AValue: TCard); inline;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure EraseBackground(DC: HDC); override;
    procedure Paint; override;
    procedure MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: Integer); override;
    procedure MouseUp(Button: TMouseButton; Shift: TShiftState; X, Y: Integer); override;
    procedure MouseMove(Shift: TShiftState; X, Y: Integer); override;
    procedure KeyDown(var Key: Word; Shift: TShiftState); override;
    procedure KeyUp(var Key: Word; Shift: TShiftState); override;
    procedure UTF8KeyPress(var UTF8Key: TUTF8Char); override;
    procedure DblClick; override;
    procedure Resize; override;
    procedure GoToLastVisited;
    property BgMode: Boolean read FBgMode write SetBgMode;
    property Card: TCard read FCard write SetCard;
    property BgCard: TCard read GetBgCard;
    property TargetCard: TCard read GetTargetCard;
  end;

  { TMain }

  TMain = class(TForm)
    ApplicationProperties1: TApplicationProperties;
    ColorDialog1: TColorDialog;
    FontDialog1: TFontDialog;
    LazHelp1: TLazHelp;
    LazHelpWindowedViewer1: TLazHelpWindowedViewer;
    LILRuntime: TLIL;
    MainMenu1: TMainMenu;
    mEditBar5: TMenuItem;
    mEditMoveTo: TMenuItem;
    mEditNewCardAndBackground: TMenuItem;
    mEditBar2: TMenuItem;
    mEditCopyPicture: TMenuItem;
    mEditPastePicture: TMenuItem;
    mEditCardInspector: TMenuItem;
    mEditBar8: TMenuItem;
    mEditStackInspector: TMenuItem;
    mToolFullscreen: TMenuItem;
    mHelpLILCardSite: TMenuItem;
    mHelpContents: TMenuItem;
    mHelpBar1: TMenuItem;
    mFileBar1: TMenuItem;
    mFileSaveACopy: TMenuItem;
    mFileSaveAs: TMenuItem;
    mFileSave: TMenuItem;
    mFileOpen: TMenuItem;
    mFileNew: TMenuItem;
    mToolBar2: TMenuItem;
    mToolCommander: TMenuItem;
    mFileExit: TMenuItem;
    mEdit: TMenuItem;
    mEditBackgroundCard: TMenuItem;
    mEditBar1: TMenuItem;
    mEditPartInspector: TMenuItem;
    mEditUseGrid: TMenuItem;
    mEditBar3: TMenuItem;
    mEditGridSize: TMenuItem;
    mEditGridSizeTiny: TMenuItem;
    mEditGridSizeSmall: TMenuItem;
    mEditGridSizeNormal: TMenuItem;
    mEditGridSizeBig: TMenuItem;
    mEditGridSizeHuge: TMenuItem;
    mEditGridSizeEnormous: TMenuItem;
    mEditBar4: TMenuItem;
    mEditRaiseToTop: TMenuItem;
    mEditSinkToBottom: TMenuItem;
    mEditPaintSettings: TMenuItem;
    mEditBar6: TMenuItem;
    mEditNewCard: TMenuItem;
    mEditDeleteCard: TMenuItem;
    mEditFont: TMenuItem;
    mEditFontColor: TMenuItem;
    mEditBar7: TMenuItem;
    mEditDuplicatePart: TMenuItem;
    mEditDeletePart: TMenuItem;
    mGoTo: TMenuItem;
    mGoBar2: TMenuItem;
    mGoLastVisited: TMenuItem;
    mGoBar1: TMenuItem;
    mGoLastCard: TMenuItem;
    mGoPreviousCard: TMenuItem;
    mGoNext: TMenuItem;
    mGoFirstCard: TMenuItem;
    mGo: TMenuItem;
    mToolField: TMenuItem;
    mToolFloodFill: TMenuItem;
    mToolLine: TMenuItem;
    mToolBox: TMenuItem;
    mToolOval: TMenuItem;
    mHelpAbout: TMenuItem;
    mHelp: TMenuItem;
    mToolPen: TMenuItem;
    mToolBar1: TMenuItem;
    mToolButton: TMenuItem;
    mToolBrowse: TMenuItem;
    mTool: TMenuItem;
    mFile: TMenuItem;
    OpenDialog1: TOpenDialog;
    SaveDialog1: TSaveDialog;
    ScriptTimer: TTimer;
    StartupTimer: TTimer;
    mEditOpenPicture: TMenuItem;
    OpenPictureDialog1: TOpenPictureDialog;
    procedure ApplicationProperties1Idle(Sender: TObject; var Done: Boolean);
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure FormKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState);
    procedure LazHelp1LinkClick(Sender: TObject; var ATarget: string;
      var Processed: Boolean);
    procedure LILRuntimeError(LIL: TLIL; APosition: Integer; AMessage: string);
    procedure LILRuntimeExit(LIL: TLIL; Arg: TLILValue);
    procedure LILRuntimeRead(LIL: TLIL; AFileName: string; out AContent: string
      );
    procedure LILRuntimeSource(LIL: TLIL; AFileName: string; out
      AContent: string);
    procedure LILRuntimeStore(LIL: TLIL; AFileName, AContent: string);
    procedure LILRuntimeWrite(LIL: TLIL; Chars: string);
    procedure mEditBackgroundCardClick(Sender: TObject);
    procedure mEditCardInspectorClick(Sender: TObject);
    procedure mEditClick(Sender: TObject);
    procedure mEditCopyPictureClick(Sender: TObject);
    procedure mEditDeleteCardClick(Sender: TObject);
    procedure mEditDeletePartClick(Sender: TObject);
    procedure mEditDuplicatePartClick(Sender: TObject);
    procedure mEditFontClick(Sender: TObject);
    procedure mEditFontColorClick(Sender: TObject);
    procedure mEditGridSizeClick(Sender: TObject);
    procedure mEditMoveToClick(Sender: TObject);
    procedure mEditNewCardAndBackgroundClick(Sender: TObject);
    procedure mEditNewCardClick(Sender: TObject);
    procedure mEditPartInspectorClick(Sender: TObject);
    procedure mEditRaiseToTopClick(Sender: TObject);
    procedure mEditSinkToBottomClick(Sender: TObject);
    procedure mEditStackInspectorClick(Sender: TObject);
    procedure mEditUseGridClick(Sender: TObject);
    procedure mEditPaintSettingsClick(Sender: TObject);
    procedure mEditPastePictureClick(Sender: TObject);
    procedure mFileExitClick(Sender: TObject);
    procedure mFileNewClick(Sender: TObject);
    procedure mFileOpenClick(Sender: TObject);
    procedure mFileSaveACopyClick(Sender: TObject);
    procedure mFileSaveAsClick(Sender: TObject);
    procedure mFileSaveClick(Sender: TObject);
    procedure mGoClick(Sender: TObject);
    procedure mGoLastVisitedClick(Sender: TObject);
    procedure mGoPreviousCardClick(Sender: TObject);
    procedure mGoFirstCardClick(Sender: TObject);
    procedure mGoNextClick(Sender: TObject);
    procedure mGoLastCardClick(Sender: TObject);
    procedure mGoToClick(Sender: TObject);
    procedure mHelpAboutClick(Sender: TObject);
    procedure mHelpContentsClick(Sender: TObject);
    procedure mHelpLILCardSiteClick(Sender: TObject);
    procedure mToolBoxClick(Sender: TObject);
    procedure mToolBrowseClick(Sender: TObject);
    procedure mToolButtonClick(Sender: TObject);
    procedure mToolCommanderClick(Sender: TObject);
    procedure mToolFieldClick(Sender: TObject);
    procedure mToolFloodFillClick(Sender: TObject);
    procedure mToolFullscreenClick(Sender: TObject);
    procedure mToolLineClick(Sender: TObject);
    procedure mToolOvalClick(Sender: TObject);
    procedure mToolPenClick(Sender: TObject);
    procedure GridSizeMenuItemClick(Sender: TObject);
    procedure ScriptTimerTimer(Sender: TObject);
    procedure StartupTimerTimer(Sender: TObject);
    procedure mEditOpenPictureClick(Sender: TObject);
  private
    FForcedCursor: TCursor;
    FFileName: string;
    FFullscreenMode: Boolean;
    FStack: TStack;
    FScreenArea: TScreenArea;
    function GetCard: TCard;
    function GetUIMode: TUIMode;
    procedure Initialize;
    procedure SetCard(const AValue: TCard);
    procedure SetForcedCursor(AValue: TCursor);
    procedure SetFileName(const AValue: string);
    procedure SetFullscreenMode(AValue: Boolean);
    procedure SetUIMode(const AValue: TUIMode);
    procedure Shutdown;
    procedure BackgroundModeChanged;
    procedure UpdateCaption;
    procedure EnableFullscreen;
    procedure DisableFullscreen;
    function IsFullscreenSupported: Boolean;
  public
    procedure RepaintCard;
    procedure RunCardBrowser;
    procedure DeleteThisCard;
    procedure NewStack;
    function SaveToFile(AFileName: string): Boolean;
    function OpenFromFile(AFileName: string): Boolean;
    property UIMode: TUIMode read GetUIMode write SetUIMode;
    property Card: TCard read GetCard write SetCard;
    property Stack: TStack read FStack;
    property ScreenArea: TScreenArea read FScreenArea;
    property FileName: string read FFileName write SetFileName;
    property FullscreenMode: Boolean read FFullscreenMode write SetFullscreenMode;
    property ForcedCursor: TCursor read FForcedCursor write SetForcedCursor;
  end; 

var
  Main: TMain;

implementation

uses
  PartInspectorUnit, PaintSettingsUnit, CardBrowserUnit, Clipbrd,
  CardInspectorUnit, Scripting, CommanderUnit, ScriptErrorBoxUnit,
  StackInspectorUnit, AboutBoxUnit;

{ TScreenArea }

procedure TScreenArea.SetCard(const AValue: TCard);
var
  PrevBackgroundCard: TCard = nil;
begin
  if FCard=AValue then Exit;
  if BgMode then BgMode:=False;
  if Assigned(FCard) then PrevBackgroundCard:=FCard.Background;
  if Assigned(FCard) then FCard.HandleCardExit;
  if Assigned(PrevBackgroundCard) then PrevBackgroundCard.HandleCardExit;
  PushToHistory(FCard);
  FCard:=AValue;
  if Assigned(FCard) and Assigned(FCard.Background) then FCard.Background.HandleCardEnter;
  if Assigned(FCard) then FCard.HandleCardEnter;
  Repaint;
  if Assigned(CardInspector) then CardInspector.Card:=Card;
end;

function TScreenArea.GetBgCard: TCard;
begin
  if Assigned(Card) then Result:=Card.Background else Result:=nil;
end;

function TScreenArea.GetTargetCard: TCard;
begin
  if BgMode then Result:=BgCard else Result:=Card;
  if not Assigned(Result) then Result:=Card;
end;

procedure TScreenArea.PushToHistory(ACard: TCard);
begin
  SetLength(FHistory, FHistHeader + 1);
  FHistory[FHistHeader]:=ACard;
  Inc(FHistHeader);
end;

procedure TScreenArea.SetBgMode(const AValue: Boolean);
begin
  if FBgMode=AValue then Exit;
  if AValue and (Main.UIMode=uiBrowse) then Main.UIMode:=uiButton;
  FBgMode:=AValue;
  if Assigned(Card) then
    if BgMode then
      Card.HandleCardExit
    else
      Card.HandleCardEnter;
  Repaint;
  Main.BackgroundModeChanged;
  if Assigned(CardInspector) then CardInspector.Card:=TargetCard;
end;

constructor TScreenArea.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FBackgroundBitmap:=TBitmap.Create;
//  FBackgroundBitmap.PixelFormat:=pf32bit;
  FBackgroundBitmap.SetSize(640, 480);
  Constraints.MinWidth:=640;
  Constraints.MaxWidth:=640;
  Constraints.MinHeight:=480;
  Constraints.MaxHeight:=480;
end;

destructor TScreenArea.Destroy;
begin
  FreeAndNil(FBackgroundBitmap);
  inherited Destroy;
end;

procedure TScreenArea.EraseBackground(DC: HDC);
begin
end;

procedure TScreenArea.Paint;
begin
  FBackgroundBitmap.Canvas.Brush.Style:=bsSolid;
  FBackgroundBitmap.Canvas.Brush.Color:=clGray;
  FBackgroundBitmap.Canvas.Rectangle(-1, -1, 641, 481);
  if Assigned(TargetCard) then TargetCard.PaintIn(FBackgroundBitmap.Canvas);
  Canvas.Draw(0, 0, FBackgroundBitmap);
  inherited Paint;
end;

procedure TScreenArea.MouseDown(Button: TMouseButton; Shift: TShiftState; X,
  Y: Integer);
begin
  SetFocus;
  if Assigned(TargetCard) then TargetCard.HandleMouseDown(X, Y, Shift, Button);
end;

procedure TScreenArea.MouseUp(Button: TMouseButton; Shift: TShiftState; X,
  Y: Integer);
begin
  if Assigned(TargetCard) then TargetCard.HandleMouseUp(X, Y, Shift, Button);
end;

procedure TScreenArea.MouseMove(Shift: TShiftState; X, Y: Integer);
var
  Done: Boolean = True;
  NewCursor: TCursor;
begin
  if Assigned(TargetCard) then begin
    TargetCard.HandleMouseMotion(X, Y, Shift);
    if Main.ForcedCursor <> crDefault then
      NewCursor:=Main.ForcedCursor
    else
      NewCursor:=TargetCard.Cursor;
    Cursor:=NewCursor;
  end;
  Main.ApplicationProperties1Idle(nil, Done);
end;

procedure TScreenArea.KeyDown(var Key: Word; Shift: TShiftState);
begin
  if Assigned(Card) and (Card.UIMode=uiBrowse) then Card.HandleKeyDown(Key, Shift);
end;

procedure TScreenArea.KeyUp(var Key: Word; Shift: TShiftState);
begin
  if Assigned(Card) and (Card.UIMode=uiBrowse) then Card.HandleKeyUp(Key, Shift);
end;

procedure TScreenArea.UTF8KeyPress(var UTF8Key: TUTF8Char);
begin
  if Assigned(Card) and (Card.UIMode=uiBrowse) then Card.HandleCharKey(UTF8Key);
end;

procedure TScreenArea.DblClick;
begin
  if Assigned(TargetCard) then TargetCard.HandleDoubleClick;
end;

procedure TScreenArea.Resize;
begin
  Repaint;
end;

procedure TScreenArea.GoToLastVisited;
var
  SaveHistHeader: Integer;
begin
  if FHistHeader=1 then Exit;
  while FHistHeader > 1 do begin
    Dec(FHistHeader);
    if Card.Stack.IndexOf(FHistory[FHistHeader]) <> -1 then Break;
  end;
  SaveHistHeader:=FHistHeader;
  Card:=FHistory[FHistHeader];
  FHistHeader:=SaveHistHeader;
end;

{$R *.lfm}

{ TMain }

procedure TMain.LazHelp1LinkClick(Sender: TObject; var ATarget: string;
  var Processed: Boolean);
begin
  if ATarget='openlilcardpage' then begin
    mHelpLILCardSite.Click;
    Processed:=True;
  end else if ATarget='lildocumentationurl' then begin
    OpenURL('http://runtimeterror.com/rep/lil/doc/tip/readme.txt');
    Processed:=True;
  end;
end;

procedure TMain.LILRuntimeError(LIL: TLIL; APosition: Integer; AMessage: string);
begin
  ScriptErrorBox.ShowError(APosition, LIL.EvaluatedCode, AMessage);
end;

procedure TMain.LILRuntimeExit(LIL: TLIL; Arg: TLILValue);
begin
  Close;
end;

procedure TMain.LILRuntimeRead(LIL: TLIL; AFileName: string; out
  AContent: string);
begin
  AContent:='';
end;

procedure TMain.LILRuntimeSource(LIL: TLIL; AFileName: string; out
  AContent: string);
begin
  AContent:='';
end;

procedure TMain.LILRuntimeStore(LIL: TLIL; AFileName, AContent: string);
begin
  // do nothing
end;

procedure TMain.LILRuntimeWrite(LIL: TLIL; Chars: string);
begin
  Commander.Put(Chars);
end;

procedure TMain.mEditBackgroundCardClick(Sender: TObject);
begin
  ScreenArea.BgMode:=not ScreenArea.BgMode;
end;

procedure TMain.mEditCardInspectorClick(Sender: TObject);
begin
  CardInspector.InspectCard(ScreenArea.TargetCard);
end;

procedure TMain.mEditClick(Sender: TObject);
var
  ActivePartExists: Boolean;
begin
  ActivePartExists:=Assigned(Card) and Assigned(Card.ActivePart);
  mEditPartInspector.Enabled:=ActivePartExists;
  mEditFont.Enabled:=ActivePartExists;
  mEditFontColor.Enabled:=ActivePartExists;
  mEditRaiseToTop.Enabled:=ActivePartExists;
  mEditSinkToBottom.Enabled:=ActivePartExists;
  mEditMoveTo.Enabled:=ActivePartExists;
  mEditDuplicatePart.Enabled:=ActivePartExists;
  mEditDeletePart.Enabled:=ActivePartExists;

  if ActivePartExists then begin
    if Assigned(Card.ActivePart.Card.Background) then
      mEditMoveTo.Caption:='Move to Background'
    else
      mEditMoveTo.Caption:='Move to Foreground';
  end else
    mEditMoveTo.Caption:='Move to...';
end;

procedure TMain.mEditCopyPictureClick(Sender: TObject);
begin
  try
    ScreenArea.BgCard.Image.SaveToClipboardFormat(PredefinedClipboardFormat(pcfBitmap));
  except
    MessageDlg('Copy Picture', 'Failed to copy the picture.', mtError, [mbOK], 0);
  end;
end;

procedure TMain.mEditDeleteCardClick(Sender: TObject);
begin
  if Assigned(Card) and Assigned(Stack) then begin
    if Stack.Count=1 then begin
      MessageDlg('Last Card', 'You cannot delete the last card!', mtError, [mbOK], 0);
      Exit;
    end;
    if MessageDlg('Delete Card', 'Are you sure that you want to delete this card?', mtConfirmation, mbYesNo, 0)=mrYes then
      DeleteThisCard;
  end;
end;

procedure TMain.mEditDeletePartClick(Sender: TObject);
begin
  if UIMode=uiBrowse then UIMode:=uiButton;
  if Assigned(Card) and Assigned(Card.ActivePart) then Card.ActivePart.Free;
end;

procedure TMain.mEditDuplicatePartClick(Sender: TObject);
begin
  if UIMode=uiBrowse then UIMode:=uiButton;
  if Assigned(ScreenArea) then ScreenArea.TargetCard.DuplicateActivePart;
end;

procedure TMain.mEditFontClick(Sender: TObject);
begin
  if not Assigned(Card) then Exit;
  if not Card.SetupLazFont(FontDialog1.Font) then Exit;
  if FontDialog1.Execute then Card.ApplyLazFont(FontDialog1.Font);
end;

procedure TMain.mEditFontColorClick(Sender: TObject);
var
  CardFont: TCardFont;
begin
  if not Assigned(Card) or not Assigned(Card.ActivePart) then Exit;
  CardFont:=Card.ActivePart.Font;
  ColorDialog1.Color:=CardFont.Color;
  if ColorDialog1.Execute then begin
    CardFont.Color:=ColorDialog1.Color;
    Card.ActivePart.Font:=CardFont;
  end;
end;

procedure TMain.mEditGridSizeClick(Sender: TObject);
var
  i: Integer;
begin
  if not Assigned(Card) then Exit;
  for i:=0 to mEditGridSize.Count - 1 do
    with mEditGridSize[i] do begin
      RadioItem:=True;
      Checked:=Tag=GridSize;
    end;
end;

procedure TMain.mEditMoveToClick(Sender: TObject);
var
  Part: TPart;
  NewCard: TCard = nil;
begin
  if Assigned(Card) and Assigned(Card.ActivePart) then begin
    Part:=Card.ActivePart;
    if Assigned(Part.Card.Background) then
      NewCard:=Part.Card.Background
    else if Assigned(Part.Card.Foreground) then
      NewCard:=Part.Card.Foreground;
    if Assigned(NewCard) then begin
      Part.Card.Remove(Part);
      NewCard.Add(Part);
    end;
  end;
end;

procedure TMain.mEditNewCardAndBackgroundClick(Sender: TObject);
var
  NewCard: TCard;
begin
  NewCard:=Stack.Add(nil, Stack.IndexOf(Card) + 1);
  ScreenArea.Card:=NewCard;
end;

procedure TMain.mEditNewCardClick(Sender: TObject);
var
  NewCard: TCard;
begin
  NewCard:=Stack.Add(ScreenArea.BgCard, Stack.IndexOf(Card) + 1);
  ScreenArea.Card:=NewCard;
end;

procedure TMain.mEditPartInspectorClick(Sender: TObject);
begin
  if Assigned(Card) then PartInspector.InspectPart(Card.ActivePart);
end;

procedure TMain.mEditRaiseToTopClick(Sender: TObject);
begin
  // the card.part.card access is done because cards can contain active parts
  // that belong to other cards (backgrounds)
  if Assigned(Card) and Assigned(Card.ActivePart) then
    Card.ActivePart.Card.RaiseToTop(Card.ActivePart);
end;

procedure TMain.mEditSinkToBottomClick(Sender: TObject);
begin
  // the card.part.card access is done because cards can contain active parts
  // that belong to other cards (backgrounds)
  if Assigned(Card) and Assigned(Card.ActivePart) then
    Card.ActivePart.Card.SinkToBottom(Card.ActivePart);
end;

procedure TMain.mEditStackInspectorClick(Sender: TObject);
begin
  StackInspector.Stack:=Stack;
  StackInspector.Show;
end;

procedure TMain.mEditUseGridClick(Sender: TObject);
begin
  UseGrid:=not UseGrid;
  mEditUseGrid.Checked:=UseGrid;
end;

procedure TMain.mEditPaintSettingsClick(Sender: TObject);
begin
  PaintSettings.Show;
end;

procedure TMain.mEditPastePictureClick(Sender: TObject);
var
  TmpBmp: TBitmap;
  CF: TClipboardFormat;
  x, y: Integer;
begin
  TmpBmp:=TBitmap.Create;
  try
    CF:=Clipboard.FindPictureFormatID;
    {$IFDEF WINDOWS}
    if CF=2 then // CF_BITMAP
      TmpBmp.LoadFromClipboardFormat(PredefinedClipboardFormat(pcfBitmap))
    else
    {$ENDIF}
      TmpBmp.LoadFromClipboardFormat(CF);
  except
    MessageDlg('Paste Picture', 'Failed to paste the picture. Does the clipboard contain a valid bitmap picture?', mtError, [mbOK], 0);
    FreeAndNil(TmpBmp);
    Exit;
  end;
  ScreenArea.BgCard.Image.Canvas.StretchDraw(Rect(0, 0, 640, 480), TmpBmp);
  // this is sadly needed to get rid of the alpha channel
  with ScreenArea.BgCard.Image.Canvas do begin
    for y:=0 to 479 do
      for x:=0 to 639 do
        Pixels[x, y]:=Pixels[x, y];
  end;
  FreeAndNil(TmpBmp);
  RepaintCard;
end;

procedure TMain.mFileExitClick(Sender: TObject);
begin
  Close;
end;

procedure TMain.mFileNewClick(Sender: TObject);
begin
  if MessageDlg('New Stack', 'Any modifications you may have made will be lost. Do you want to continue?', mtConfirmation, mbYesNo, 0)=mrYes then NewStack;
end;

procedure TMain.mFileOpenClick(Sender: TObject);
begin
  if MessageDlg('Open Stack', 'Any modifications you may have made will be lost. Do you want to continue?', mtConfirmation, mbYesNo, 0) <> mrYes then Exit;
  OpenDialog1.FileName:=FileName;
  if OpenDialog1.Execute then begin
    if not OpenFromFile(OpenDialog1.FileName) then
      MessageDlg('Failed to open the stack file', 'There was a problem while trying to open the stack file ' + OpenDialog1.FileName, mtError, [mbOK], 0);
  end;
end;

procedure TMain.mFileSaveACopyClick(Sender: TObject);
begin
  SaveDialog1.FileName:=FileName;
  if SaveDialog1.Execute then
    SaveToFile(SaveDialog1.FileName);
end;

procedure TMain.mFileSaveAsClick(Sender: TObject);
begin
  SaveDialog1.FileName:=FileName;
  if SaveDialog1.Execute then begin
    if SaveToFile(SaveDialog1.FileName) then FileName:=SaveDialog1.FileName;
  end;
end;

procedure TMain.mFileSaveClick(Sender: TObject);
begin
  if FileName='' then begin
    mFileSaveAsClick(Sender);
    Exit;
  end;
  SaveToFile(FileName);
end;

procedure TMain.mGoClick(Sender: TObject);
begin
  mGoFirstCard.Enabled:=Assigned(Card) and Assigned(Stack) and (Stack.IndexOf(Card) > 0);
  mGoLastCard.Enabled:=Assigned(Card) and Assigned(Stack) and (Stack.IndexOf(Card) < Stack.Count - 1);
end;

procedure TMain.mGoLastVisitedClick(Sender: TObject);
begin
  ScreenArea.GoToLastVisited;
end;

procedure TMain.mGoPreviousCardClick(Sender: TObject);
begin
  Card:=Card.Previous;
end;

procedure TMain.mGoFirstCardClick(Sender: TObject);
begin
  Card:=Stack.FirstCard;
end;

procedure TMain.mGoNextClick(Sender: TObject);
begin
  Card:=Card.Next;
end;

procedure TMain.mGoLastCardClick(Sender: TObject);
begin
  Card:=Stack.LastCard;
end;

procedure TMain.mGoToClick(Sender: TObject);
begin
  RunCardBrowser;
end;

procedure TMain.mHelpAboutClick(Sender: TObject);
begin
  ShowAboutBox;
end;

procedure TMain.mHelpContentsClick(Sender: TObject);
begin
  LazHelpWindowedViewer1.ShowHelp;
end;

procedure TMain.mHelpLILCardSiteClick(Sender: TObject);
begin
  OpenURL('http://runtimeterror.com/rep/lilcard');
end;

procedure TMain.mToolBoxClick(Sender: TObject);
begin
  UIMode:=uiBox;
end;

procedure TMain.mToolBrowseClick(Sender: TObject);
begin
  UIMode:=uiBrowse;
end;

procedure TMain.mToolButtonClick(Sender: TObject);
begin
  UIMode:=uiButton;
end;

procedure TMain.mToolCommanderClick(Sender: TObject);
begin
  Commander.Show;
end;

procedure TMain.mToolFieldClick(Sender: TObject);
begin
  UIMode:=uiField;
end;

procedure TMain.mToolFloodFillClick(Sender: TObject);
begin
  UIMode:=uiFloodFill;
end;

procedure TMain.mToolFullscreenClick(Sender: TObject);
begin
  FullscreenMode:=not FullscreenMode;
  mToolFullscreen.Checked:=FullscreenMode;
end;

procedure TMain.mToolLineClick(Sender: TObject);
begin
  UIMode:=uiLine;
end;

procedure TMain.mToolOvalClick(Sender: TObject);
begin
  UIMode:=uiOval;
end;

procedure TMain.mToolPenClick(Sender: TObject);
begin
  UIMode:=uiPen;
end;

procedure TMain.GridSizeMenuItemClick(Sender: TObject);
begin
  if Assigned(Card) then GridSize:=TMenuItem(Sender).Tag;
end;

procedure TMain.ScriptTimerTimer(Sender: TObject);
begin
  if UIMode=uiBrowse then begin
    if Assigned(Stack) then Stack.TimerTick;
    if Assigned(ScreenArea.BgCard) then ScreenArea.BgCard.TimerTick;
    if Assigned(ScreenArea.Card) then ScreenArea.Card.TimerTick;
  end;
end;

procedure TMain.StartupTimerTimer(Sender: TObject);
var
  I: Integer = 1;
begin
  StartupTimer.Enabled:=False;
  if UpperCase(ParamStr(I))='/FULLSCREEN' then begin
    FullscreenMode:=True;
    mToolFullscreen.Checked:=True;
    Inc(I);
  end;
  if ParamStr(I) <> '' then begin
    if not OpenFromFile(ParamStr(I)) then
      MessageDlg('Failed to open the stack file', 'There was a problem while trying to open the stack file ' + ParamStr(I), mtError, [mbOK], 0);
  end;
end;

procedure TMain.mEditOpenPictureClick(Sender: TObject);
var
  Pic: TPicture;
  x, y: Integer;
begin
  if OpenPictureDialog1.Execute then begin
    Pic:=TPicture.Create;
    try
      Pic.LoadFromFile(OpenPictureDialog1.FileName);
      ScreenArea.BgCard.Image.Canvas.StretchDraw(Rect(0, 0, 640, 480), Pic.Graphic);
      // this is sadly needed to get rid of the alpha channel
      with ScreenArea.BgCard.Image.Canvas do begin
        for y:=0 to 479 do
          for x:=0 to 639 do
            Pixels[x, y]:=Pixels[x, y];
      end;
      RepaintCard;
    except
      MessageDlg('Load Image', 'Failed to load ' + OpenPictureDialog1.FileName, mtError, [mbOK], 0);
    end;
    FreeAndNil(Pic);
  end;
end;

procedure TMain.FormCreate(Sender: TObject);

  {$IFDEF DARWIN}
  procedure Macize;

    procedure DoMenus;

      procedure DoMenu(Menu: TMenu);
      var
        i: Integer;

        procedure DoMenuItem(MI: TMenuItem);
        var
          i: Integer;
        begin
          if ((MI.ShortCut and scCtrl)=scCtrl) and ((MI.ShortCut and scMeta)=0) then
            MI.ShortCut:=(MI.ShortCut-scCtrl) or scMeta;
          for i:=0 to MI.Count - 1 do DoMenuItem(MI[i]);
        end;

      begin
        for i:=0 to Menu.Items.Count - 1 do DoMenuItem(Menu.Items[i]);
      end;

    begin
      DoMenu(MainMenu1);
    end;

    procedure DoAppleMenu;
    var
      Apple: TMenuItem;
    begin
      Apple:=TMenuItem.Create(MainMenu1);
      Apple.Caption:='';
      mHelp.Remove(mHelpLILCardSite);
      mHelp.Remove(mHelpBar1);
      mHelp.Remove(mHelpAbout);
      mFile.Remove(mFileExit);
      mFile.Remove(mFile.Items[mFile.Count - 1]);
      Apple.Add(mHelpAbout);
      Apple.Add(mHelpBar1);
      Apple.Add(mHelpLILCardSite);
      mHelpAbout.Caption:='About LILCard';
      MainMenu1.Items.Insert(0, Apple);
    end;

  begin
    DoMenus;
    DoAppleMenu;
  end;

  {$ENDIF}

begin
  ClientWidth:=640;
  ClientHeight:=480;
  mToolFullscreen.Visible:=IsFullscreenSupported;
  {$IFDEF DARWIN}
  Macize;
  {$ENDIF}
  Initialize;
end;

procedure TMain.ApplicationProperties1Idle(Sender: TObject; var Done: Boolean);
var
  DoRepaint: Boolean = False;
begin
  if Assigned(ScreenArea.Card) and ScreenArea.Card.NeedsUpdate then DoRepaint:=True;
  if Assigned(ScreenArea.BgCard) and ScreenArea.BgCard.NeedsUpdate then DoRepaint:=True;
  if DoRepaint then RepaintCard;
end;

procedure TMain.FormDestroy(Sender: TObject);
begin
  Shutdown;
end;

procedure TMain.FormKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState);

  procedure ScanMenuForKey(AMenu: TMenu);
  var
    Item: TMenuItem;
  begin
    Item:=AMenu.FindItem(ShortCut(Key, Shift), fkShortCut);
    if Assigned(Item) and Assigned(Item.OnClick) then begin
      Key:=0;
      Item.OnClick(Item);
    end;
  end;

begin
  if Menu=nil then begin
    ScanMenuForKey(MainMenu1);
  end;
end;

procedure TMain.Initialize;
begin
  {$IFDEF LILCARD_RELEASE}
  {$I doc/lilcardhelp.inc}
  {$ELSE}
  LazHelp1.FileName:='doc/lilcard.lazhelp';
  {$ENDIF}
  {$IFNDEF WINDOWS}
  // flood fill isn't supported by non-windows LCL backends
  mToolFloodFill.Visible:=False;
  {$ENDIF}
  FScreenArea:=TScreenArea.Create(Self);
  FScreenArea.Align:=alClient;
  FScreenArea.Parent:=Self;
  Scripting.LILRuntime:=LILRuntime;
  InitializeScripting;
  NewStack;
end;

function TMain.GetUIMode: TUIMode;
begin
  Result:=Stack.UIMode;
end;

function TMain.GetCard: TCard;
begin
  Result:=ScreenArea.Card;
end;

procedure TMain.SetCard(const AValue: TCard);
begin
  ScreenArea.Card:=AValue;
end;

procedure TMain.SetForcedCursor(AValue: TCursor);
begin
  if FForcedCursor=AValue then Exit;
  FForcedCursor:=AValue;
  ScreenArea.Cursor:=AValue;
end;

procedure TMain.SetFileName(const AValue: string);
begin
  if FFileName=AValue then Exit;
  FFileName:=AValue;
  UpdateCaption;
end;

procedure TMain.SetFullscreenMode(AValue: Boolean);
begin
  if FFullscreenMode=AValue then Exit;
  FFullscreenMode:=AValue;
  iF FullscreenMode then EnableFullscreen else DisableFullscreen;
end;

procedure TMain.SetUIMode(const AValue: TUIMode);
begin
  if AValue=UIMode then Exit;
  if AValue=uiBrowse then ScreenArea.BgMode:=False;
  Stack.UIMode:=AValue;
  mToolBrowse.Checked:=UIMode=uiBrowse;
  mToolButton.Checked:=UIMode=uiButton;
  mToolField.Checked:=UIMode=uiField;
  mToolPen.Checked:=UIMode=uiPen;
  mToolLine.Checked:=UIMode=uiLine;
  mToolBox.Checked:=UIMode=uiBox;
  mToolOval.Checked:=UIMode=uiOval;
  mToolFloodFill.Checked:=UIMode=uiFloodFill;
  UpdateCaption;
  RepaintCard;
end;

procedure TMain.Shutdown;
begin
  FreeAndNil(FStack);
end;

procedure TMain.BackgroundModeChanged;
begin
  UpdateCaption;
  mEditBackgroundCard.Checked:=ScreenArea.BgMode;
end;

procedure TMain.UpdateCaption;
var
  NewCaption: string;
begin
  if FileName='' then
    NewCaption:='LILCard'
  else
    NewCaption:=FileName + ' - LILCard';
  case UIMode of
    uiButton: NewCaption:=NewCaption + ' [Button]';
    uiField: NewCaption:=NewCaption + ' [Field]';
    uiPen: NewCaption:=NewCaption + ' [Pen';
    uiLine: NewCaption:=NewCaption + ' [Line';
    uiBox: NewCaption:=NewCaption + ' [Box';
    uiOval: NewCaption:=NewCaption + ' [Oval';
    uiFloodFill: NewCaption:=NewCaption + ' [Flood Fill';
  end;
  if UIMode in [uiPen, uiLine, uiBox, uiOval, uiFloodFill] then
    if ScreenArea.BgMode then
      NewCaption:=NewCaption + ']'
    else
      NewCaption:=NewCaption + ' for background]';
  if ScreenArea.BgMode then
    NewCaption:=NewCaption + ' [Background]';
  Caption:=NewCaption;
end;

{$IFDEF WINDOWS}
var
  PreviousMode: DEVMODE;
  PreviousRect: TRect;
{$ENDIF}

procedure TMain.EnableFullscreen;
{$IFDEF WINDOWS}
var
  I: Integer;
  Mode, NewMode: DEVMODE;
{$ENDIF}
begin
  {$IFDEF WINDOWS}
  PreviousRect:=BoundsRect;
  if not EnumDisplaySettings(nil, ENUM_CURRENT_SETTINGS, @PreviousMode) then begin
    FFullscreenMode:=False;
    MessageDlg('Fullscreen Mode', 'Failed to get current settings', mtError, [mbOK], 0);
    Exit;
  end;
  NewMode.dmPelsWidth:=0;
  for I:=0 to 1024 do begin
    if not EnumDisplaySettings(nil, I, @Mode) then Break;
    if (Mode.dmDisplayFlags and DM_INTERLACED)=DM_INTERLACED then Continue;
    if (Mode.dmDisplayFlags and DM_GRAYSCALE)=DM_GRAYSCALE then Continue;
    if (Mode.dmPelsWidth=640) and (Mode.dmPelsHeight=480) then begin
      if (NewMode.dmPelsWidth=0) or (Mode.dmDisplayFrequency >= PreviousMode.dmDisplayFrequency) or (Mode.dmBitsPerPel >= NewMode.dmBitsPerPel) then begin
        NewMode:=Mode;
        if (Mode.dmDisplayFrequency=PreviousMode.dmDisplayFrequency) and (Mode.dmBitsPerPel=PreviousMode.dmBitsPerPel) then Break;
      end
    end;
  end;
  if NewMode.dmPelsWidth=0 then begin
    FFullscreenMode:=False;
    MessageDlg('Fullscreen Mode', 'Failed to find an appropriate 640x480 graphics mode', mtError, [mbOK], 0);
    Exit;
  end;
  if ChangeDisplaySettings(@NewMode, CDS_FULLSCREEN or CDS_TEST) <> DISP_CHANGE_SUCCESSFUL then begin
    FFullscreenMode:=False;
    MessageDlg('Fullscreen Mode', 'Failed to test initialize 640x480 graphics mode', mtError, [mbOK], 0);
    Exit;
  end;
  BorderStyle:=bsNone;
  Left:=0;
  Top:=0;
  Width:=Screen.Width;
  Height:=Screen.Height;
  FormStyle:=fsStayOnTop;
  Menu:=nil;
  Application.ProcessMessages;
  if ChangeDisplaySettings(@NewMode, CDS_FULLSCREEN) <> DISP_CHANGE_SUCCESSFUL then begin
    FFullscreenMode:=False;
    BorderStyle:=bsSingle;
    Menu:=MainMenu1;
    BoundsRect:=PreviousRect;
    FormStyle:=fsNormal;
    MessageDlg('Fullscreen Mode', 'Failed to initialize 640x480 graphics mode', mtError, [mbOK], 0);
    Exit;
  end;
  BorderStyle:=bsNone;
  Left:=0;
  Top:=0;
  Width:=640;
  Height:=480;
  FormStyle:=fsStayOnTop;
  {$ENDIF}
end;

procedure TMain.DisableFullscreen;
begin
  {$IFDEF WINDOWS}
  ChangeDisplaySettings(nil, 0);
  BorderStyle:=bsSingle;
  Menu:=MainMenu1;
  BoundsRect:=PreviousRect;
  FormStyle:=fsNormal;
  {$ENDIF}
end;

function TMain.IsFullscreenSupported: Boolean;
begin
  {$IFDEF WINDOWS}
  Result:=True;
  {$ELSE}
  Result:=False;
  {$ENDIF}
end;

procedure TMain.RepaintCard;
begin
  ScreenArea.Repaint;
end;

procedure TMain.RunCardBrowser;
var
  SelectedCard: TCard;
begin
  if SelectCard(Stack, Card, SelectedCard, 'Go To...', '&GO') then
    Card:=SelectedCard;
end;

procedure TMain.DeleteThisCard;
var
  PrevCard: TCard;
begin
    PrevCard:=Card;
    if Card=Stack.LastCard then
      Card:=Stack.PreviousCardOf(Card)
    else
      Card:=Stack.NextCardOf(Card);
    Stack.Remove(PrevCard);
end;

procedure TMain.NewStack;
begin
  if Assigned(Stack) and Assigned(ScreenArea) then UIMode:=uiBrowse;
  Card:=nil;
  ScriptTimer.Enabled:=False;
  if Assigned(StackInspector) then StackInspector.Stack:=nil;
  FreeAndNil(FStack);
  FStack:=TStack.Create;
  FStack.ControlParent:=ScreenArea;
  Card:=Stack.Add(nil);
  Card.Name:='Start';
  if Assigned(StackInspector) then StackInspector.Stack:=Stack;
  FileName:='';
end;

function TMain.SaveToFile(AFileName: string): Boolean;
begin
  Result:=True;
  try
    Stack.WriteToFile(AFileName);
  except
    MessageDlg('Failed to save the stack file', 'There was a problem while trying to save the stack file ' + SaveDialog1.FileName, mtError, [mbOK], 0);
    Result:=False;
  end;
end;

function TMain.OpenFromFile(AFileName: string): Boolean;
begin
  UIMode:=uiBrowse;
  try
    Card:=nil;
    StackInspector.Stack:=nil;
    ScriptTimer.Enabled:=False;
    FreeAndNil(FStack);
    FStack:=TStack.Create;
    FStack.ControlParent:=ScreenArea;
    FStack.ReadFromFile(AFileName);
    StackInspector.Stack:=Stack;
    Card:=FStack.Cards[0];
  except
    NewStack;
    Exit(False);
  end;
  if FStack.Count=0 then NewStack;
  FileName:=AFileName;
  Result:=True;
end;

end.

