unit Cards;

{$mode objfpc}{$H+}

interface

uses
  Classes, SysUtils, FGL, GraphType, Graphics, Controls, Types, StdCtrls,
  Scripting, ChunkIO, LCLType;

const
  StackChunkID = 'STAK';
  CardChunkID = 'CARD';
  ButtonPartChunkID = 'PBTN';
  FieldPartChunkID = 'PFLD';

type
  TUIMode = (uiBrowse, uiButton, uiField, uiPen, uiLine, uiBox, uiOval, uiFloodFill);
  TPartArea = (paOutside, paCenter, paLeftTop, paTop, paRightTop, paRight, paRightBottom, paBottom, paLeftBottom, paLeft);
  TCard = class;
  TStack = class;

  { TCardFont }

  TCardFont = object
  private
    FBold: Boolean;
    FColor: TColor;
    FItalic: Boolean;
    FName: string;
    FPitch: TFontPitch;
    FSize: Integer;
    FUnderline: Boolean;
  public
    procedure Assign(Font: TFont);
    procedure Setup(Font: TFont);
    procedure WriteToChunkIO(ChunkIO: TChunkIO);
    procedure ReadFromChunkIO(ChunkIO: TChunkIO);
    property Name: string read FName write FName;
    property Size: Integer read FSize write FSize;
    property Bold: Boolean read FBold write FBold;
    property Italic: Boolean read FItalic write FItalic;
    property Underline: Boolean read FUnderline write FUnderline;
    property Pitch: TFontPitch read FPitch write FPitch;
    property Color: TColor read FColor write FColor;
  end;

  { TPart }

  TPart = class
  private
    FFont: TCardFont;
    FID: Cardinal;
    FAutoLockMouse: Boolean;
    FCard: TCard;
    FColor: TColor;
    FName: string;
    FScriptHandlerSet: TScriptHandlerSet;
    FScriptPropertySet: TScriptPropertySet;
    FSharedData: Boolean;
    FVisible: Boolean;
    FX1: Integer;
    FX2: Integer;
    FY1: Integer;
    FY2: Integer;
    function GetActive: Boolean;
    function GetLeftButton: Boolean; inline;
    function GetMiddleButton: Boolean; inline;
    function GetMouseX: Integer; inline;
    function GetMouseY: Integer; inline;
    function GetRightButton: Boolean; inline;
    function GetUIMode: TUIMode; inline;
    function GetUnderMouse: Boolean; inline;
    procedure SetActive(const AValue: Boolean);
    procedure SetCard(const AValue: TCard); inline;
    procedure SetColor(const AValue: TColor);
    procedure SetFont(const AValue: TCardFont);
    procedure SetLazFont(const AValue: TFont);
    procedure SetName(const AValue: string);
    procedure SetVisible(AValue: Boolean);
    procedure SetX1(const AValue: Integer); inline;
    procedure SetX2(const AValue: Integer); inline;
    procedure SetY1(const AValue: Integer); inline;
    procedure SetY2(const AValue: Integer); inline;
  protected
    procedure Invalidate;
    function GetAllowEditing: Boolean; virtual;
    function GetAllowInteraction: Boolean; virtual;
    function GetAllowOffScreen: Boolean; virtual;
    function GetPartEditor: TWinControl; virtual;
    function GetCursor: TCursor; virtual;
    procedure SetSharedData(const AValue: Boolean); virtual;
    procedure CardChanged; virtual;
    procedure SizeChanged; virtual;
    procedure UIModeChanged; virtual;
    procedure ControlParentChanged; virtual;
    procedure VisibilityChanged; virtual;
    procedure Activated; virtual;
    procedure Deactivated; virtual;
    procedure PrepareCanvas(Canvas: TCanvas); virtual;
    procedure PaintInCanvas(Canvas: TCanvas); virtual;
    procedure PaintEditingOverlay(Canvas: TCanvas); virtual;
    procedure Reset; virtual;
    procedure LockMouse; inline;
    procedure UnlockMouse; inline;
    procedure Activate; inline;
    procedure Deactivate; inline;
    function HandleMouseDown(X, Y: Integer; Shift: TShiftState; Button: TMouseButton): Boolean; virtual;
    function HandleMouseMotion(X, Y: Integer; Shift: TShiftState): Boolean; virtual;
    function HandleMouseUp(X, Y: Integer; Shift: TShiftState; Button: TMouseButton): Boolean; virtual;
    function HandleMouseEnter(X, Y: Integer): Boolean; virtual;
    function HandleMouseExit(X, Y: Integer): Boolean; virtual;
    function HandleDoubleClick(X, Y: Integer): Boolean; virtual;
    procedure HandleCardEnter; virtual;
    procedure HandleCardExit; virtual;
    procedure TimerTick; virtual;
    procedure FillDuplicate(ATargetPart: TPart); virtual;
    procedure WritePropertiesToChunkIO(ChunkIO: TChunkIO); virtual;
    function ReadPropertyFromChunkIO(ChID: Cardinal; ChunkIO: TChunkIO): Boolean; virtual;
    procedure BeforeWriteToDisk; virtual;
    procedure AfterReadFromDisk; virtual;
  public
    destructor Destroy; override;
    procedure Construct; virtual;
    procedure Place(AX1, AY1, AX2, AY2: Integer);
    function IsPointInside(X, Y: Integer): Boolean; virtual;
    function PartAreaAt(X, Y: Integer): TPartArea; virtual;
    function MousePartArea: TPartArea; inline;
    procedure RaiseToTop; inline;
    procedure SinkToBottom; inline;
    function VisibleInEditMode: Boolean; inline;
    function GetSubObjectByID(AID: Cardinal): TObject; virtual;
    function GetObjectByID(AID: Cardinal): TObject;
    function GetSubObjectByName(AName: string): TObject; virtual;
    function GetObjectByName(AName: string): TObject;
    procedure SetScriptProperty(AName, AValue: string); virtual;
    function GetScriptProperty(AName: string): string; virtual;
    function Duplicate: TPart; virtual;
    function CallScript(AHandlerName: string; PassToCard: Boolean=True): Boolean;
    function GetChunkID: Cardinal; virtual;
    procedure WriteToChunkIO(ChunkIO: TChunkIO); virtual;
    procedure ReadFromChunkIO(ChunkIO: TChunkIO); virtual;
    property ID: Cardinal read FID;
    property Name: string read FName write SetName;
    property UIMode: TUIMode read GetUIMode;
    property ScriptHandlerSet: TScriptHandlerSet read FScriptHandlerSet;
    property AllowEditing: Boolean read GetAllowEditing;
    property AllowInteraction: Boolean read GetAllowInteraction;
    property AllowOffScreen: Boolean read GetAllowOffScreen;
    property PartEditor: TWinControl read GetPartEditor;
    property SharedData: Boolean read FSharedData write SetSharedData;
    property UnderMouse: Boolean read GetUnderMouse;
    property Active: Boolean read GetActive write SetActive;
    property AutoLockMouse: Boolean read FAutoLockMouse write FAutoLockMouse;
    property MouseX: Integer read GetMouseX;
    property MouseY: Integer read GetMouseY;
    property LeftButton: Boolean read GetLeftButton;
    property MiddleButton: Boolean read GetMiddleButton;
    property RightButton: Boolean read GetRightButton;
    property Color: TColor read FColor write SetColor;
    property Cursor: TCursor read GetCursor;
    property Font: TCardFont read FFont write SetFont;
    property LazFont: TFont write SetLazFont;
    property X1: Integer read FX1 write SetX1;
    property Y1: Integer read FY1 write SetY1;
    property X2: Integer read FX2 write SetX2;
    property Y2: Integer read FY2 write SetY2;
    property Visible: Boolean read FVisible write SetVisible;
    property Card: TCard read FCard write SetCard;
  end;

  TPartList = specialize TFPGObjectList<TPart>;

  { TButtonPart }
  TButtonPartStyle = (bpsInvisible, bpsOpaque, bpsRectangle, bpsRoundRect, bpsShadowed, bpsNative);

  TButtonPart = class(TPart)
  private
    FAutoHighlight: Boolean;
    FDrawName: Boolean;
    FHighlight: Boolean;
    FImage: TBitmap;
    FInteractive: Boolean;
    FStyle: TButtonPartStyle;
    procedure SetDrawName(const AValue: Boolean);
    procedure SetHighlight(const AValue: Boolean);
    procedure SetImage(AValue: TBitmap);
    procedure SetInteractive(AValue: Boolean);
    procedure SetStyle(const AValue: TButtonPartStyle);
  protected
    function GetAllowInteraction: Boolean; override;
    function GetPartEditor: TWinControl; override;
    function GetCursor: TCursor; override;
    procedure PaintInCanvas(Canvas: TCanvas); override;
    procedure PaintEditingOverlay(Canvas: TCanvas); override;
    procedure Reset; override;
    function HandleMouseDown(X, Y: Integer; Shift: TShiftState; Button: TMouseButton): Boolean; override;
    function HandleMouseUp(X, Y: Integer; Shift: TShiftState; Button: TMouseButton): Boolean; override;
    function HandleMouseEnter(X, Y: Integer): Boolean; override;
    function HandleMouseExit(X, Y: Integer): Boolean; override;
    procedure FillDuplicate(ATargetPart: TPart); override;
    procedure WritePropertiesToChunkIO(ChunkIO: TChunkIO); override;
    function ReadPropertyFromChunkIO(ChID: Cardinal; ChunkIO: TChunkIO): Boolean; override;
  public
    procedure SetScriptProperty(AName, AValue: string); override;
    function GetScriptProperty(AName: string): string; override;
    function GetChunkID: Cardinal; override;
    property Style: TButtonPartStyle read FStyle write SetStyle;
    property DrawName: Boolean read FDrawName write SetDrawName;
    property Highlight: Boolean read FHighlight write SetHighlight;
    property AutoHighlight: Boolean read FAutoHighlight write FAutoHighlight;
    property Image: TBitmap read FImage write SetImage;
    property Interactive: Boolean read FInteractive write SetInteractive;
  end;

  { TControlPart }

  TControlPart = class(TPart)
  protected
    FControl: TWinControl;
    FControlPad: Integer;
    procedure SetControl(const AValue: TWinControl); virtual;
    procedure SetControlBounds; virtual;
    procedure SizeChanged; override;
    procedure ControlParentChanged; override;
    procedure VisibilityChanged; override;
    procedure CardChanged; override;
    procedure PrepareCanvas(Canvas: TCanvas); override;
    procedure CreateControl; virtual;
    procedure DestroyControl; virtual;
    procedure CreateAndSetupControl;
    procedure Reset; override;
    procedure HandleCardEnter; override;
    procedure HandleCardExit; override;
    procedure Activated; override;
  public
    destructor Destroy; override;
    property Control: TWinControl read FControl write SetControl;
  end;

  { TFieldPart }

  TFieldPart = class(TControlPart)
  private
    FBorder: Boolean;
    FBorderColor: TColor;
    FMemo: TMemo;
    FReadOnly: Boolean;
    FWordWrap: Boolean;
    procedure SetBorder(const AValue: Boolean); inline;
    procedure SetBorderColor(const AValue: TColor); inline;
    procedure SetReadOnly(const AValue: Boolean); inline;
    procedure SetWordWrap(const AValue: Boolean);
  protected
    function GetPartEditor: TWinControl; override;
    procedure CreateControl; override;
    procedure PaintInCanvas(Canvas: TCanvas); override;
    function HandleMouseDown(X, Y: Integer; Shift: TShiftState;
       Button: TMouseButton): Boolean; override;
    procedure Activated; override;
    procedure Deactivated; override;
    procedure UIModeChanged; override;
    procedure HandleCardEnter; override;
    procedure HandleCardExit; override;
    procedure SaveData;
    procedure Reset; override;
    procedure FillDuplicate(ATargetPart: TPart); override;
    procedure WritePropertiesToChunkIO(ChunkIO: TChunkIO); override;
    function ReadPropertyFromChunkIO(ChID: Cardinal; ChunkIO: TChunkIO): Boolean; override;
    procedure BeforeWriteToDisk; override;
  public
    procedure SetScriptProperty(AName, AValue: string); override;
    function GetScriptProperty(AName: string): string; override;
    function GetChunkID: Cardinal; override;
    property Border: Boolean read FBorder write SetBorder;
    property BorderColor: TColor read FBorderColor write SetBorderColor;
    property ReadOnly: Boolean read FReadOnly write SetReadOnly;
    property WordWrap: Boolean read FWordWrap write SetWordWrap;
    property Memo: TMemo read FMemo;
  end;

  { TCardData }

  PCardData = ^TCardData;
  TCardData = object
    PartID: Cardinal;
    Data: string;
  end;

  { TCard }

  TCard = class
  private
    FID: Cardinal;
    FStack: TStack;
    FActivePart: TPart;
    FBackground: TCard;
    FControlParent: TWinControl;
    FForeground: TCard;
    FImage: TBitmap;
    FLeftButton: Boolean;
    FMiddleButton: Boolean;
    FMouseX: Integer;
    FMouseY: Integer;
    FName: string;
    FNeedsUpdate: Boolean;
    FParts: TPartList;
    FPartUnderMouse: TPart;
    FMouseLockedPart: TPart;
    FDestroying: Boolean;
    FRightButton: Boolean;
    FScriptHandlerSet: TScriptHandlerSet;
    FScriptPropertySet: TScriptPropertySet;
    FUIMode: TUIMode;
    FMovingPart: TPart;
    FMovingPartArea: TPartArea;
    FMovingPrevRect: TRect;
    FMakingPart: Boolean;
    FAltMaking: Boolean;
    FPrevX, FPrevY: Integer;
    FEntered: Boolean;
    FData: array of TCardData;
    FLoadedBackgroundID: Cardinal; // used when loading and saving
    function GetActivePart: TPart; inline;
    function GetCount: Integer; inline;
    function GetCursor: TCursor;
    function GetNext: TCard; inline;
    function GetPainting: Boolean;
    function GetPartEditing: Boolean; inline;
    function GetParts(AIndex: Integer): TPart; inline;
    function GetPrevious: TCard; inline;
    procedure SetActivePart(const AValue: TPart); inline;
    procedure InvalidatePartReference(APart: TPart); inline;
    procedure PartVisibilityChanged(APart: TPart); inline;
    procedure SetBackground(const AValue: TCard); inline;
    procedure PaintInCanvas(Canvas: TCanvas);
    procedure Invalidate; inline;
    procedure SetControlParent(const AValue: TWinControl);
    procedure UpdateMouseProperties(X, Y: Integer; Button: TMouseButton; ButtonMotion: Integer); inline;
    procedure UIModeChanged;
    function PreparePainter: TObject;
    procedure AllocateImage;
    function GetDataIndex(AID: Cardinal): Integer;
    procedure SetCommonScriptVariables;
    procedure BeforeWriteToDisk;
    procedure AfterReadFromDisk;
  public
    constructor Create;
    destructor Destroy; override;
    procedure PaintIn(TargetCanvas: TCanvas);
    procedure Add(APart: TPart);
    procedure Remove(APart: TPart);
    function HandleMouseDown(X, Y: Integer; Shift: TShiftState; Button: TMouseButton): Boolean;
    function HandleMouseMotion(X, Y: Integer; Shift: TShiftState): Boolean;
    function HandleMouseUp(X, Y: Integer; Shift: TShiftState; Button: TMouseButton): Boolean;
    function HandleDoubleClick: Boolean;
    function HandleKeyDown(KeyCode: Integer; Shift: TShiftState): Boolean;
    function HandleKeyUp(KeyCode: Integer; Shift: TShiftState): Boolean;
    function HandleCharKey(CharKey: TUTF8Char): Boolean;
    procedure HandleCardEnter;
    procedure HandleCardExit;
    procedure TimerTick;
    function PartAt(X, Y: Integer): TPart;
    procedure ActivateAt(X, Y: Integer);
    procedure LockMouse(ToPart: TPart);
    procedure UnlockMouse(FromPart: TPart);
    procedure RaiseToTop(APart: TPart);
    procedure SinkToBottom(APart: TPart);
    procedure ApplyLazFont(ALazFont: TFont);
    function SetupLazFont(ALazFont: TFont): Boolean;
    procedure MakeNewPartAt(X1, Y1, X2, Y2: Integer; AltMode: Boolean=False; ForMode: TUIMode=uiBrowse);
    procedure DuplicateActivePart;
    function GetDataForID(AID: Cardinal; out ACardData: PCardData): Boolean;
    function GetDataForPart(APart: TPart; out ACardData: PCardData): Boolean;
    procedure ReleaseDataForID(AID: Cardinal);
    procedure ReleaseDataForPart(APart: TPart);
    procedure ReleaseAllData;
    function GetNextID: Cardinal; inline;
    function GetSubObjectByID(AID: Cardinal): TObject;
    function GetObjectByID(AID: Cardinal): TObject;
    function GetSubObjectByName(AName: string): TObject;
    function GetObjectByName(AName: string): TObject;
    function CallScript(AHandlerName: string; SetCommonVars: Boolean=True; PassUp: Boolean=True): Boolean;
    procedure SetScriptProperty(AName, AValue: string);
    function GetScriptProperty(AName: string): string;
    procedure WriteToChunkIO(ChunkIO: TChunkIO);
    procedure ReadFromChunkIO(ChunkIO: TChunkIO);
    property ID: Cardinal read FID;
    property Name: string read FName write FName;
    property ScriptHandlerSet: TScriptHandlerSet read FScriptHandlerSet;
    property Stack: TStack read FStack;
    property Destroying: Boolean read FDestroying;
    property ControlParent: TWinControl read FControlParent write SetControlParent;
    property UIMode: TUIMode read FUIMode;
    property Entered: Boolean read FEntered;
    property PartEditing: Boolean read GetPartEditing;
    property Painting: Boolean read GetPainting;
    property ActivePart: TPart read GetActivePart write SetActivePart;
    property PartUnderMouse: TPart read FPartUnderMouse;
    property MouseLockedPart: TPart read FMouseLockedPart;
    property MouseX: Integer read FMouseX;
    property MouseY: Integer read FMouseY;
    property LeftButton: Boolean read FLeftButton;
    property MiddleButton: Boolean read FMiddleButton;
    property RightButton: Boolean read FRightButton;
    property NeedsUpdate: Boolean read FNeedsUpdate;
    property Cursor: TCursor read GetCursor;
    property Background: TCard read FBackground write SetBackground;
    property Foreground: TCard read FForeground;
    property Previous: TCard read GetPrevious;
    property Next: TCard read GetNext;
    property Image: TBitmap read FImage;
    property Parts[AIndex: Integer]: TPart read GetParts; default;
    property Count: Integer read GetCount;
  end;

  TCardList = specialize TFPGObjectList<TCard>;

  { TStack }

  TStack = class
  private
    FCards: TCardList;
    FBackgrounds: TCardList;
    FControlParent: TWinControl;
    FScriptHandlerSet: TScriptHandlerSet;
    FUIMode: TUIMode;
    FNextID: Cardinal;
    function GetBackgroundCount: Integer; inline;
    function GetBackgrounds(AIndex: Integer): TCard; inline;
    function GetCards(AIndex: Integer): TCard; inline;
    function GetCount: Integer; inline;
    function GetFirstCard: TCard; inline;
    function GetLastCard: TCard; inline;
    procedure SetControlParent(const AValue: TWinControl);
    procedure SetUIMode(const AValue: TUIMode);
    procedure SetupNewCard(ACard: TCard);
    procedure BeforeWriteToDisk;
    procedure AfterReadFromDisk;
  public
    constructor Create;
    destructor Destroy; override;
    function Add(BgCard: TCard; AtIndex: Integer=-1): TCard;
    procedure Remove(ACard: TCard);
    function IndexOf(ACard: TCard): Integer;
    function PreviousCardOf(ACard: TCard): TCard;
    function NextCardOf(ACard: TCard): TCard;
    function AddBackground: TCard;
    procedure RemoveBackground(ACard: TCard);
    function IndexOfBackground(ACard: TCard): Integer;
    function GetNextID: Cardinal;
    function GetObjectByID(AID: Cardinal): TObject;
    function GetObjectByName(AName: string): TObject;
    function GetCardByName(AName: string): TCard;
    function CallScript(AHandlerName: string): Boolean;
    procedure TimerTick;
    procedure WriteToChunkIO(ChunkIO: TChunkIO);
    procedure ReadFromChunkIO(ChunkIO: TChunkIO);
    procedure WriteToStream(AStream: TStream);
    procedure ReadFromStream(AStream: TStream);
    procedure WriteToFile(AFileName: string);
    procedure ReadFromFile(AFileName: string);
    property ControlParent: TWinControl read FControlParent write SetControlParent;
    property UIMode: TUIMode read FUIMode write SetUIMode;
    property ScriptHandlerSet: TScriptHandlerSet read FScriptHandlerSet;
    property Cards[AIndex: Integer]: TCard read GetCards; default;
    property Count: Integer read GetCount;
    property FirstCard: TCard read GetFirstCard;
    property LastCard: TCard read GetLastCard;
    property Backgrounds[AIndex: Integer]: TCard read GetBackgrounds;
    property BackgroundCount: Integer read GetBackgroundCount;
  end;

var
  UseGrid: Boolean;
  GridSize: Integer;

function CursorForPartArea(const APartArea: TPartArea): TCursor;
function GridSnap(V: Integer): Integer;

implementation

uses
  PartInspectorUnit, PaintSettingsUnit, ButtonPartEditorUnit,
  FieldPartEditorUnit, Forms, IntfGraphics, FPimage,
  zstream;

type

  { TPainter }

  TPainter = class
  public
    Card: TCard;
    Canvas: TCanvas;
    SavedBitmap: TBitmap;
    constructor Create;
    destructor Destroy; override;
    function LeftButton: Boolean; inline;
    function MiddleButton: Boolean; inline;
    function RightButton: Boolean; inline;
    procedure SaveBitmap;
    procedure RestoreBitmap;
    procedure ReleaseBitmap;
    procedure MouseDown(X, Y: Integer; Button: TMouseButton; Shift: TShiftState); virtual; abstract;
    procedure MouseMotion(X, Y: Integer; Shift: TShiftState); virtual; abstract;
    procedure MouseUp(X, Y: Integer; Button: TMouseButton; Shift: TShiftState); virtual; abstract;
  end;

  { TPenPainter }

  TPenPainter = class(TPainter)
  public
    PrevX, PrevY: Integer;
    procedure PenLine(X1, Y1, X2, Y2: Integer); virtual;
    procedure MouseDown(X, Y: Integer; Button: TMouseButton; Shift: TShiftState); override;
    procedure MouseMotion(X, Y: Integer; Shift: TShiftState); override;
    procedure MouseUp(X, Y: Integer; Button: TMouseButton; Shift: TShiftState); override;
  end;

  { TShapePainter }
  TShapePainterType = (sptLine, sptBox, sptOval);

  TShapePainter = class(TPainter)
  public
    ShapeType: TShapePainterType;
    StartX, StartY: Integer;
    constructor Create(AShapeType: TShapePainterType);
    procedure DrawShape(X1, Y1, X2, Y2: Integer); virtual;
    procedure MouseDown(X, Y: Integer; Button: TMouseButton; Shift: TShiftState); override;
    procedure MouseMotion(X, Y: Integer; Shift: TShiftState); override;
    procedure MouseUp(X, Y: Integer; Button: TMouseButton; Shift: TShiftState); override;
  end;

  { TFloodFillPainter }

  TFloodFillPainter = class(TPainter)
  public
    procedure MouseDown(X, Y: Integer; Button: TMouseButton; Shift: TShiftState); override;
    procedure MouseMotion(X, Y: Integer; Shift: TShiftState); override;
    procedure MouseUp(X, Y: Integer; Button: TMouseButton; Shift: TShiftState); override;
  end;

var
  PenPainter: TPenPainter;
  LinePainter: TShapePainter;
  BoxPainter: TShapePainter;
  OvalPainter: TShapePainter;
  FloodFillPainter: TFloodFillPainter;
  FillNextID: PCardinal;

function GetPainterForUIMode(UIMode: TUIMode): TPainter;
begin
  case UIMode of
    uiPen: Result:=PenPainter;
    uiLine: Result:=LinePainter;
    uiBox: Result:=BoxPainter;
    uiOval: Result:=OvalPainter;
    uiFloodFill: Result:=FloodFillPainter;
    else Result:=nil;
  end;
end;

function CursorForPartArea(const APartArea: TPartArea): TCursor;
begin
  case APartArea of
    paLeftTop, paRightBottom: Result:=crSizeNWSE;
    paRightTop, paLeftBottom: Result:=crSizeNESW;
    paLeft, paRight: Result:=crSizeWE;
    paTop, paBottom: Result:=crSizeNS;
    paCenter: Result:=crSizeAll;
    else Result:=crDefault;
  end;
end;

function GridSnap(V: Integer): Integer;
begin
  if UseGrid then Result:=Trunc(V/GridSize)*GridSize else Result:=V;
end;

procedure WriteImageToChunkIO(ChunkIO: TChunkIO; Image: TBitmap);
var
  x, y: Integer;
  P: TFPColor;
  LazImg: TLazIntfImage;
begin
  ChunkIO.WriteWord(ChunkID('XRES'), Image.Width);
  ChunkIO.WriteWord(ChunkID('YRES'), Image.Height);
  ChunkIO.BeginChunk(ChunkID('DATA'));
  LazImg:=Image.CreateIntfImage;
  for y:=0 to Image.Height - 1 do
    for x:=0 to Image.Width - 1 do begin
      P:=LazImg.Colors[x, y];
      ChunkIO.Stream.WriteByte(P.red shr 8);
      ChunkIO.Stream.WriteByte(P.green shr 8);
      ChunkIO.Stream.WriteByte(P.blue shr 8);
    end;
  ChunkIO.EndChunk();
  FreeAndNil(LazImg);
end;

procedure ReadImageFromChunkIO(ChunkIO: TChunkIO; Image: TBitmap);
var
  x, y: Integer;
  R, G, B: Byte;
  ID, Size: Cardinal;
  Width, Height: Integer;
  LazImg: TLazIntfImage;
  RawImg: TRawImage;
begin
  while ChunkIO.NextChunk(ID, Size) do begin
    if ID=ChunkID('XRES') then
      Width:=ChunkIO.ReadWord
    else if ID=ChunkID('YRES') then
      Height:=ChunkIO.ReadWord
    else if ID=ChunkID('DATA') then begin
      RawImg.Init;
      {$IFDEF LINUX}
      RawImg.Description.Init_BPP32_B8G8R8_BIO_TTB(Width, Height);
      {$ELSE}
      RawImg.Description.Init_BPP24_B8G8R8_BIO_TTB(Width, Height);
      {$ENDIF}
      RawImg.CreateData(True);
      LazImg:=TLazIntfImage.Create(Width, Height);
      LazImg.SetRawImage(RawImg);
      for y:=0 to Height - 1 do
        for x:=0 to Width - 1 do begin
          {$IFDEF DARWIN}
          B:=ChunkIO.Stream.ReadByte;
          G:=ChunkIO.Stream.ReadByte;
          R:=ChunkIO.Stream.ReadByte;
          {$ELSE}
          R:=ChunkIO.Stream.ReadByte;
          G:=ChunkIO.Stream.ReadByte;
          B:=ChunkIO.Stream.ReadByte;
          {$ENDIF}
          LazImg.Colors[x, y]:=FPColor(R shl 8, G shl 8, B shl 8);
        end;
      ChunkIO.Stream.ReadDWord; // skip ChunkEndID
      Image.LoadFromIntfImage(LazImg);
      FreeAndNil(LazImg);
    end else ChunkIO.SkipChunk(Size);
  end;
end;

function PrettyKeyCodeName(Key: Integer): string;
begin
  case Key of
    VK_BACK: Result:='backspace';
    VK_TAB: Result:='tab';
    VK_CLEAR: Result:='clear';
    VK_RETURN: Result:='enter';
    VK_SHIFT: Result:='shift';
    VK_CONTROL: Result:='control';
    VK_MENU: Result:='alt';
    VK_PAUSE: Result:='pause';
    VK_ESCAPE: Result:='escape';
    VK_SPACE: Result:='space';
    VK_PRIOR: Result:='page-up';
    VK_NEXT: Result:='page-down';
    VK_END: Result:='end';
    VK_HOME: Result:='home';
    VK_LEFT: Result:='left';
    VK_UP: Result:='up';
    VK_RIGHT: Result:='right';
    VK_DOWN: Result:='down';
    VK_PRINT: Result:='print';
    VK_INSERT: Result:='insert';
    VK_DELETE: Result:='delete';
    VK_HELP: Result:='help';
    VK_0: Result:='0';
    VK_1: Result:='1';
    VK_2: Result:='2';
    VK_3: Result:='3';
    VK_4: Result:='4';
    VK_5: Result:='5';
    VK_6: Result:='6';
    VK_7: Result:='7';
    VK_8: Result:='8';
    VK_9: Result:='9';
    VK_A: Result:='a';
    VK_B: Result:='b';
    VK_C: Result:='c';
    VK_D: Result:='d';
    VK_E: Result:='e';
    VK_F: Result:='f';
    VK_G: Result:='g';
    VK_H: Result:='h';
    VK_I: Result:='i';
    VK_J: Result:='j';
    VK_K: Result:='k';
    VK_L: Result:='l';
    VK_M: Result:='m';
    VK_N: Result:='n';
    VK_O: Result:='o';
    VK_P: Result:='p';
    VK_Q: Result:='q';
    VK_R: Result:='r';
    VK_S: Result:='s';
    VK_T: Result:='t';
    VK_U: Result:='u';
    VK_V: Result:='v';
    VK_W: Result:='w';
    VK_X: Result:='x';
    VK_Y: Result:='y';
    VK_Z: Result:='z';
    VK_LWIN: Result:='left-win';
    VK_RWIN: Result:='right-win';
    VK_NUMPAD0: Result:='numpad0';
    VK_NUMPAD1: Result:='numpad1';
    VK_NUMPAD2: Result:='numpad2';
    VK_NUMPAD3: Result:='numpad3';
    VK_NUMPAD4: Result:='numpad4';
    VK_NUMPAD5: Result:='numpad5';
    VK_NUMPAD6: Result:='numpad6';
    VK_NUMPAD7: Result:='numpad7';
    VK_NUMPAD8: Result:='numpad8';
    VK_NUMPAD9: Result:='numpad9';
    VK_MULTIPLY: Result:='multiply';
    VK_ADD: Result:='add';
    VK_SEPARATOR: Result:='separator';
    VK_SUBTRACT: Result:='subtract';
    VK_DECIMAL: Result:='decimal';
    VK_DIVIDE: Result:='divide';
    VK_F1: Result:='f1';
    VK_F2: Result:='f2';
    VK_F3: Result:='f3';
    VK_F4: Result:='f4';
    VK_F5: Result:='f5';
    VK_F6: Result:='f6';
    VK_F7: Result:='f7';
    VK_F8: Result:='f8';
    VK_F9: Result:='f9';
    VK_F10: Result:='f10';
    VK_F11: Result:='f11';
    VK_F12: Result:='f12';
    VK_NUMLOCK: Result:='numlock';
    VK_SCROLL: Result:='scroll';
    else Result:='k' + IntToStr(Key);
  end;
end;

{ TCardFont }

procedure TCardFont.Assign(Font: TFont);
begin
  Bold:=fsBold in Font.Style;
  Italic:=fsItalic in Font.Style;
  Underline:=fsUnderline in Font.Style;
  Pitch:=Font.Pitch;
  Size:=Font.Size;
  Name:=Font.Name;
  Color:=Font.Color;
end;

procedure TCardFont.Setup(Font: TFont);
var
  Style: set of TFontStyle;
begin
  Style:=[];
  if Bold then Style:=Style + [fsBold];
  if Italic then Style:=Style + [fsItalic];
  if Underline then Style:=Style + [fsUnderline];
  Font.Style:=Style;
  Font.Pitch:=Pitch;
  Font.Size:=Size;
  Font.Name:=Name;
  Font.Color:=Color;
end;

procedure TCardFont.WriteToChunkIO(ChunkIO: TChunkIO);
begin
  with ChunkIO do begin
    WriteString(ChunkID('NAME'), Name);
    WriteInteger(ChunkID('SIZE'), Size);
    WriteBoolean(ChunkID('BOLD'), Bold);
    WriteBoolean(ChunkID('ITLC'), Italic);
    WriteBoolean(ChunkID('UNDR'), Underline);
    WriteByte(ChunkID('PICH'), Byte(Pitch));
    WriteCardinal(ChunkID('COLO'), Cardinal(Color));
  end;
end;

procedure TCardFont.ReadFromChunkIO(ChunkIO: TChunkIO);
var
  ID, ChunkSize: Cardinal;
begin
  Name:='';
  Size:=8;
  Bold:=False;
  Italic:=False;
  Underline:=False;
  Pitch:=fpVariable;
  Color:=clWindowText;
  with ChunkIO do begin
    while NextChunk(ID, ChunkSize) do begin
      if ID=ChunkID('NAME') then
        Name:=ReadString
      else if ID=ChunkID('SIZE') then
        Size:=ReadInteger
      else if ID=ChunkID('BOLD') then
        Bold:=ReadBoolean
      else if ID=ChunkID('ITLC') then
        Italic:=ReadBoolean
      else if ID=ChunkID('UNDR') then
        Underline:=ReadBoolean
      else if ID=ChunkID('PICH') then
        Pitch:=TFontPitch(ReadByte)
      else if ID=ChunkID('COLO') then
        Color:=TColor(ReadCardinal)
      else SkipChunk(ChunkSize);
    end;
  end;
end;

{ TFieldPart }

procedure TFieldPart.SetBorder(const AValue: Boolean);
begin
  if FBorder=AValue then Exit;
  FBorder:=AValue;
  Invalidate;
end;

procedure TFieldPart.SetBorderColor(const AValue: TColor);
begin
  if FBorderColor=AValue then Exit;
  FBorderColor:=AValue;
  Invalidate;
end;

procedure TFieldPart.SetReadOnly(const AValue: Boolean);
begin
  if FReadOnly=AValue then Exit;
  FReadOnly:=AValue;
  if Assigned(Control) then Memo.ReadOnly:=AValue;
  Invalidate;
end;

procedure TFieldPart.SetWordWrap(const AValue: Boolean);
begin
  if FWordWrap=AValue then Exit;
  FWordWrap:=AValue;
  if Assigned(Control) then Memo.WordWrap:=AValue;
  Invalidate;
end;

function TFieldPart.GetPartEditor: TWinControl;
begin
  Result:=GetFieldPartEditor;
  TFieldPartEditor(Result).Part:=Self;
end;

procedure TFieldPart.CreateControl;
begin
  FMemo:=TMemo.Create(nil);
  FMemo.Text:='';
  FMemo.BorderStyle:=bsNone;
  FMemo.Color:=clWhite;
  FMemo.Visible:=False;
  FMemo.WantReturns:=True;
  FMemo.WantTabs:=True;
  FMemo.WordWrap:=False;
  FMemo.ScrollBars:=ssNone;
  FControlPad:=3;
  FControl:=FMemo;
end;

procedure TFieldPart.PaintInCanvas(Canvas: TCanvas);
var
  TextStyle: TTextStyle;
begin
  Canvas.Pen.Style:=psSolid;
  Canvas.Pen.Width:=1;
  if Border then begin
    Canvas.Pen.Color:=BorderColor;
  end else
    Canvas.Pen.Color:=Color;
  Canvas.Brush.Style:=bsSolid;
  Canvas.Brush.Color:=Color;
  Canvas.Rectangle(X1, Y1, X2, Y2);
  if Assigned(Control) then begin
    if not Memo.Visible then begin
      TextStyle.Alignment:=taLeftJustify;
      TextStyle.Clipping:=True;
      TextStyle.ExpandTabs:=True;
      TextStyle.Layout:=tlTop;
      TextStyle.Opaque:=False;
      TextStyle.RightToLeft:=False;
      TextStyle.ShowPrefix:=False;
      TextStyle.SingleLine:=False;
      TextStyle.SystemFont:=False;
      TextStyle.Wordbreak:=WordWrap;
      Canvas.TextRect(Rect(X1 + 3, Y1 + 3, X2 - 3, Y2 - 3), X1 + 3, Y1 + 3, FMemo.Text, TextStyle);
    end else
      FMemo.Color:=Color;
  end;
end;

function TFieldPart.HandleMouseDown(X, Y: Integer; Shift: TShiftState;
  Button: TMouseButton): Boolean;
begin
  Activate;
  Result:=inherited HandleMouseDown(X, Y, Shift, Button);
end;

procedure TFieldPart.Activated;
begin
  inherited Activated;
  if Assigned(Control) then begin
    Memo.Visible:=UIMode=uiBrowse;
    if Memo.Visible then begin
      if Memo.Enabled then Memo.SetFocus;
    end;
  end;
  Invalidate;
end;

procedure TFieldPart.Deactivated;
begin
  if Assigned(Control) then Memo.Visible:=False;
  Invalidate;
  inherited Deactivated;
end;

procedure TFieldPart.UIModeChanged;
begin
  inherited UIModeChanged;
  if Assigned(Control) then Memo.Visible:=Active and (UIMode=uiBrowse);
  Invalidate;
end;

procedure TFieldPart.HandleCardEnter;
var
  Data: PCardData;
begin
  inherited HandleCardEnter;
  if Assigned(Control) then
    if Assigned(Card) and Card.GetDataForPart(Self, Data) then
      FMemo.Text:=Data^.Data
    else
      FMemo.Text:='';
end;

procedure TFieldPart.HandleCardExit;
begin
  SaveData;
  inherited HandleCardExit;
end;

procedure TFieldPart.SaveData;
var
  Data: PCardData;
begin
  if Assigned(Control) and Assigned(Card) and Card.GetDataForPart(Self, Data) then
    Data^.Data:=FMemo.Text;
end;

procedure TFieldPart.Reset;
begin
  inherited Reset;
  Border:=True;
  BorderColor:=clBlack;
  WordWrap:=False;
  ReadOnly:=False;
end;

procedure TFieldPart.FillDuplicate(ATargetPart: TPart);
begin
  inherited FillDuplicate(ATargetPart);
  TFieldPart(ATargetPart).Border:=Border;
  TFieldPart(ATargetPart).BorderColor:=BorderColor;
  TFieldPart(ATargetPart).ReadOnly:=ReadOnly;
  TFieldPart(ATargetPart).WordWrap:=WordWrap;
  TFieldPart(ATargetPart).Memo.Lines.Assign(Memo.Lines);
end;

procedure TFieldPart.WritePropertiesToChunkIO(ChunkIO: TChunkIO);
begin
  inherited WritePropertiesToChunkIO(ChunkIO);
  with ChunkIO do begin
    WriteBoolean(ChunkID('BRDR'), Border);
    WriteBoolean(ChunkID('RDOL'), ReadOnly);
    WriteBoolean(ChunkID('WRDW'), WordWrap);
    WriteCardinal(ChunkID('BRCL'), Cardinal(BorderColor));
    WriteString(ChunkID('TEXT'), Memo.Text);
  end;
end;

function TFieldPart.ReadPropertyFromChunkIO(ChID: Cardinal; ChunkIO: TChunkIO): Boolean;
begin
  Result:=True;
  with ChunkIO do begin
    if ChID=ChunkID('BRDR') then
      Border:=ReadBoolean
    else if ChID=ChunkID('RDOL') then
      ReadOnly:=ReadBoolean
    else if ChID=ChunkID('WRDW') then
      WordWrap:=ReadBoolean
    else if ChID=ChunkID('BRCL') then
      BorderColor:=TColor(ReadCardinal)
    else if ChID=ChunkID('TEXT') then
      Memo.Text:=ReadString
    else Result:=inherited ReadPropertyFromChunkIO(ChID, ChunkIO);
  end;
end;

procedure TFieldPart.BeforeWriteToDisk;
begin
  inherited BeforeWriteToDisk;
  SaveData;
end;

procedure TFieldPart.SetScriptProperty(AName, AValue: string);
begin
  if AName='draw-border' then Border:=not ((AValue='') or (AValue='0') or (AValue='no'))
  else if AName='read-only' then ReadOnly:=not ((AValue='') or (AValue='0') or (AValue='no'))
  else if AName='word-wrap' then WordWrap:=not ((AValue='') or (AValue='0') or (AValue='no'))
  else if AName='text' then Memo.Text:=AValue
  else inherited SetScriptProperty(AName, AValue);
end;

function TFieldPart.GetScriptProperty(AName: string): string;
begin
  if AName='draw-border' then Result:=IntToStr(Integer(Border))
  else if AName='read-only' then Result:=IntToStr(Integer(ReadOnly))
  else if AName='word-wrap' then Result:=IntToStr(Integer(WordWrap))
  else if AName='text' then Result:=Memo.Text
  else Result:=inherited GetScriptProperty(AName);
end;

function TFieldPart.GetChunkID: Cardinal;
begin
  Result:=ChunkID(FieldPartChunkID);
end;

{ TControlPart }

procedure TControlPart.SetControl(const AValue: TWinControl);
begin
  if FControl=AValue then Exit;
  FControl:=AValue;
end;

procedure TControlPart.SetControlBounds;
begin
  if Assigned(Control) then begin
    if (X2 - X1 > 1 + FControlPad*2) and (Y2 - Y1 > 1 + FControlPad*2) then begin
      Control.SetBounds(X1 + FControlPad, Y1 + FControlPad, X2 - X1 - FControlPad*2, Y2 - Y1 - FControlPad*2);
    end;
  end;
end;

procedure TControlPart.SizeChanged;
begin
  SetControlBounds;
end;

procedure TControlPart.ControlParentChanged;
begin
  if Assigned(Card) and Assigned(Control) and Card.Entered then Control.Parent:=Card.ControlParent;
end;

procedure TControlPart.VisibilityChanged;
begin
  inherited VisibilityChanged;
  if Assigned(Control) then Control.Visible:=Visible;
end;

procedure TControlPart.CardChanged;
begin
  inherited CardChanged;
  if Assigned(Card) and Assigned(Control) and Card.Entered and (Control.Parent <> Card.ControlParent) then Control.Parent:=Card.ControlParent;
end;

procedure TControlPart.PrepareCanvas(Canvas: TCanvas);
begin
  inherited PrepareCanvas(Canvas);
  if Assigned(Control) then Font.Setup(Control.Font);
end;

procedure TControlPart.CreateControl;
begin
end;

procedure TControlPart.DestroyControl;
begin
  FreeAndNil(FControl);
end;

procedure TControlPart.CreateAndSetupControl;
begin
  CreateControl;
  if Assigned(Control) then begin
    if Assigned(Card) and Card.Entered then Control.Parent:=Card.ControlParent;
  end;
end;

procedure TControlPart.Reset;
begin
  inherited Reset;
  DestroyControl;
  CreateAndSetupControl;
end;

procedure TControlPart.HandleCardEnter;
begin
  inherited HandleCardEnter;
  if Assigned(Card) and Assigned(Control) then Control.Parent:=Card.ControlParent;
end;

procedure TControlPart.HandleCardExit;
begin
  if Assigned(Card) and Assigned(Control) then Control.Parent:=nil;
  inherited HandleCardExit;
end;

procedure TControlPart.Activated;
begin
  inherited Activated;
  SetControlBounds;
end;

destructor TControlPart.Destroy;
begin
  DestroyControl;
  inherited Destroy;
end;

{ TFloodFillPainter }

procedure TFloodFillPainter.MouseDown(X, Y: Integer; Button: TMouseButton;
  Shift: TShiftState);
begin
  X:=GridSnap(X);
  Y:=GridSnap(Y);
  Canvas.FloodFill(X, Y, Canvas.Pixels[X, Y], fsSurface);
  Card.Invalidate;
end;

procedure TFloodFillPainter.MouseMotion(X, Y: Integer; Shift: TShiftState);
begin
end;

procedure TFloodFillPainter.MouseUp(X, Y: Integer; Button: TMouseButton;
  Shift: TShiftState);
begin
end;

constructor TShapePainter.Create(AShapeType: TShapePainterType);
begin
  ShapeType:=AShapeType;
end;

procedure TShapePainter.DrawShape(X1, Y1, X2, Y2: Integer);
begin
  case ShapeType of
    sptLine: Canvas.Line(X1, Y1, X2, Y2);
    sptBox: Canvas.Rectangle(X1, Y1, X2, Y2);
    sptOval: Canvas.Ellipse(X1, Y1, X2, Y2);
  end;
end;

procedure TShapePainter.MouseDown(X, Y: Integer; Button: TMouseButton;
  Shift: TShiftState);
begin
  StartX:=GridSnap(X);
  StartY:=GridSnap(Y);
  SaveBitmap;
end;

procedure TShapePainter.MouseMotion(X, Y: Integer; Shift: TShiftState);
begin
  if LeftButton then begin
    X:=GridSnap(X);
    Y:=GridSnap(Y);
    RestoreBitmap;
    DrawShape(StartX, StartY, X, Y);
    Card.Invalidate;
  end;
end;

procedure TShapePainter.MouseUp(X, Y: Integer; Button: TMouseButton;
  Shift: TShiftState);
begin
  ReleaseBitmap;
end;

{ TPainter }

constructor TPainter.Create;
begin
end;

destructor TPainter.Destroy;
begin
  FreeAndNil(SavedBitmap);
  inherited Destroy;
end;

function TPainter.LeftButton: Boolean;
begin
  Result:=Card.LeftButton;
end;

function TPainter.MiddleButton: Boolean;
begin
  Result:=Card.MiddleButton;
end;

function TPainter.RightButton: Boolean;
begin
  Result:=Card.RightButton;
end;

procedure TPainter.SaveBitmap;
begin
  if not Assigned(SavedBitmap) then begin
    SavedBitmap:=TBitmap.Create;
//    SavedBitmap.PixelFormat:=pf32bit;
    SavedBitmap.SetSize(640, 480);
  end;
  SavedBitmap.Canvas.CopyRect(Rect(0, 0, 640, 480), Canvas, Rect(0, 0, 640, 480));
end;

procedure TPainter.RestoreBitmap;
begin
  if Assigned(SavedBitmap) then
    Canvas.CopyRect(Rect(0, 0, 640, 480), SavedBitmap.Canvas, Rect(0, 0, 640, 480));
end;

procedure TPainter.ReleaseBitmap;
begin
  FreeAndNil(SavedBitmap);
end;

{ TPenPainter }

procedure TPenPainter.PenLine(X1, Y1, X2, Y2: Integer);
begin
  Canvas.Line(X1, Y1, X2, Y2);
end;

procedure TPenPainter.MouseDown(X, Y: Integer; Button: TMouseButton;
  Shift: TShiftState);
begin
  PrevX:=GridSnap(X);
  PrevY:=GridSnap(Y);
  if Canvas.Pen.Width < 2 then PenLine(PrevX, PrevY, PrevX + 1, PrevY);
  Card.Invalidate;
end;

procedure TPenPainter.MouseMotion(X, Y: Integer; Shift: TShiftState);
begin
  if LeftButton then begin
    X:=GridSnap(X);
    Y:=GridSnap(Y);
    PenLine(PrevX, PrevY, X, Y);
    PrevX:=X;
    PrevY:=Y;
    Card.Invalidate;
  end;
end;

procedure TPenPainter.MouseUp(X, Y: Integer; Button: TMouseButton; Shift: TShiftState);
begin
end;

{ TButtonPart }

procedure TButtonPart.SetStyle(const AValue: TButtonPartStyle);
begin
  if FStyle=AValue then Exit;
  FStyle:=AValue;
  Invalidate;
end;

function TButtonPart.GetAllowInteraction: Boolean;
begin
  Result:=(inherited GetAllowInteraction) and Interactive;
end;

function TButtonPart.GetPartEditor: TWinControl;
begin
  Result:=GetButtonPartEditor;
  TButtonPartEditor(Result).Part:=Self;
end;

function TButtonPart.GetCursor: TCursor;
begin
  if AllowInteraction then
    Result:=crHandPoint
  else
    Result:=inherited GetCursor;
end;

procedure TButtonPart.SetDrawName(const AValue: Boolean);
begin
  if FDrawName=AValue then Exit;
  FDrawName:=AValue;
  Invalidate;
end;

procedure TButtonPart.SetHighlight(const AValue: Boolean);
begin
  if FHighlight=AValue then Exit;
  FHighlight:=AValue;
  Invalidate;
end;

procedure TButtonPart.SetImage(AValue: TBitmap);
begin
  if FImage=AValue then Exit;
  FImage:=AValue;
  Invalidate;
end;

procedure TButtonPart.SetInteractive(AValue: Boolean);
begin
  if FInteractive=AValue then Exit;
  FInteractive:=AValue;
end;

procedure TButtonPart.PaintInCanvas(Canvas: TCanvas);
var
  Extent: TSize;
  Y: Integer;
begin
  if Style in [bpsOpaque, bpsRectangle, bpsNative] then begin
    if Highlight then begin
      Canvas.Brush.Color:=clBlack;
      Canvas.Pen.Color:=clBlack;
    end else begin
      Canvas.Brush.Color:=Color;
      Canvas.Pen.Color:=Color;
    end;
    Canvas.Pen.Width:=1;
    Canvas.Rectangle(X1, Y1, X2, Y2);
  end;
  if Highlight then Canvas.Pen.Color:=Color else Canvas.Pen.Color:=clBlack;
  case Style of
    bpsRectangle: begin
      Canvas.Brush.Style:=bsSolid;
      Canvas.Rectangle(X1, Y1, X2, Y2);
    end;
    bpsRoundRect, bpsNative: begin
      Canvas.Brush.Style:=bsSolid;
      if Highlight then Canvas.Brush.Color:=clBlack else Canvas.Brush.Color:=Color;
      Canvas.RoundRect(X1, Y1, X2, Y2, 10, 10);
    end;
    bpsShadowed: begin
      Canvas.Brush.Style:=bsSolid;
      Canvas.Brush.Color:=clBlack;
      Canvas.Pen.Color:=clBlack;
      Canvas.RoundRect(X1 + 2, Y1 + 2, X2 + 2, Y2 + 2, 10, 10);
      Canvas.Brush.Color:=Color;
      Canvas.RoundRect(X1, Y1, X2, Y2, 10, 10);
      if Highlight then begin
        Canvas.Pen.Color:=clBlack;
        Canvas.Brush.Color:=clBlack;
        Canvas.RoundRect(X1 + 2, Y1 + 2, X2 - 2, Y2 - 2, 6, 6);
      end;
    end;
  end;
  if Assigned(Image) then begin
    if Style=bpsInvisible then
      Canvas.StretchDraw(Rect(X1, Y1, X2, Y2), Image)
    else
      Canvas.StretchDraw(Rect(X1 + 2, Y1 + 2, X2 - 2, Y2 - 2), Image);
  end;
  if FDrawName then begin
    if Highlight and (Style <> bpsInvisible) then Canvas.Font.Color:=Color else Canvas.Font.Color:=Font.Color;
    Extent:=Canvas.TextExtent(Name);
    Canvas.Brush.Style:=bsClear;
    Canvas.Clipping:=True;
    Canvas.ClipRect:=Rect(X1 + 2, Y1 + 2, X2 - 2, Y2 - 2);
    Canvas.TextOut(X1 + ((X2 - X1) - Extent.cx) div 2,
                   Y1 + ((Y2 - Y1) - Extent.cy) div 2,
                   Name);
    Canvas.Clipping:=False;
  end;
  if (UIMode <> uiButton) and (UIMode <> uiBrowse) and (Style=bpsInvisible) then begin
    Canvas.Brush.Style:=bsClear;
    Canvas.Pen.Style:=psSolid;
    Canvas.Pen.Color:=clBlack;
    Canvas.Rectangle(X1 - 1, Y1 - 1, X2 + 1, Y2 + 1);
  end;
  if Highlight and (Style=bpsInvisible) then begin
    Canvas.Brush.Style:=bsClear;
    Canvas.Pen.Style:=psSolid;
    Canvas.Pen.Color:=clBlack;
    Canvas.Pen.Mode:=pmNotMerge;
    for Y:=Y1 to Y2 do
      Canvas.Line(X1, Y, X2, Y);
    Canvas.Pen.Mode:=pmCopy;
  end;
end;

procedure TButtonPart.PaintEditingOverlay(Canvas: TCanvas);
begin
  if Active or (Style=bpsInvisible) then
    inherited PaintEditingOverlay(Canvas);
end;

procedure TButtonPart.Reset;
begin
  inherited Reset;
  Name:='New Button';
  Style:=bpsShadowed;
  DrawName:=True;
  AutoHighlight:=True;
  Font.Bold:=True;
  FreeAndNil(FImage);
  FImage:=nil;
  Interactive:=True;
end;

function TButtonPart.HandleMouseDown(X, Y: Integer; Shift: TShiftState;
  Button: TMouseButton): Boolean;
begin
  if (Button=mbLeft) and AutoHighlight then begin
    Activate;
    Highlight:=True;
    Result:=True;
    LockMouse;
  end;
  if not Result then Result:=inherited HandleMouseDown(X, Y, Shift, Button);
  Activate;
end;

function TButtonPart.HandleMouseUp(X, Y: Integer; Shift: TShiftState;
  Button: TMouseButton): Boolean;
begin
  if Button=mbLeft then begin
    if AutoHighlight then begin
      Highlight:=False;
      Result:=True;
    end;
    UnlockMouse;
    if UnderMouse then Result:=CallScript(EV_ACTION);
  end;
  if not Result then Result:=inherited HandleMouseUp(X, Y, Shift, Button);
end;

function TButtonPart.HandleMouseEnter(X, Y: Integer): Boolean;
begin
  Result:=inherited HandleMouseEnter(X, Y);
  if not Result and LeftButton and AutoHighlight and not Highlight then Highlight:=True;
end;

function TButtonPart.HandleMouseExit(X, Y: Integer): Boolean;
begin
  Result:=inherited HandleMouseExit(X, Y);
  if not Result and LeftButton and AutoHighlight and Highlight then Highlight:=False;
end;

procedure TButtonPart.FillDuplicate(ATargetPart: TPart);
begin
  inherited FillDuplicate(ATargetPart);
  TButtonPart(ATargetPart).Style:=Style;
  TButtonPart(ATargetPart).DrawName:=DrawName;
  TButtonPart(ATargetPart).Highlight:=Highlight;
  TButtonPart(ATargetPart).AutoHighlight:=AutoHighlight;
  TButtonPart(ATargetPart).Interactive:=Interactive;
  if Assigned(Image) then begin
    TButtonPart(ATargetPart).Image:=TBitmap.Create;
    TButtonPart(ATargetPart).Image.Assign(Image);
  end;
end;

procedure TButtonPart.WritePropertiesToChunkIO(ChunkIO: TChunkIO);
begin
  inherited WritePropertiesToChunkIO(ChunkIO);
  with ChunkIO do begin
    WriteByte(ChunkID('STYL'), Byte(Style));
    WriteBoolean(ChunkID('DRWN'), DrawName);
    WriteBoolean(ChunkID('AUHI'), AutoHighlight);
    if Assigned(Image) then begin
      ChunkIO.BeginChunk(ChunkID('IMAG'));
      WriteImageToChunkIO(ChunkIO, Image);
      ChunkIO.EndChunk;
    end;
    WriteBoolean(ChunKID('INTR'), Interactive);
  end;
end;

function TButtonPart.ReadPropertyFromChunkIO(ChID: Cardinal; ChunkIO: TChunkIO): Boolean;
begin
  Result:=True;
  with ChunkIO do begin
    if ChID=ChunkID('STYL') then
      Style:=TButtonPartStyle(ReadByte)
    else if ChID=ChunkID('DRWN') then
      DrawName:=ReadBoolean
    else if ChID=ChunkID('AUHI') then
      AutoHighlight:=ReadBoolean
    else if ChID=ChunkID('INTR') then
      Interactive:=ReadBoolean
    else if ChID=ChunkID('IMAG') then begin
      FreeAndNil(FImage);
      Image:=TBitmap.Create;
      ReadImageFromChunkIO(ChunkIO, Image);
    end else Result:=inherited ReadPropertyFromChunkIO(ChID, ChunkIO);
  end;
end;

procedure TButtonPart.SetScriptProperty(AName, AValue: string);
begin
  if AName='style' then begin
    if AValue='invisible' then Style:=bpsInvisible
    else if AValue='opaque' then Style:=bpsOpaque
    else if AValue='rectangle' then Style:=bpsRectangle
    else if AValue='roundrect' then Style:=bpsRoundRect
    else if AValue='shadowed' then Style:=bpsShadowed
    else LILRuntime.SetError('"' + AValue + '" is an invalid style for button');
  end else if AName='draw-name' then DrawName:=not ((AValue='') or (AValue='0') or (AValue='no'))
  else if AName='highlight' then Highlight:=not ((AValue='') or (AValue='0') or (AValue='no'))
  else if AName='auto-highlight' then AutoHighlight:=not ((AValue='') or (AValue='0') or (AValue='no'))
  else inherited SetScriptProperty(AName, AValue);
end;

function TButtonPart.GetScriptProperty(AName: string): string;
begin
  if AName='style' then begin
    Result:='';
    case Style of
      bpsInvisible: Result:='invisible';
      bpsOpaque: Result:='opaque';
      bpsRectangle: Result:='rectangle';
      bpsRoundRect: Result:='roundrect';
      bpsShadowed: Result:='shadowed';
    end;
  end else if AName='draw-name' then Result:=IntToStr(Integer(DrawName))
  else if AName='highlight' then Result:=IntToStr(Integer(Highlight))
  else if AName='auto-highlight' then Result:=IntToStr(Integer(AutoHighlight))
  else Result:=inherited GetScriptProperty(AName);
end;

function TButtonPart.GetChunkID: Cardinal;
begin
  Result:=ChunkID(ButtonPartChunkID);
end;

{ TPart }

procedure TPart.SetX1(const AValue: Integer);
begin
  if FX1=AValue then Exit;
  FX1:=AValue;
  Invalidate;
  SizeChanged;
end;

procedure TPart.SetCard(const AValue: TCard);
begin
  if FCard=AValue then Exit;
  if Assigned(AValue) then
    AValue.Add(Self)
  else
    FCard:=nil;
end;

function TPart.GetUnderMouse: Boolean;
begin
  if Assigned(Card) then begin
    if Assigned(Card.Foreground) then
      Result:=Card.Foreground.PartUnderMouse=Self
    else
      Result:=Card.PartUnderMouse=Self;
  end else Result:=False;
end;

procedure TPart.SetActive(const AValue: Boolean);
begin
  if AValue then Activate else Deactivate;
end;

function TPart.GetLeftButton: Boolean;
begin
  if Assigned(Card) then begin
    if Assigned(Card.Foreground) then
      Result:=Card.Foreground.LeftButton
    else
      Result:=Card.LeftButton;
  end;
end;

function TPart.GetActive: Boolean;
begin
  if Assigned(Card) then
    Result:=Card.ActivePart=Self
  else
    Result:=False;
end;

function TPart.GetAllowOffScreen: Boolean;
begin
  Result:=False;
end;

function TPart.GetCursor: TCursor;
begin
  Result:=crDefault;
end;

function TPart.GetAllowInteraction: Boolean;
begin
  Result:=UIMode=uiBrowse;
end;

function TPart.GetAllowEditing: Boolean;
begin
  Result:=True;
end;

function TPart.GetMiddleButton: Boolean;
begin
  if Assigned(Card) then begin
    if Assigned(Card.Foreground) then
      Result:=Card.Foreground.MiddleButton
    else
      Result:=Card.MiddleButton;
  end;
end;

function TPart.GetMouseX: Integer;
begin
  if Assigned(Card) then begin
    if Assigned(Card.Foreground) then
      Result:=Card.Foreground.MouseX
    else
      Result:=Card.MouseX;
  end else Result:=0;
end;

function TPart.GetMouseY: Integer;
begin
  if Assigned(Card) then begin
    if Assigned(Card.Foreground) then
      Result:=Card.Foreground.MouseY
    else
      Result:=Card.MouseY;
  end else Result:=0;
end;

function TPart.GetPartEditor: TWinControl;
begin
  Result:=nil;
end;

function TPart.GetRightButton: Boolean;
begin
  if Assigned(Card) then begin
    if Assigned(Card.Foreground) then
      Result:=Card.Foreground.RightButton
    else
      Result:=Card.RightButton;
  end;
end;

function TPart.GetUIMode: TUIMode;
begin
  if Assigned(Card) then Result:=Card.UIMode else Result:=uiBrowse;
end;

procedure TPart.SetColor(const AValue: TColor);
begin
  if FColor=AValue then Exit;
  FColor:=AValue;
  Invalidate;
end;

procedure TPart.SetFont(const AValue: TCardFont);
begin
  if CompareMem(@AValue, @FFont, SizeOf(TCardFont)) then Exit;
  FFont:=AValue;
  Invalidate;
end;

procedure TPart.SetLazFont(const AValue: TFont);
var
  NewFont: TCardFont;
begin
  NewFont.Assign(AValue);
  Font:=NewFont;
end;

procedure TPart.SetName(const AValue: string);
begin
  if FName=AValue then Exit;
  FName:=AValue;
  Invalidate;
end;

procedure TPart.SetVisible(AValue: Boolean);
begin
  if FVisible=AValue then Exit;
  FVisible:=AValue;
  if Assigned(Card) then Card.PartVisibilityChanged(Self);
  Invalidate;
  VisibilityChanged;
end;

procedure TPart.SetSharedData(const AValue: Boolean);
begin
  FSharedData:=AValue;
end;

procedure TPart.SetX2(const AValue: Integer);
begin
  if FX2=AValue then Exit;
  FX2:=AValue;
  Invalidate;
  SizeChanged;
end;

procedure TPart.SetY1(const AValue: Integer);
begin
  if FY1=AValue then Exit;
  FY1:=AValue;
  Invalidate;
  SizeChanged;
end;

procedure TPart.SetY2(const AValue: Integer);
begin
  if FY2=AValue then Exit;
  FY2:=AValue;
  Invalidate;
  SizeChanged;
end;

procedure TPart.CardChanged;
begin
end;

procedure TPart.SizeChanged;
begin
end;

procedure TPart.UIModeChanged;
begin
  if UIMode=uiBrowse then CallScript(EV_BROWSE, False);
end;

procedure TPart.HandleCardEnter;
begin
  CallScript(EV_CARD_VISIT, False);
end;

procedure TPart.HandleCardExit;
begin
  CallScript(EV_CARD_LEAVE, False);
end;

procedure TPart.TimerTick;
begin
  CallScript(EV_TIMER, False);
end;

procedure TPart.FillDuplicate(ATargetPart: TPart);
begin
  ATargetPart.Name:=Name;
  ATargetPart.Color:=Color;
  ATargetPart.Font:=Font;
  ATargetPart.Place(X1, Y1, X2, Y2);
  ATargetPart.SharedData:=SharedData;
  ATargetPart.Visible:=Visible;
  ATargetPart.ScriptHandlerSet.Assign(ScriptHandlerSet);
  ATargetPart.FScriptPropertySet.Assign(FScriptPropertySet);
end;

procedure TPart.WritePropertiesToChunkIO(ChunkIO: TChunkIO);
begin
  with ChunkIO do begin
    WriteCardinal(ChunkID('IDID'), ID);
    WriteString(ChunkID('NAME'), Name);
    WriteCardinal(ChunkID('COLO'), Cardinal(Color));
    WriteInteger(ChunkID('ATX1'), X1);
    WriteInteger(ChunkID('ATY1'), Y1);
    WriteInteger(ChunkID('ATX2'), X2);
    WriteInteger(ChunkID('ATY2'), Y2);
    BeginChunk(ChunkID('FONT'));
    Font.WriteToChunkIO(ChunkIO);
    EndChunk();
    FScriptHandlerSet.WriteToChunkIO(ChunkIO);
    FScriptPropertySet.WriteToChunkIO(ChunkIO);
  end;
end;

function TPart.ReadPropertyFromChunkIO(ChID: Cardinal; ChunkIO: TChunkIO): Boolean;
begin
  Result:=True;
  with ChunkIO do begin
    if ChID=ChunkID('IDID') then
      FID:=ReadCardinal
    else if ChID=ChunkID('NAME') then
      Name:=ReadString
    else if ChID=ChunkID('COLO') then
      Color:=TColor(ReadCardinal)
    else if ChID=ChunkID('ATX1') then
      X1:=ReadInteger
    else if ChID=ChunkID('ATY1') then
      Y1:=ReadInteger
    else if ChID=ChunkID('ATX2') then
      X2:=ReadInteger
    else if ChID=ChunkID('ATY2') then
      Y2:=ReadInteger
    else if ChID=ChunkID('FONT') then
      Font.ReadFromChunkIO(ChunkIO)
    else if ChID=ChunkID(ScriptHandlerSetChunkID) then
      FScriptHandlerSet.ReadFromChunkIO(ChunkIO)
    else if ChID=ChunkID(ScriptPropertySetChunkID) then
      FScriptPropertySet.ReadFromChunkIO(ChunkIO)
    else Result:=False;
  end;
  if Assigned(FillNextID) and (FillNextID^ <= ID) then FillNextID^:=ID;
end;

procedure TPart.BeforeWriteToDisk;
begin
  CallScript(EV_STACK_SAVE, False);
end;

procedure TPart.AfterReadFromDisk;
begin
  CallScript(EV_STACK_LOAD, False);
end;

procedure TPart.ControlParentChanged;
begin
end;

procedure TPart.VisibilityChanged;
begin
end;

procedure TPart.Activated;
begin
end;

procedure TPart.Deactivated;
begin
end;

procedure TPart.PrepareCanvas(Canvas: TCanvas);
begin
  Font.Setup(Canvas.Font);
end;

procedure TPart.PaintInCanvas(Canvas: TCanvas);
begin
end;

procedure TPart.PaintEditingOverlay(Canvas: TCanvas);
begin
  Canvas.Brush.Style:=bsClear;
  Canvas.Pen.Style:=psSolid;
  Canvas.Pen.Width:=1;
  if Active then Canvas.Pen.Color:=clFuchsia else Canvas.Pen.Color:=clBlue;
  Canvas.Rectangle(X1 - 1, Y1 - 1, X2 + 1, Y2 + 1);
end;

procedure TPart.Reset;
begin
  Name:='';
  Visible:=True;
  Color:=clWhite;
  AutoLockMouse:=True;
  ScriptHandlerSet.RemoveAll;
  Font.Assign(Screen.SystemFont);
end;

procedure TPart.LockMouse;
begin
  if Assigned(Card) then
    if Assigned(Card.Foreground) then
      Card.Foreground.LockMouse(Self)
    else
      Card.LockMouse(Self);
end;

procedure TPart.UnlockMouse;
begin
  if Assigned(Card) then
    if Assigned(Card.Foreground) then
      Card.Foreground.UnlockMouse(Self)
    else
      Card.UnlockMouse(Self);
end;

procedure TPart.Activate;
begin
  if Assigned(Card) then Card.ActivePart:=Self;
end;

procedure TPart.Deactivate;
begin
  if Assigned(Card) then Card.ActivePart:=nil;
end;

function TPart.HandleMouseDown(X, Y: Integer; Shift: TShiftState;
  Button: TMouseButton): Boolean;
begin
  Activate;
  if AutoLockMouse and Assigned(Card) and (Button=mbLeft) then LockMouse;
  Result:=CallScript(EV_BUTTON_PUSH);
end;

function TPart.HandleMouseMotion(X, Y: Integer; Shift: TShiftState): Boolean;
begin
  Result:=CallScript(EV_POINTER_MOTION);
end;

function TPart.HandleMouseUp(X, Y: Integer; Shift: TShiftState;
  Button: TMouseButton): Boolean;
begin
  if AutoLockMouse and Assigned(Card) and (Button=mbLeft) then UnlockMouse;
  Result:=CallScript(EV_BUTTON_RELEASE);
end;

function TPart.HandleMouseEnter(X, Y: Integer): Boolean;
begin
  Result:=CallScript(EV_POINTER_HOVER);
end;

function TPart.HandleMouseExit(X, Y: Integer): Boolean;
begin
  Result:=CallScript(EV_POINTER_LEAVE);
end;

function TPart.HandleDoubleClick(X, Y: Integer): Boolean;
begin
  Result:=CallScript(EV_DOUBLE_CLICK);
end;

destructor TPart.Destroy;
begin
  if Assigned(Card) then Card.Remove(Self);
  FreeAndNil(FScriptHandlerSet);
  inherited Destroy;
end;

procedure TPart.Construct;
begin
  FScriptHandlerSet:=TScriptHandlerSet.Create;
  Reset;
end;

procedure TPart.Place(AX1, AY1, AX2, AY2: Integer);
begin
  if (FX1=AX1) and (FY1=AY1) and (FX2=AX2) and (FY2=AY2) then Exit;
  FX1:=AX1;
  FY1:=AY1;
  FX2:=AX2;
  FY2:=AY2;
  Invalidate;
  SizeChanged;
end;

function TPart.IsPointInside(X, Y: Integer): Boolean;
begin
  Result:=(X >= X1) and (Y >= Y1) and (X <= X2) and (Y <= Y2);
end;

function TPart.PartAreaAt(X, Y: Integer): TPartArea;
begin
  if not IsPointInside(X, Y) then Exit(paOutside);
  if (X <= X1 + 3) and (Y <= Y1 + 3) then Exit(paLeftTop);
  if (X >= X2 - 3) and (Y <= Y1 + 3) then Exit(paRightTop);
  if (X <= X1 + 3) and (Y >= Y2 - 3) then Exit(paLeftBottom);
  if (X >= X2 - 3) and (Y >= Y2 - 3) then Exit(paRightBottom);
  if X <= X1 + 3 then Exit(paLeft);
  if X >= X2 - 3 then Exit(paRight);
  if Y <= Y1 + 3 then Exit(paTop);
  if Y >= Y2 - 3 then Exit(paBottom);
  Result:=paCenter;
end;

function TPart.MousePartArea: TPartArea;
begin
  Result:=PartAreaAt(MouseX, MouseY);
end;

procedure TPart.RaiseToTop;
begin
  if Assigned(Card) then Card.RaiseToTop(Self);
end;

procedure TPart.SinkToBottom;
begin
  if Assigned(Card) then Card.SinkToBottom(Self);
end;

function TPart.VisibleInEditMode: Boolean;
begin
  Result:=Visible or (Assigned(Card) and Card.PartEditing);
end;

function TPart.GetSubObjectByID(AID: Cardinal): TObject;
begin
  if AID=ID then Exit(Self);
  Result:=nil;
end;

function TPart.GetObjectByID(AID: Cardinal): TObject;
begin
  if Assigned(Card) then Result:=Card.GetObjectByID(AID) else Result:=GetSubObjectByID(AID);
end;

function TPart.GetSubObjectByName(AName: string): TObject;
begin
  if AName=Name then Exit(Self);
  Result:=nil;
end;

function TPart.GetObjectByName(AName: string): TObject;
begin
  if Assigned(Card) then Result:=Card.GetObjectByName(AName) else Result:=GetSubObjectByName(AName);
end;

procedure TPart.SetScriptProperty(AName, AValue: string);
var
  i: Integer;
begin
  // too much copypaste here...
  if AName='id' then LILRuntime.SetError('Cannot change the ID of a part')
  else if AName='name' then Name:=AValue
  else if AName='shared' then SharedData:=not ((AValue='') or (AValue='0') or (AValue='no'))
  else if AName='hovered' then LILRuntime.SetError('Cannot change the hovered property of a part')
  else if AName='active' then Active:=not ((AValue='') or (AValue='0') or (AValue='no'))
  else if AName='visibility' then begin
    if AValue='visible' then Visible:=True
    else if AValue='hidden' then Visible:=False
    else LILRuntime.SetError('Invalid value given for the "visibility" property - it must be either visible or hidden');
  end else if AName='left' then begin
    i:=X2 - X1;
    try
      X1:=StrToInt(AValue);
    except
      LILRuntime.SetError('Invalid number given for the "left" property');
      Exit;
    end;
    X2:=X1 + i;
  end else if AName='top' then begin
    i:=Y2 - Y1;
    try
      Y1:=StrToInt(AValue);
    except
      LILRuntime.SetError('Invalid number given for the "top" property');
      Exit;
    end;
    Y2:=Y1 + i;
  end else if AName='width' then begin
    try
      i:=StrToInt(AValue);
    except
      LILRuntime.SetError('Invalid number given for the "width" property');
      Exit;
    end;
    if i < 1 then begin
      LILRuntime.SetError('A value less than one given for the "width" property');
      Exit;
    end;
    X2:=X1 + i;
  end else if AName='height' then begin
    try
      i:=StrToInt(AValue);
    except
      LILRuntime.SetError('Invalid number given for the "height" property');
      Exit;
    end;
    if i < 1 then begin
      LILRuntime.SetError('A value less than one given for the "height" property');
      Exit;
    end;
    Y2:=Y1 + i;
  end else FScriptPropertySet.Put(AName, AValue);
end;

function TPart.GetScriptProperty(AName: string): string;
begin
  if AName='id' then Result:=IntToStr(ID)
  else if AName='name' then Result:=Name
  else if AName='shared' then Result:=IntToStr(Integer(SharedData))
  else if AName='hovered' then Result:=IntToStr(Integer(UnderMouse))
  else if AName='active' then Result:=IntToStr(Integer(Active))
  else if AName='left' then Result:=IntToStr(X1)
  else if AName='top' then Result:=IntToStr(Y1)
  else if AName='width' then Result:=IntToStr(X2 - X1 + 1)
  else if AName='height' then Result:=IntToStr(Y2 - Y1 + 1)
  else Result:=FScriptPropertySet.Get(AName);
end;

function TPart.Duplicate: TPart;
begin
  Result:=ClassType.NewInstance as TPart;
  if Assigned(Result) then begin
    Result.Construct;
    if Assigned(Card) then Card.Add(Result);
    FillDuplicate(Result);
  end;
end;

function TPart.CallScript(AHandlerName: string; PassToCard: Boolean): Boolean;
begin
  if UIMode=uiBrowse then begin
    SetScriptVariable('me', '&' + IntToStr(ID));
    SetScriptVariable('part', '&' + IntToStr(ID));
    if Assigned(Card) then Card.SetCommonScriptVariables;
    Result:=ScriptHandlerSet.RunHandler(AHandlerName);
    if PassToCard and not Result and Assigned(Card) then Result:=Card.CallScript(AHandlerName, False);
  end else Result:=False;
end;

function TPart.GetChunkID: Cardinal;
begin
  Result:=ChunkID('PART');
end;

procedure TPart.WriteToChunkIO(ChunkIO: TChunkIO);
begin
  with ChunkIO do begin
    BeginChunk(GetChunkID);
    WritePropertiesToChunkIO(ChunkIO);
    EndChunk();
  end;
end;

procedure TPart.ReadFromChunkIO(ChunkIO: TChunkIO);
var
  ChID, ChunkSize: Cardinal;
begin
  Reset;
  FID:=0;
  with ChunkIO do begin
    while NextChunk(ChID, ChunkSize) do begin
      if not ReadPropertyFromChunkIO(ChID, ChunkIO) then SkipChunk(ChunkSize);
    end;
  end;
end;

procedure TPart.Invalidate;
begin
  if Assigned(Card) then Card.Invalidate;
end;

{ TStack }

function TStack.GetCards(AIndex: Integer): TCard;
begin
  Result:=FCards[AIndex];
end;

function TStack.GetBackgroundCount: Integer;
begin
  Result:=FBackgrounds.Count;
end;

function TStack.GetBackgrounds(AIndex: Integer): TCard;
begin
  Result:=FBackgrounds[AIndex];
end;

function TStack.GetCount: Integer;
begin
  Result:=FCards.Count;
end;

function TStack.GetFirstCard: TCard;
begin
  if Count > 0 then Result:=Cards[0] else Result:=nil;
end;

function TStack.GetLastCard: TCard;
begin
  if Count > 0 then Result:=Cards[Count - 1] else Result:=nil;
end;

procedure TStack.SetControlParent(const AValue: TWinControl);
var
  i: Integer;
begin
  if FControlParent=AValue then Exit;
  FControlParent:=AValue;
  for i:=0 to Count - 1 do
    Cards[i].ControlParent:=AValue;
  for i:=0 to BackgroundCount - 1 do
    Backgrounds[i].ControlParent:=AValue;
end;

procedure TStack.SetUIMode(const AValue: TUIMode);
var
  i: Integer;
begin
  if FUIMode=AValue then Exit;
  FUIMode:=AValue;
  for i:=0 to Count - 1 do begin
    Cards[i].FUIMode:=AValue;
    Cards[i].UIModeChanged;
  end;
  for i:=0 to BackgroundCount - 1 do begin
    Backgrounds[i].FUIMode:=AValue;
    Backgrounds[i].UIModeChanged;
  end;
  CallScript(EV_BROWSE);
end;

procedure TStack.SetupNewCard(ACard: TCard);
begin
  ACard.ControlParent:=ControlParent;
  ACard.FUIMode:=UIMode;
  ACard.UIModeChanged;
end;

procedure TStack.BeforeWriteToDisk;
var
  i: Integer;
begin
  CallScript(EV_STACK_SAVE);
  for i:=0 to BackgroundCount - 1 do Backgrounds[i].BeforeWriteToDisk;
  for i:=0 to Count - 1 do Cards[i].BeforeWriteToDisk;
end;

procedure TStack.AfterReadFromDisk;
var
  i: Integer;
begin
  CallScript(EV_STACK_LOAD);
  for i:=0 to BackgroundCount - 1 do Backgrounds[i].AfterReadFromDisk;
  for i:=0 to Count - 1 do Cards[i].AfterReadFromDisk;
end;

constructor TStack.Create;
begin
  FCards:=TCardList.Create(True);
  FBackgrounds:=TCardList.Create(True);
  FScriptHandlerSet:=TScriptHandlerSet.Create;
  FNextID:=1;
end;

destructor TStack.Destroy;
begin
  FreeAndNil(FScriptHandlerSet);
  FreeAndNil(FBackgrounds);
  FreeAndNil(FCards);
  inherited Destroy;
end;

function TStack.Add(BgCard: TCard; AtIndex: Integer): TCard;
begin
  Result:=TCard.Create;
  if BgCard=nil then BgCard:=AddBackground;
  Result.FStack:=Self;
  Result.FID:=GetNextID;
  Result.Background:=BgCard;
  SetupNewCard(Result);
  if AtIndex=-1 then
    FCards.Add(Result)
  else
    FCards.Insert(AtIndex, Result);
end;

procedure TStack.Remove(ACard: TCard);
begin
  FCards.Remove(ACard);
end;

function TStack.IndexOf(ACard: TCard): Integer;
begin
  Result:=FCards.IndexOf(ACard);
end;

function TStack.PreviousCardOf(ACard: TCard): TCard;
var
  Index: Integer;
begin
  Index:=IndexOf(ACard);
  if Index=-1 then Result:=nil
  else if Index=0 then Result:=Cards[Count - 1]
  else Result:=Cards[Index - 1];
end;

function TStack.NextCardOf(ACard: TCard): TCard;
var
  Index: Integer;
begin
  Index:=IndexOf(ACard);
  if Index=-1 then Result:=nil
  else if Index=Count - 1 then Result:=Cards[0]
  else Result:=Cards[Index + 1];
end;

function TStack.AddBackground: TCard;
begin
  Result:=TCard.Create;
  Result.FStack:=Self;
  Result.FID:=GetNextID;
  Result.AllocateImage;
  SetupNewCard(Result);
  FBackgrounds.Add(Result);
end;

procedure TStack.RemoveBackground(ACard: TCard);
begin
  FBackgrounds.Remove(ACard);
end;

function TStack.IndexOfBackground(ACard: TCard): Integer;
begin
  Result:=FBackgrounds.IndexOf(ACard);
end;

function TStack.GetNextID: Cardinal;
begin
  if not Assigned(FillNextID) then begin
    Result:=FNextID;
    Inc(FNextID);
  end else Result:=0;
end;

function TStack.GetObjectByID(AID: Cardinal): TObject;
var
  i: Integer;
begin
  Result:=nil;
  for i:=0 to Count - 1 do begin
    Result:=Cards[i].GetSubObjectByID(AID);
    if Assigned(Result) then Exit;
  end;
  for i:=0 to BackgroundCount - 1do begin
    Result:=Backgrounds[i].GetSubObjectByID(AID);
    if Assigned(Result) then Exit;
  end;
end;

function TStack.GetObjectByName(AName: string): TObject;
var
  i: Integer;
begin
  for i:=0 to Count - 1 do begin
    Result:=Cards[i].GetSubObjectByName(AName);
    if Assigned(Result) then Exit;
  end;
  Result:=nil;
end;

function TStack.GetCardByName(AName: string): TCard;
var
  i: Integer;
begin
  for i:=0 to Count - 1 do
    if Cards[i].Name=AName then Exit(Cards[i]);
  Result:=nil;
end;

function TStack.CallScript(AHandlerName: string): Boolean;
begin
  if UIMode=uiBrowse then begin
    SetScriptVariable('me', '');
    Result:=ScriptHandlerSet.RunHandler(AHandlerName);
  end else Result:=False;
end;

procedure TStack.TimerTick;
begin
  CallScript(EV_TIMER);
end;

procedure TStack.WriteToChunkIO(ChunkIO: TChunkIO);

  procedure WriteCardList(List: TCardList);
  var
    i: Integer;
  begin
    for i:=0 to List.Count - 1 do
      if List[i].FLoadedBackgroundID=0 then
        List[i].WriteToChunkIO(ChunkIO);
  end;

var
  i: Integer;
begin
  BeforeWriteToDisk;
  // this whole FLoadedBackgroundID marking thing is used to make sure that
  // only used backgrounds are saved on disk
  for i:=0 to Count - 1 do Cards[i].FLoadedBackgroundID:=0;
  for i:=0 to BackgroundCount - 1 do Backgrounds[i].FLoadedBackgroundID:=MAXDWORD;
  for i:=0 to Count - 1 do if Assigned(Cards[i].Background) then Cards[i].Background.FLoadedBackgroundID:=0;
  with ChunkIO do begin
    BeginChunk(ChunkID(StackChunkID));
    BeginChunk(ChunkID('CRDS'));
    WriteCardList(FCards);
    EndChunk();
    BeginChunk(ChunkID('BGDS'));
    WriteCardList(FBackgrounds);
    EndChunk();
    FScriptHandlerSet.WriteToChunkIO(ChunkIO);
    EndChunk();
  end;
end;

procedure TStack.ReadFromChunkIO(ChunkIO: TChunkIO);
var
  ChID, ChunkSize: Cardinal;

  procedure ReadCardList(List: TCardList);
  begin
    with ChunkIO do begin
      while NextChunk(ChID, ChunkSize) do begin
        if ChID=ChunkID(CardChunkID) then begin
          if List=FCards then
            Add(nil).ReadFromChunkIO(ChunkIO)
          else
            AddBackground.ReadFromChunkIO(ChunkIO);
        end else SkipChunk(ChunkSize);
      end;
    end;
  end;

  procedure FillBackgrounds;
  var
    i: Integer;
  begin
    for i:=0 to Count - 1 do
      Cards[i].Background:=GetObjectByID(Cards[i].FLoadedBackgroundID) as TCard;
  end;

begin
  FCards.Clear;
  FBackgrounds.Clear;
  FNextID:=1;
  FillNextID:=@FNextID;
  FScriptHandlerSet.RemoveAll;
  with ChunkIO do begin
    while NextChunk(ChID, ChunkSize) do begin
      if ChID=ChunkID('CRDS') then
        ReadCardList(FCards)
      else if ChID=ChunkID('BGDS') then
        ReadCardList(FBackgrounds)
      else if ChID=ChunkID(ScriptHandlerSetChunkID) then
        FScriptHandlerSet.ReadFromChunkIO(ChunkIO)
      else SkipChunk(ChunkSize);
    end;
  end;
  FillNextID:=nil;
  FillBackgrounds;
  AfterReadFromDisk;
  // force a reset of UI mode
  FUIMode:=uiPen;
  SetUIMode(uiBrowse);
end;

procedure TStack.WriteToStream(AStream: TStream);
var
  ChunkIO: TChunkIO;
begin
  try
    ChunkIO:=TChunkIO.Create;
    ChunkIO.Stream:=AStream;
    WriteToChunkIO(ChunkIO);
  finally
    FreeAndNil(ChunkIO);
  end;
end;

procedure TStack.ReadFromStream(AStream: TStream);
var
  ChunkIO: TChunkIO;
  ID, Size: Cardinal;
begin
  try
    ChunkIO:=TChunkIO.Create;
    ChunkIO.Stream:=AStream;
    while ChunkIO.NextChunk(ID, Size) do
      if ID=ChunkID(StackChunkID) then begin
        ReadFromChunkIO(ChunkIO);
        Break;
      end else ChunkIO.SkipChunk(Size);
  finally
    FreeAndNil(ChunkIO);
  end;
end;

procedure TStack.WriteToFile(AFileName: string);
var
  FileStream: TFileStream;
  CompressionStream: TCompressionStream;
  MemoryStream: TMemoryStream;
  I: Integer;
  Command: String;
begin
  try
    MemoryStream:=TMemoryStream.Create;
    WriteToStream(MemoryStream);
    FileStream:=TFileStream.Create(AFileName, fmCreate);
    Command:='#!/usr/bin/env lilcard'#10;
    for I:=1 to Length(Command) do FileStream.WriteByte(Ord(Command[I]));
    CompressionStream:=TCompressionStream.Create(clMax, FileStream, False);
    CompressionStream.Write(MemoryStream.Memory^, MemoryStream.Size);
  finally
    FreeAndNil(MemoryStream);
    FreeAndNil(CompressionStream);
    FreeAndNil(FileStream);
  end;
end;

procedure TStack.ReadFromFile(AFileName: string);
var
  FileStream: TFileStream;
  MemoryStream: TMemoryStream;
  DecompressionStream: TDecompressionStream;
  Buffer: array [0..65535] of Byte;
  DataRead: Integer;
begin
  try
    FileStream:=TFileStream.Create(AFileName, fmOpenRead);
    // Optional unix-like header that can be used to make stacks executable
    if FileStream.ReadByte=Ord('#') then begin
      while FileStream.ReadByte <> 10 do;
      while FileStream.ReadByte=10 do;
    end;
    FileStream.Position:=FileStream.Position - 1;
    DecompressionStream:=TDecompressionStream.Create(FileStream, False);
    MemoryStream:=TMemoryStream.Create;
    while True do begin
      DataRead:=DecompressionStream.Read(Buffer, SizeOf(Buffer));
      MemoryStream.Write(Buffer, DataRead);
      if DataRead < SizeOf(Buffer) then Break;
    end;
    MemoryStream.Position:=0;
    ReadFromStream(MemoryStream);
  finally
    FreeAndNil(MemoryStream);
    FreeAndNil(DecompressionStream);
    FreeAndNil(FileStream);
  end;
end;

{ TCard }

procedure TCard.PaintInCanvas(Canvas: TCanvas);
var
  i: Integer;
begin
  if Assigned(Image) then Canvas.Draw(0, 0, Image);
  if Assigned(Background) then begin
    Background.FForeground:=Self;
    Background.PaintInCanvas(Canvas);
  end;
  for i:=0 to Count - 1 do if Parts[I].VisibleInEditMode then begin
    Parts[i].PrepareCanvas(Canvas);
    Parts[i].PaintInCanvas(Canvas);
  end;
  if PartEditing then
    for i:=0 to Count - 1 do
      if Parts[I].VisibleInEditMode and Parts[i].AllowEditing then begin
        Parts[i].PaintEditingOverlay(Canvas);
        if not Parts[I].Visible then with Canvas do begin
          Font.Color:=clRed;
          if Parts[I] is TControlPart then
            TextOut(Parts[I].X1, Parts[I].Y2, '(invisible)')
          else
            TextOut(Parts[I].X1, Parts[I].Y2 - TextHeight('W'), '(invisible)');
        end;
      end;
end;

procedure TCard.Invalidate;
begin
  FNeedsUpdate:=True;
end;

procedure TCard.SetControlParent(const AValue: TWinControl);
var
  i: Integer;
begin
  if FControlParent=AValue then Exit;
  FControlParent:=AValue;
  for i:=0 to Count - 1 do Parts[i].ControlParentChanged;
end;

procedure TCard.UpdateMouseProperties(X, Y: Integer; Button: TMouseButton;
  ButtonMotion: Integer);
begin
  FMouseX:=X;
  FMouseY:=Y;
  if ButtonMotion <> 0 then begin
    case Button of
      mbLeft: FLeftButton:=ButtonMotion > 0;
      mbMiddle: FMiddleButton:=ButtonMotion > 0;
      mbRight: FRightButton:=ButtonMotion > 0;
    end;
  end;
end;

procedure TCard.UIModeChanged;
var
  i: Integer;
begin
  for i:=0 to Count - 1 do Parts[i].UIModeChanged;
  CallScript(EV_BROWSE, True, False);
end;

procedure TCard.HandleCardEnter;
var
  i: Integer;
begin
  FEntered:=True;
  for i:=0 to Count - 1 do Parts[i].HandleCardEnter;
  CallScript(EV_CARD_VISIT, True, False);
  if Assigned(Stack) then Stack.CallScript(EV_CARD_VISIT);
end;

procedure TCard.HandleCardExit;
var
  i: Integer;
begin
  FEntered:=False;
  ActivePart:=nil;
  if Assigned(Stack) then Stack.CallScript(EV_CARD_LEAVE);
  CallScript(EV_CARD_LEAVE, True, False);
  for i:=0 to Count - 1 do Parts[i].HandleCardExit;
end;

procedure TCard.TimerTick;
var
  i: Integer;
begin
  CallScript(EV_TIMER, True, False);
  for i:=0 to Count - 1 do Parts[i].TimerTick;
end;

function TCard.PreparePainter: TObject;
var
  Painter: TPainter;
begin
  if not Assigned(Background) and not Assigned(Image) then Exit(nil);
  Painter:=GetPainterForUIMode(UIMode);
  if not Assigned(Painter) then Exit(nil);
  Painter.Card:=Self;
  if Assigned(Background) then
    Painter.Canvas:=Background.Image.Canvas
  else
    Painter.Canvas:=Image.Canvas;
  PaintSettings.SetupCanvas(Painter.Canvas);
  Result:=Painter;
end;

procedure TCard.AllocateImage;
begin
  FreeAndNil(FImage);
  FImage:=TBitmap.Create;
//  FImage.PixelFormat:=pf32bit;
  FImage.SetSize(640, 480);
  FImage.Canvas.Brush.Color:=clWhite;
  FImage.Canvas.Pen.Style:=psClear;
  FImage.Canvas.Rectangle(-1, -1, 641, 481);
end;

function TCard.GetDataIndex(AID: Cardinal): Integer;
var
  i: Integer;
begin
  for i:=0 to Length(FData) - 1 do
    if FData[i].PartID=AID then Exit(i);
  Result:=-1;
end;

procedure TCard.SetCommonScriptVariables;
begin
  if Assigned(Foreground) then begin
    SetScriptVariable('card', '&' + IntToStr(Foreground.ID));
    SetScriptVariable('bgcard', '&' + IntToStr(ID));
  end else begin
    SetScriptVariable('card', '&' + IntToStr(ID));
    if Assigned(Background) then
      SetScriptVariable('bgcard', '&' + IntToStr(Background.ID))
    else
      SetScriptVariable('bgcard', '');
  end;
  SetScriptVariable('x', MouseX);
  SetScriptVariable('y', MouseY);
  SetScriptVariable('button', LeftButton);
  SetScriptVariable('left-button', LeftButton);
  SetScriptVariable('middle-button', MiddleButton);
  SetScriptVariable('right-button', RightButton);
  if Assigned(ActivePart) then
    SetScriptVariable('active-part', '&' + IntToStr(ActivePart.ID))
  else
    SetScriptVariable('active-part', '');
  if Assigned(PartUnderMouse) then
    SetScriptVariable('pointed-part', '&' + IntToStr(PartUnderMouse.ID))
  else
    SetScriptVariable('pointed-part', '');
end;

procedure TCard.BeforeWriteToDisk;
var
  i: Integer;
begin
  CallScript(EV_STACK_SAVE, True, False);
  for i:=0 to Count - 1 do Parts[i].BeforeWriteToDisk;
end;

procedure TCard.AfterReadFromDisk;
var
  i: Integer;
begin
  CallScript(EV_STACK_LOAD, True, False);
  for i:=0 to Count - 1 do Parts[i].AfterReadFromDisk;
end;

constructor TCard.Create;
begin
  FParts:=TPartList.Create(False);
  GridSize:=8;
  FScriptHandlerSet:=TScriptHandlerSet.Create;
  Invalidate;
end;

destructor TCard.Destroy;
var
  i: Integer;
begin
  FDestroying:=True;
  if Assigned(Background) and (Background.FForeground=Self) then Background.FForeground:=nil;
  ReleaseAllData;
  for i:=0 to Count - 1 do Parts[i].Free;
  FreeAndNil(FParts);
  FreeAndNil(FScriptHandlerSet);
  FreeAndNil(FImage);
  inherited Destroy;
end;

procedure TCard.PaintIn(TargetCanvas: TCanvas);
var
  TS: TTextStyle;

  procedure PaintGridsnappedCursor;
  var
    X, Y: Integer;
  begin
    with TargetCanvas do begin
      X:=GridSnap(MouseX);
      Y:=GridSnap(MouseY);
      Pen.Style:=psSolid;
      Pen.Width:=1;
      Pen.Color:=clWhite;
      Line(X - 6, Y, X + 7, Y);
      Line(X, Y - 6, X, Y + 7);
      Pen.Color:=clBlack;
      Line(X - 4, Y, X + 5, Y);
      Line(X, Y - 4, X, Y + 5);
    end;
  end;

  procedure PaintMakingOutline;
  begin
    with TargetCanvas do begin
      Pen.Style:=psSolid;
      Pen.Width:=1;
      Pen.Color:=clWhite;
      Brush.Style:=bsClear;
      Rectangle(FPrevX, FPrevY, GridSnap(MouseX), GridSnap(MouseY));
      Pen.Color:=clBlack;
      Rectangle(FPrevX + 1, FPrevY + 1, GridSnap(MouseX) - 1, GridSnap(MouseY) - 1);
      Pen.Style:=psSolid;
    end;
  end;

begin
  TS:=TargetCanvas.TextStyle;
  TS.Opaque:=False;
  TargetCanvas.TextStyle:=TS;
  PaintInCanvas(TargetCanvas);
  if FMakingPart then PaintMakingOutline;
  if UseGrid and Painting then PaintGridsnappedCursor;
  FNeedsUpdate:=False;
end;

procedure TCard.Add(APart: TPart);
begin
  if APart.Card=Self then Exit;
  if Assigned(APart.Card) then APart.Card.Remove(APart);
  APart.FCard:=Self;
  if (APart.FID=0) or (Assigned(GetObjectByID(APart.FID))) then
    APart.FID:=GetNextID;
  FParts.Add(APart);
  APart.Invalidate;
  APart.CardChanged;
end;

procedure TCard.Remove(APart: TPart);
begin
  if Destroying or (APart.Card <> Self) then Exit;
  if Assigned(Background) then Background.ReleaseDataForPart(APart) else ReleaseDataForPart(APart);
  APart.FCard:=nil;
  FParts.Remove(APart);
  InvalidatePartReference(APart);
  APart.CardChanged;
end;

function TCard.HandleMouseDown(X, Y: Integer; Shift: TShiftState;
  Button: TMouseButton): Boolean;
var
  TargetPart: TPart;
  Painter: TPainter;
begin
  Result:=False;
  UpdateMouseProperties(X, Y, Button, 1);
  if Assigned(FMovingPart) then Exit(True);
  if PartEditing and (Button=mbLeft) and (Shift=[ssLeft, ssAlt]) then
    TargetPart:=nil
  else
    if Assigned(MouseLockedPart) then
      TargetPart:=MouseLockedPart
    else
      TargetPart:=PartAt(X, Y);
  if Assigned(TargetPart) then begin
    if TargetPart.AllowInteraction then Result:=TargetPart.HandleMouseDown(X, Y, Shift, Button);
    if not Result and (Button=mbLeft) and (Shift=[ssLeft]) and PartEditing and TargetPart.AllowEditing then begin
      ActivePart:=TargetPart;
      FMovingPart:=ActivePart;
      FMovingPartArea:=ActivePart.PartAreaAt(X, Y);
      FMovingPrevRect:=Rect(ActivePart.X1, ActivePart.Y1, ActivePart.X2, ActivePart.Y2);
      FPrevX:=X;
      FPrevY:=Y;
      Result:=True;
    end;
  end else begin
    ActivePart:=nil;
    if PartEditing then begin
      if (Button=mbLeft) and ((Shift=[ssLeft]) or (Shift=[ssLeft, ssAlt])) then begin
        FMakingPart:=True;
        FPrevX:=GridSnap(X);
        FPrevY:=GridSnap(Y);
        FAltMaking:=Shift=[ssLeft, ssAlt];
        Result:=True;
      end;
    end;
  end;
  if not Result and Painting then begin
    Painter:=TPainter(PreparePainter);
    if Assigned(Painter) then Painter.MouseDown(X, Y, Button, Shift);
  end;
  if not Result then Result:=CallScript(EV_BUTTON_PUSH);
end;

function TCard.HandleMouseMotion(X, Y: Integer; Shift: TShiftState): Boolean;
var
  TargetPart, UnderPart: TPart;
  Painter: TPainter;
  W, H: Integer;
begin
  Result:=False;
  UpdateMouseProperties(X, Y, mbLeft, 0);
  if Assigned(FMovingPart) then begin
    if FMovingPartArea=paCenter then begin
      W:=FMovingPart.X2 - FMovingPart.X1;
      H:=FMovingPart.Y2 - FMovingPart.Y1;
    end;
    if FMovingPartArea in [paCenter, paLeft, paLeftTop, paLeftBottom] then FMovingPart.X1:=GridSnap(FMovingPrevRect.Left + X - FPrevX);
    if FMovingPartArea in [paCenter, paTop, paLeftTop, paRightTop] then FMovingPart.Y1:=GridSnap(FMovingPrevRect.Top + Y - FPrevY);
    if FMovingPartArea in [paRight, paRightTop, paRightBottom] then FMovingPart.X2:=GridSnap(FMovingPrevRect.Right + X - FPrevX);
    if FMovingPartArea in [paBottom, paLeftBottom, paRightBottom] then FMovingPart.Y2:=GridSnap(FMovingPrevRect.Bottom + Y - FPrevY);
    if FMovingPart.X2 < FMovingPart.X1 + 1 then
      if FMovingPartArea in [paLeft, paLeftTop, paLeftBottom] then
        FMovingPart.X1:=FMovingPart.X2 - 1
      else
        FMovingPart.X2:=FMovingPart.X1 + 1;
    if FMovingPart.Y2 < FMovingPart.Y1 + 1 then
      if FMovingPartArea in [paTop, paLeftTop, paRightTop] then
        FMovingPart.Y1:=FMovingPart.Y2 - 1
      else
        FMovingPart.Y2:=FMovingPart.Y1 + 1;
    if FMovingPartArea=paCenter then begin
      FMovingPart.X2:=FMovingPart.X1 + W;
      FMovingPart.Y2:=FMovingPart.Y1 + H;
    end;
    if not FMovingPart.AllowOffScreen then begin
      if FMovingPart.X1 < 0 then begin
        if FMovingPartArea=paCenter then FMovingPart.X2:=FMovingPart.X2 - FMovingPart.X1;
        FMovingPart.X1:=0;
      end;
      if FMovingPart.Y1 < 0 then begin
        if FMovingPartArea=paCenter then FMovingPart.Y2:=FMovingPart.Y2 - FMovingPart.Y1;
        FMovingPart.Y1:=0;
      end;
      if FMovingPart.X2 > 639 then begin
        if FMovingPartArea=paCenter then FMovingPart.X1:=FMovingPart.X1 + (639 - FMovingPart.X2);
        FMovingPart.X2:=639;
      end;
      if FMovingPart.Y2 > 479 then begin
        if FMovingPartArea=paCenter then FMovingPart.Y1:=FMovingPart.Y1 + (479 - FMovingPart.Y2);
        FMovingPart.Y2:=479;
      end;
    end;
    Exit;
  end;
  if FMakingPart then begin
    Invalidate;
    Exit;
  end;
  UnderPart:=PartAt(X, Y);
  if Assigned(MouseLockedPart) then TargetPart:=MouseLockedPart else TargetPart:=UnderPart;
  if UnderPart <> FPartUnderMouse then begin
    if Assigned(FPartUnderMouse) and FPartUnderMouse.AllowInteraction then FPartUnderMouse.HandleMouseExit(X, Y);
    if Assigned(Background) and Assigned(Background.FPartUnderMouse) and Background.FPartUnderMouse.AllowInteraction then Background.FPartUnderMouse.HandleMouseExit(X, Y);
    FPartUnderMouse:=UnderPart;
    if Assigned(FPartUnderMouse) and (TargetPart=UnderPart) and FPartUnderMouse.AllowInteraction then FPartUnderMouse.HandleMouseEnter(X, Y);
  end;
  if Assigned(TargetPart) and TargetPart.AllowInteraction then Result:=TargetPart.HandleMouseMotion(X, Y, Shift);
  if not Result and Painting then begin
    Painter:=TPainter(PreparePainter);
    if Assigned(Painter) then Painter.MouseMotion(X, Y, Shift);
  end;
  if UseGrid and (UIMode <> uiBrowse) then Invalidate;
  if not Result then Result:=CallScript(EV_POINTER_MOTION);
end;

function TCard.HandleMouseUp(X, Y: Integer; Shift: TShiftState;
  Button: TMouseButton): Boolean;
var
  TargetPart: TPart;
  Painter: TPainter;
begin
  Result:=False;
  UpdateMouseProperties(X, Y, Button, -1);
  if Assigned(FMovingPart) then begin
    FMovingPart:=nil;
    Exit;
  end;
  if FMakingPart then begin
    FMakingPart:=False;
    MakeNewPartAt(FPrevX, FPrevY, GridSnap(X), GridSnap(Y), FAltMaking);
    Invalidate;
    Exit;
  end;
  if Assigned(MouseLockedPart) then TargetPart:=MouseLockedPart else TargetPart:=PartAt(X, Y);
  if Assigned(TargetPart) and TargetPart.AllowInteraction then Result:=TargetPart.HandleMouseUp(X, Y, Shift, Button);
  if not Result and Painting then begin
    Painter:=TPainter(PreparePainter);
    if Assigned(Painter) then Painter.MouseUp(X, Y, Button, Shift);
  end;
  if not Result then Result:=CallScript(EV_BUTTON_RELEASE);
end;

function TCard.HandleDoubleClick: Boolean;
var
  TargetPart: TPart;
begin
  Result:=False;
  if Assigned(MouseLockedPart) then TargetPart:=MouseLockedPart else TargetPart:=PartAt(MouseX, MouseY);
  if Assigned(TargetPart) then begin
    if TargetPart.AllowInteraction then Result:=TargetPart.HandleDoubleClick(MouseX, MouseY);
    if not Result and PartEditing and TargetPart.AllowEditing then PartInspector.InspectPart(TargetPart);
  end;
  if not Result then Result:=CallScript(EV_DOUBLE_CLICK);
end;

function TCard.HandleKeyDown(KeyCode: Integer; Shift: TShiftState): Boolean;
var
  i: Integer;
begin
  SetScriptVariable('key', PrettyKeyCodeName(KeyCode));
  // and why not define some handlekeydown function in TPart again..?
  for i:=0 to Count - 1 do begin
    Result:=Parts[i].CallScript(EV_KEY_PRESS, False);
    if Result then Break;
  end;
  if not Result then begin
    Result:=CallScript(EV_KEY_PRESS);
    if not Result then begin
      case KeyCode of
        VK_LEFT: SetScriptVariable('arrow', 'left');
        VK_RIGHT: SetScriptVariable('arrow', 'right');
        VK_UP: SetScriptVariable('arrow', 'up');
        VK_DOWN: SetScriptVariable('arrow', 'down');
        else Exit(False);
      end;
      Result:=CallScript(EV_ARROW);
    end;
  end;
end;

function TCard.HandleKeyUp(KeyCode: Integer; Shift: TShiftState): Boolean;
var
  i: Integer;
begin
  SetScriptVariable('key', PrettyKeyCodeName(KeyCode));
  for i:=0 to Count - 1 do begin
    Result:=Parts[i].CallScript(EV_KEY_RELEASE, False);
    if Result then Break;
  end;
  if not Result then Result:=CallScript(EV_KEY_RELEASE);
end;

function TCard.HandleCharKey(CharKey: TUTF8Char): Boolean;
begin
  SetScriptVariable('character', CharKey);
  Result:=CallScript(EV_CHARACTER);
end;

function TCard.PartAt(X, Y: Integer): TPart;
var
  i: Integer;
begin
  for i:=Count - 1 downto 0 do
    if Parts[I].VisibleInEditMode and Parts[i].IsPointInside(X, Y) then Exit(Parts[i]);
  if Assigned(Background) then
    Result:=Background.PartAt(X, Y)
  else
    Result:=nil;
end;

procedure TCard.ActivateAt(X, Y: Integer);
begin
  ActivePart:=PartAt(X, Y);
end;

procedure TCard.LockMouse(ToPart: TPart);
begin
  if Assigned(FMouseLockedPart) then UnlockMouse(FMouseLockedPart);
  FMouseLockedPart:=ToPart;
end;

procedure TCard.UnlockMouse(FromPart: TPart);
begin
  if FromPart <> FMouseLockedPart then Exit;
  FMouseLockedPart:=nil;
end;

procedure TCard.RaiseToTop(APart: TPart);
begin
  if Assigned(APart) and (APart.Card=Self) then begin
    FParts.Move(FParts.IndexOf(APart), Count - 1);
    Invalidate;
  end;
end;

procedure TCard.SinkToBottom(APart: TPart);
begin
  if Assigned(APart) and (APart.Card=Self) then begin
    FParts.Move(FParts.IndexOf(APart), 0);
    Invalidate;
  end;
end;

procedure TCard.ApplyLazFont(ALazFont: TFont);
begin
  if Assigned(ActivePart) then ActivePart.LazFont:=ALazFont;
end;

function TCard.SetupLazFont(ALazFont: TFont): Boolean;
begin
  if Assigned(ActivePart) then begin
    ActivePart.Font.Setup(ALazFont);
    Result:=True;
  end else Result:=False;
end;

procedure TCard.MakeNewPartAt(X1, Y1, X2, Y2: Integer; AltMode: Boolean; ForMode: TUIMode);
var
  APart: TPart;
  Tmp: Integer;
begin
  if X1 > X2 then begin Tmp:=X1; X1:=X2; X2:=Tmp end;
  if Y1 > Y2 then begin Tmp:=Y1; Y1:=Y2; Y2:=Tmp end;
  if X1 < 0 then X1:=0;
  if Y1 < 0 then Y1:=0;
  if X2 > 639 then X2:=639;
  if Y2 > 479 then Y2:=479;
  if (X2-X1 < 4) or (Y2-Y1 < 4) then Exit;
  if ForMode=uiBrowse then ForMode:=UIMode;
  case ForMode of
    uiButton: APart:=TButtonPart.Create;
    uiField: APart:=TFieldPart.Create;
    else Exit;
  end;
  APart.Construct;
  APart.Place(X1, Y1, X2, Y2);
  if AltMode then begin
    case ForMode of
      uiButton: with TButtonPart(APart) do begin
        DrawName:=False;
        Style:=bpsInvisible;
      end;
    end;
  end;
  Add(APart);
  APart.Activate;
end;

procedure TCard.DuplicateActivePart;
var
  DupPart: TPart;
  X, Y, W, H: Integer;
begin
  if Assigned(FMovingPart) or not Assigned(ActivePart) then Exit;
  DupPart:=ActivePart.Duplicate;
  if not Assigned(DupPart) then Exit;
  ActivePart:=DupPart;
  FMovingPart:=DupPart;
  FMovingPartArea:=paCenter;
  if not DupPart.IsPointInside(MouseX, MouseY) then begin
    W:=DupPart.X2 - DupPart.X1;
    H:=DupPart.Y2 - DupPart.Y1;
    X:=MouseX - W div 2;
    Y:=MouseY - H div 2;
    DupPart.Place(X, Y, X + W, Y + H);
  end;
  FMovingPrevRect:=Rect(DupPart.X1, DupPart.Y1, DupPart.X2, DupPart.Y2);
  FPrevX:=MouseX;
  FPrevY:=MouseY;
end;

function TCard.GetDataForID(AID: Cardinal; out ACardData: PCardData): Boolean;
var
  Index: Integer;
begin
  Index:=GetDataIndex(AID);
  if Index=-1 then begin
    SetLength(FData, Length(FData) + 1);
    Index:=Length(FData) - 1;
    FData[Index].PartID:=AID;
  end;
  ACardData:=@FData[Index];
  Result:=True;
end;

function TCard.GetDataForPart(APart: TPart; out ACardData: PCardData): Boolean;
begin
  if not Assigned(APart) or (APart.ID=0) then Exit(False);
  if Assigned(Foreground) and not APart.SharedData then Exit(Foreground.GetDataForPart(APart, ACardData));
  Result:=GetDataForID(APart.ID, ACardData);
end;

procedure TCard.ReleaseDataForID(AID: Cardinal);
var
  i, Index: Integer;
begin
  if Assigned(Foreground) then Foreground.ReleaseDataForID(AID);
  Index:=GetDataIndex(AID);
  if Index=-1 then Exit;
  for i:=Index to Length(FData) - 2 do FData[i]:=FData[i + 1];
  SetLength(FData, Length(FData) - 1);
end;

procedure TCard.ReleaseDataForPart(APart: TPart);
begin
  if Assigned(APart) and (APart.ID <> 0) then ReleaseDataForID(APart.ID);
end;

procedure TCard.ReleaseAllData;
begin
  SetLength(FData, 0);
end;

function TCard.GetNextID: Cardinal;
begin
  if Assigned(Stack) then Result:=Stack.GetNextID else Result:=0;
end;

function TCard.GetSubObjectByID(AID: Cardinal): TObject;
var
  i: Integer;
begin
  Result:=nil;
  if AID=ID then Exit(Self);
  for i:=0 to Count - 1 do begin
    Result:=Parts[i].GetSubObjectByID(AID);
    if Assigned(Result) then Exit;
  end;
end;

function TCard.GetObjectByID(AID: Cardinal): TObject;
begin
  if Assigned(Stack) then Result:=Stack.GetObjectByID(AID) else Result:=GetSubObjectByID(AID);
end;

function TCard.GetSubObjectByName(AName: string): TObject;
var
  i: Integer;
begin
  Result:=nil;
  if AName=Name then Exit(Self);
  for i:=0 to Count - 1 do begin
    Result:=Parts[i].GetSubObjectByName(AName);
    if Assigned(Result) then Exit;
  end;
end;

function TCard.GetObjectByName(AName: string): TObject;
begin
  if Assigned(Stack) then Result:=Stack.GetObjectByName(AName) else Result:=GetSubObjectByName(AName);
end;

function TCard.CallScript(AHandlerName: string; SetCommonVars: Boolean; PassUp: Boolean): Boolean;
begin
  if UIMode=uiBrowse then begin
    SetScriptVariable('me', '&' + IntToStr(ID));
    if SetCommonVars then SetCommonScriptVariables;
    Result:=ScriptHandlerSet.RunHandler(AHandlerName);
    if PassUp and not Result and Assigned(Stack) then Result:=Stack.CallScript(AHandlerName);
  end else Result:=False;
end;

procedure TCard.SetScriptProperty(AName, AValue: string);
begin
  if AName='id' then LILRuntime.SetError('Cannot change the ID of a card')
  else if AName='name' then Name:=AValue
  else FScriptPropertySet.Put(AName, AValue);
end;

function TCard.GetScriptProperty(AName: string): string;
begin
  if AName='id' then Result:=IntToStr(ID)
  else if AName='name' then Result:=Name
  else Result:=FScriptPropertySet.Get(AName);
end;

procedure TCard.WriteToChunkIO(ChunkIO: TChunkIO);
var
  i: Integer;
begin
  with ChunkIO do begin
    BeginChunk(ChunkID(CardChunkID));
    WriteCardinal(ChunkID('IDID'), ID);
    WriteString(ChunkID('NAME'), Name);
    if Assigned(Background) then WriteCardinal(ChunkID('BGID'), Background.ID);
    if Assigned(Image) then begin
      BeginChunk(ChunkID('IMAG'));
      WriteImageToChunkIO(ChunkIO, Image);
      EndChunk()
    end;
    BeginChunk(ChunkID('PRTS'));
    for i:=0 to Count - 1 do Parts[i].WriteToChunkIO(ChunkIO);
    EndChunk();
    BeginChunk(ChunkID('DATA'));
    for i:=0 to Length(FData) - 1 do with FData[i] do begin
      WriteCardinal(ChunkID('PTID'), PartID);
      WriteString(ChunkID('DATA'), Data);
    end;
    EndChunk();
    FScriptHandlerSet.WriteToChunkIO(ChunkIO);
    FScriptPropertySet.WriteToChunkIO(ChunkIO);
    EndChunk();
  end;
end;

procedure TCard.ReadFromChunkIO(ChunkIO: TChunkIO);
var
  ChID, ChunkSize, i: Cardinal;

  function CreatePartForChunkID(ChID: Cardinal): Boolean;
  var
    Part: TPart;
  begin
    if ChID=ChunkID(ButtonPartChunkID) then
      Part:=TButtonPart.Create
    else if ChID=ChunkID(FieldPartChunkID) then
      Part:=TFieldPart.Create
    else
      Exit(False);
    Part.Construct;
    Part.ReadFromChunkIO(ChunkIO);
    Add(Part);
    Result:=True;
  end;

begin
  FLoadedBackgroundID:=0;
  ReleaseAllData;
  with ChunkIO do begin
    while NextChunk(ChID, ChunkSize) do begin
      if ChID=ChunkID('IDID') then
        FID:=ReadCardinal
      else if ChID=ChunkID('NAME') then
        Name:=ReadString
      else if ChID=ChunkID('BGID') then
        FLoadedBackgroundID:=ReadCardinal
      else if ChID=ChunkID('IMAG') then begin
        FreeAndNil(FImage);
        FImage:=TBitmap.Create;
//        FImage.PixelFormat:=pf32bit;
        ReadImageFromChunkIO(ChunkIO, FImage);
      end else if ChID=ChunkID('PRTS') then begin
        while NextChunk(ChID, ChunkSize) do begin
          if not CreatePartForChunkID(ChID) then SkipChunk(ChunkSize);
        end;
      end else if ChID=ChunkID('DATA') then begin
        i:=0;
        while NextChunk(ChID, ChunkSize) do begin
          if ChID=ChunkID('PTID') then
            i:=ReadCardinal
          else if ChID=ChunkID('DATA') then begin
            if i=0 then
              ReadString
            else begin
              SetLength(FData, Length(FData) + 1);
              FData[Length(FData) - 1].Data:=ReadString;
              FData[Length(FData) - 1].PartID:=i;
              i:=0;
            end;
          end;
        end;
      end else if ChID=ChunkID(ScriptHandlerSetChunkID) then
        FScriptHandlerSet.ReadFromChunkIO(ChunkIO)
      else if ChID=ChunkID(ScriptPropertySetChunkID) then
        FScriptPropertySet.ReadFromChunkIO(ChunkIO)
      else SkipChunk(ChunkSize);
    end;
  end;
  if Assigned(FillNextID) and (FillNextID^ <= ID) then FillNextID^:=ID;
end;

procedure TCard.SetBackground(const AValue: TCard);
begin
  if FBackground=AValue then Exit;
  FBackground:=AValue;
  Invalidate;
end;

function TCard.GetCount: Integer;
begin
  if Assigned(FParts) then Result:=FParts.Count else Result:=0;
end;

function TCard.GetActivePart: TPart;
begin
  if Assigned(Foreground) then Result:=Foreground.ActivePart else Result:=FActivePart;
end;

function TCard.GetCursor: TCursor;
begin
  Result:=crDefault;
  if Assigned(PartUnderMouse) then begin
    if PartEditing and PartUnderMouse.AllowEditing then
      Result:=CursorForPartArea(PartUnderMouse.PartAreaAt(MouseX, MouseY))
    else if PartUnderMouse.AllowInteraction then
      Result:=PartUnderMouse.Cursor;
  end;
  if (Result=crDefault) and Painting then
    if UseGrid then Result:=crNone else Result:=crCross;
end;

function TCard.GetNext: TCard;
begin
  if Assigned(Stack) then Result:=Stack.NextCardOf(Self) else Result:=nil;
end;

function TCard.GetPainting: Boolean;
begin
  Result:=UIMode in [uiPen, uiLine, uiBox, uiOval, uiFloodFill];
end;

function TCard.GetPartEditing: Boolean;
begin
  Result:=UIMode in [uiButton, uiField];
end;

function TCard.GetParts(AIndex: Integer): TPart;
begin
  Result:=FParts[AIndex];
end;

function TCard.GetPrevious: TCard;
begin
  if Assigned(Stack) then Result:=Stack.PreviousCardOf(Self) else Result:=nil;
end;

procedure TCard.SetActivePart(const AValue: TPart);
begin
  if Assigned(Foreground) then begin
    Foreground.ActivePart:=AValue;
    Exit;
  end;
  if FActivePart=AValue then Exit;
  if Assigned(FActivePart) then FActivePart.Deactivated;
  FActivePart:=AValue;
  if Assigned(ActivePart) then FActivePart.Activated;
  if Assigned(PartInspector) then PartInspector.Part:=ActivePart;
  Invalidate;
end;

procedure TCard.InvalidatePartReference(APart: TPart);
begin
  if APart=PartUnderMouse then FPartUnderMouse:=nil;
  if APart=ActivePart then ActivePart:=nil;
  if Assigned(Foreground) then begin
    if APart=Foreground.PartUnderMouse then Foreground.FPartUnderMouse:=nil;
    if APart=Foreground.ActivePart then Foreground.ActivePart:=nil;
  end;
end;

procedure TCard.PartVisibilityChanged(APart: TPart);
begin
  if not APart.Visible and not PartEditing then begin
    InvalidatePartReference(APart);
  end;
  // Cause events to happen in case the mouse was under the now invisible part
  if APart.IsPointInside(MouseX, MouseY) then
    HandleMouseMotion(MouseX, MouseY, []);
end;

initialization
  PenPainter:=TPenPainter.Create;
  LinePainter:=TShapePainter.Create(sptLine);
  BoxPainter:=TShapePainter.Create(sptBox);
  OvalPainter:=TShapePainter.Create(sptOval);
  FloodFillPainter:=TFloodFillPainter.Create;
finalization
  FreeAndNil(FloodFillPainter);
  FreeAndNil(OvalPainter);
  FreeAndNil(BoxPainter);
  FreeAndNil(LinePainter);
  FreeAndNil(PenPainter);
end.

