unit CardBrowserUnit;

{$mode objfpc}{$H+}

interface

uses
  Classes, SysUtils, FileUtil, Forms, Controls, Graphics, Dialogs, ComCtrls,
  StdCtrls, Cards;

type

  { TCardBrowser }

  TCardBrowser = class(TForm)
    btOk: TButton;
    btCancel: TButton;
    ilThumbs: TImageList;
    lvCards: TListView;
    procedure FormCreate(Sender: TObject);
    procedure FormShow(Sender: TObject);
    procedure lvCardsCustomDrawItem(Sender: TCustomListView; Item: TListItem;
      State: TCustomDrawState; var DefaultDraw: Boolean);
    procedure lvCardsDblClick(Sender: TObject);
  private
    FStack: TStack;
    NeedsThumbs: array of Boolean;
    function GetCard: TCard;
    function GetItemIndex: Integer;
    procedure SetCard(const AValue: TCard);
    procedure SetStack(const AValue: TStack);
    procedure SetItemIndex(const AValue: Integer);
    procedure MakeThumbnail(ACard: TCard; out ABitmap: TBitmap);
    procedure UpdateCards;
  public
    property ItemIndex: Integer read GetItemIndex write SetItemIndex;
    property Card: TCard read GetCard write SetCard;
    property Stack: TStack read FStack write SetStack;
  end;

var
  CardBrowser: TCardBrowser;

function SelectCard(AStack: TStack; ADefaultCard: TCard; out ASelectedCard: TCard; ACaption: string='Select a Card'; AOkButton: string='&OK'): Boolean;

implementation

function SelectCard(AStack: TStack; ADefaultCard: TCard;
  out ASelectedCard: TCard; ACaption: string; AOkButton: string): Boolean;
begin
  Application.CreateForm(TCardBrowser, CardBrowser);
  CardBrowser.Caption:=ACaption;
  CardBrowser.btOk.Caption:=AOkButton;
  CardBrowser.Stack:=AStack;
  CardBrowser.Card:=ADefaultCard;
  if CardBrowser.ShowModal <> mrOK then Exit(False);
  ASelectedCard:=CardBrowser.Card;
  Result:=Assigned(ASelectedCard);
  FreeAndNil(CardBrowser);
end;

{$R *.lfm}

{ TCardBrowser }

procedure TCardBrowser.SetStack(const AValue: TStack);
begin
  if FStack=AValue then Exit;
  FStack:=AValue;
  UpdateCards;
end;

procedure TCardBrowser.FormCreate(Sender: TObject);
begin
  btOk.Constraints.MinWidth:=btCancel.Width;
end;

procedure TCardBrowser.FormShow(Sender: TObject);
begin
  if lvCards.ItemIndex=-1 then
    lvCards.Items[lvCards.ItemIndex].MakeVisible(False);
end;

procedure TCardBrowser.lvCardsCustomDrawItem(Sender: TCustomListView;
  Item: TListItem; State: TCustomDrawState; var DefaultDraw: Boolean);
var
  Thumb: TBitmap;
begin
  if NeedsThumbs[Item.Index] then begin
    MakeThumbnail(TCard(Item.Data), Thumb);
    ilThumbs.Replace(Item.Index, Thumb, nil);
    FreeAndNil(Thumb);
    NeedsThumbs[Item.Index]:=False;
  end;
  DefaultDraw:=True;
end;

procedure TCardBrowser.lvCardsDblClick(Sender: TObject);
begin
  if lvCards.ItemIndex <> -1 then btOk.Click;
end;

function TCardBrowser.GetItemIndex: Integer;
begin
  Result:=lvCards.ItemIndex;
end;

function TCardBrowser.GetCard: TCard;
begin
  if lvCards.ItemIndex=-1 then
    Result:=nil
  else
    Result:=Stack[lvCards.ItemIndex];
end;

procedure TCardBrowser.SetCard(const AValue: TCard);
begin
  ItemIndex:=Stack.IndexOf(AValue);
end;

procedure TCardBrowser.SetItemIndex(const AValue: Integer);
begin
  lvCards.ItemIndex:=AValue;
end;

procedure TCardBrowser.MakeThumbnail(ACard: TCard; out ABitmap: TBitmap);
var
  TmpBmp: TBitmap;
begin
  TmpBmp:=TBitmap.Create;
  TmpBmp.SetSize(640, 480);
  ACard.PaintIn(TmpBmp.Canvas);

  ABitmap:=TBitmap.Create;
  ABitmap.SetSize(80, 60);
  with ABitmap.Canvas do begin
    StretchDraw(Rect(0, 0, 80, 60), TmpBmp);
    Brush.Style:=bsClear;
    Pen.Style:=psSolid;
    Pen.Width:=1;
    Pen.Color:=clBlack;
    Rectangle(0, 0, 80, 60);
  end;

  FreeAndNil(TmpBmp);
end;

procedure TCardBrowser.UpdateCards;
var
  i: Integer;
  Item: TListItem;
  ThumbBmp: TBitmap;
begin
  lvCards.Items.Clear;
  SetLength(NeedsThumbs, Stack.Count);
  ThumbBmp:=TBitmap.Create;
  ThumbBmp.SetSize(1, 1);
  for i:=0 to Stack.Count - 1 do begin
    ilThumbs.Add(ThumbBmp, nil);
    Item:=lvCards.Items.Add;
    Item.Data:=Stack[i];
    Item.ImageIndex:=i;
    if Trim(Stack[i].Name)='' then
      Item.Caption:='(&' + IntToStr(Stack[i].ID) + ')'
    else
      Item.Caption:=Stack[i].Name + ' (&' + IntToStr(Stack[i].ID) + ')';
    NeedsThumbs[i]:=True;
  end;
  FreeAndNil(ThumbBmp);
end;

end.

