// Very very very very very simple wiki written in C++ using ServerMe.H

// The port to listen to
#define THE_PORT 8088

// Make MSVC stop whining when using portable stuff
#define _CRT_SECURE_NO_WARNINGS

#include "ServerMe.H"
#include <signal.h>
#include <string>
#include <ctype.h>
#if (defined(_WIN32) || defined(WIN32)) && !defined(SMEH_NO_WINDOWS)
#include <direct.h>
#endif

using namespace std;

bool LRunning = true;     // Running flag

// Extract the name of the page from the given address, returns
// false if the page is invalid
bool ExtractPageName(const string& addr, string& name)
{
    size_t i = 0;
    name = "";
    // Find the first character after the last '/'
    for (int j=0; j < addr.length(); ++j)
    {
        if (addr[i] == '?' || addr[i] == '&') break;
        if (addr[j] == '/') i = j + 1;
    }
    // Collect all the characters up to the name or request parameters
    for (; i < addr.length(); ++i)
    {
        if (addr[i] == '?' || addr[i] == '&') break;
        char ch = tolower(addr[i]);
        if ((ch >= 'a' && ch <= 'z') ||
            (ch >= '0' && ch <= '9') ||
            ch == '_' || ch == '-')
        {
            name += ch;
        }
        else
        {
            return false;
        }
    }
    return name.length() > 0;
}

// Load page data, returns false on error
bool LoadPage(const string& name, string& data)
{
    string filename = "pages/" + name + ".txt";
    FILE* f = fopen(filename.c_str(), "rb");
    if (!f) return false;
    fseek(f, 0, SEEK_END);
    long size = ftell(f);
    fseek(f, 0, SEEK_SET);
    char* buff = (char*)malloc(size + 1);
    if (fread(buff, 1, size, f) != size)
    {
        free(buff);
        fclose(f);
        return false;
    }
    buff[size] = 0;
    data = buff;
    free(buff);
    fclose(f);
    return true;
}

// Save page data, returns false on error
bool SavePage(const string& name, const string& data)
{
    string newfilename = "pages/" + name + ".txt.new";
    string oldfilename = "pages/" + name + ".txt.old";
    string filename = "pages/" + name + ".txt";
    FILE* f = fopen(newfilename.c_str(), "wb");
    if (!f) return false;
    bool r = fwrite(data.c_str(), 1, data.length(), f) == data.length();
    fclose(f);
    if (r)
    {
        f = fopen(filename.c_str(), "rb");
        if (f)
        {
            fclose(f);
            remove(oldfilename.c_str());
            if (rename(filename.c_str(), oldfilename.c_str()))
            {
                remove(newfilename.c_str());
                return false;
            }
        }
        if (rename(newfilename.c_str(), filename.c_str()))
        {
            rename(oldfilename.c_str(), filename.c_str());
            remove(newfilename.c_str());
            return false;
        }
        return true;
    }
    return false;
}

// Send an HTML page for the given request
void SendPage(SWebServerRequest& request, const string& html)
{
    string code("<html><head><title>CPPWiki</title></head><body bgcolor=#252525 text=#d0d0d0 link=#ffffff alink=#ffffff vlink=#ffffff>");
    code += html;
    code += "</body></html>";
    request.Socket.SendContent(code.c_str());
}

// Handle a page edit request
void HandlePageEdit(SWebServerRequest& request, const string& name)
{
    string code, data;
    LoadPage(name, data);
    code = "Edit " + name + ":<form method='get' action='/save/" + name + "'>";
    code += "<textarea style='background: #202020; color: #d0d0d0' name=data cols=70 rows=25>" + data + "</textarea>";
    code += "<input type=hidden name=eof value=1><input type='submit'></form>";
    SendPage(request, code);
}

// Handle a page view request
void HandlePageView(SWebServerRequest& request, const string& name, bool saved)
{
    string code, data;
    if (!LoadPage(name, data))
    {
        SendPage(request, "Unknown page <b>" + name + "</b>. <a href='/edit/" + name + "'>Create it</a>.");
        return;
    }
    if (saved) code = "<b>Page saved!</b> - ";
    code += "<a href='/edit/" + name + "'>Edit this page</a> - <a href='/'>Main</a> - <a href='/bye'>Kill the server</a><br><br>";
    for (size_t i=0; i < data.length(); ++i)
    {
        if (data[i] == '[')
        {
            string linkname;
            for (size_t j=i + 1; j < data.length(); ++j)
            {
                char ch = tolower(data[j]);
                if (ch == ']')
                {
                    code += "<a href='/wiki/" + linkname + "'>" + linkname + "</a>";
                    i = j;
                    break;
                }
                if ((ch >= 'a' && ch <= 'z') ||
                    (ch >= '0' && ch <= '9') ||
                    ch == '_' || ch == '-')
                {
                    linkname += data[j];
                }
                else
                {
                    code += '[';
                    break;
                }
            }
        }
        else switch (data[i])
        {
        case '<': code += "&lt;"; break;
        case '>': code += "&gt;"; break;
        case '&': code += "&amp;"; break;
        case '\n': code += "<br>"; break;
        default: code += data[i];
        }
    }
    SendPage(request, code);
}

// Handle a page save request
void HandlePageSave(SWebServerRequest& request, const string& name)
{
    // Ensure the eof=1 mark is part of the request (it should be placed
    // after the data - if the browser sends too much data or doesn't
    // send the entire request, this will be missing)
    if (!strstr(request.Address, "&eof=1"))
    {
        SendPage(request, "The page contents is too large. Ensure it is at most around 60KB and that your browser can send GET requests that large.");
        return;
    }
    // Extract the data from the request
    string data;
    const char* reqpart = strstr(request.Address, "?data=");
    if (!reqpart) reqpart = strstr(request.Address, "&data=");
    if (!reqpart)
    {
        SendPage(request, "Missing data");
        return;
    }
    reqpart += 6;
    for (size_t i=0; reqpart[i] && reqpart[i] != '&'; ++i)
    {
        char ch = reqpart[i];
        if (ch == '+') ch = ' ';
        else if (ch == '%' && isxdigit(reqpart[i + 1]) && isxdigit(reqpart[i + 2]))
        {
            char hex[3];
            hex[0] = reqpart[i + 1];
            hex[1] = reqpart[i + 2];
            hex[2] = 0;
            i += 2;
            ch = (char)strtol(hex, NULL, 16);
        }
        data += ch;
    }
    if (SavePage(name, data))
    {
        HandlePageView(request, name, true);
    }
    else
    {
        SendPage(request, "Failed to save the page contents.");
    }
}

// Called for every connection
void ConnectionCallback(SWebServerRequest& request, void*)
{
    string name;
    
    // Store the address
    string address(request.Address);
    
    // Check if this is a request to kill the server
    if (address.substr(0, 7) == "/byebye")
    {
        LRunning = false;
        return;
    }
    
    // Check if this is a request to show the server kill page
    if (address.substr(0, 4) == "/bye")
    {
        SendPage(request, "<a href='/byebye'>Click here to kill the server</a>.");
        return;
    }
    
    // Check if this is a page edit request
    if (address.substr(0, 6) == "/edit/")
    {
        if (!ExtractPageName(address, name))
        {
            SendPage(request, "Invalid page name");
            return;
        }
        HandlePageEdit(request, name);
        return;
    }

    // Check if this is a page save request
    if (address.substr(0, 6) == "/save/")
    {
        if (!ExtractPageName(address, name))
        {
            SendPage(request, "Invalid page name");
            return;
        }
        HandlePageSave(request, name);
        return;
    }

    // Check if this is a page request and if not, make it a default page
    // request
    if (address.substr(0, 6) != "/wiki/")
    {
        address = "/wiki/main";
    }

    // Handle page view request
    if (!ExtractPageName(address, name))
    {
        SendPage(request, "Invalid page name");
        return;
    }
    HandlePageView(request, name, false);
}

// Called when the program is asked to terminate
void termsig(int signal)
{
    if (signal == SIGTERM)
    {
        LRunning = false;
    }
}

#if (defined(_WIN32) || defined(WIN32)) && !defined(SMEH_NO_WINDOWS)
int WINAPI WinMain(HINSTANCE, HINSTANCE, LPSTR, int)
#else
int main()
#endif
{
    // Initialize the server
    if (!CWebServer::Initialize())
    {
        #if (defined(_WIN32) || defined(WIN32)) && !defined(SMEH_NO_WINDOWS)
        MessageBox(NULL, "Failed to initialize", "Error", MB_ICONERROR|MB_OK);
        #else
        fprintf(stderr, "Failed to initialize\n");
        #endif
        return EXIT_FAILURE;
    }

    // Create a pages subdirectory
    #if (defined(_WIN32) || defined(WIN32)) && !defined(SMEH_NO_WINDOWS)
    mkdir("pages");
    #else
    mkdir("pages", 0700);
    #endif

    // Start listening for connections
    CWebServer server;
    if (!server.Listen(THE_PORT))
    {
        #if (defined(_WIN32) || defined(WIN32)) && !defined(SMEH_NO_WINDOWS)
        MessageBox(NULL, "Failed to start listening for connections\n", "Error", MB_ICONERROR|MB_OK);
        #else
        fprintf(stderr, "Failed to start listening for connections\n");
        #endif
        return EXIT_FAILURE;
    }

    signal(SIGTERM, termsig);

    // Setup the callback
    server.SetCallback(ConnectionCallback);

    // Handle requests
    while (LRunning)
    {
        // Block for 1sec until a request comes (the 1sec delay
        // allows us to periodically check if we should terminate)
        server.Wait(1000);
    }

    // Stop the server
    server.Stop();

    // Shutdown the server
    CWebServer::Shutdown();

    return 0;
}
