unit UMain;

{$mode objfpc}{$H+}
{$modeswitch nestedprocvars}

interface

uses
  Classes, SysUtils, dbf, Forms, Controls, Graphics, Dialogs, Menus, ComCtrls,
  ActnList, ExtCtrls, StdCtrls, UBugs, UMoveUnder, Nodes, FirstTimeInit;

type

  TElementChangeProc = procedure(Element: TElement) is nested;

  { TMain }

  TMain = class(TForm)
    acElementsRemove: TAction;
    acFilterShowGroups: TAction;
    acFilterShowTopics: TAction;
    acFilterShowTasks: TAction;
    acFilterShowBugs: TAction;
    acFilterShowRequests: TAction;
    acFilterShowToDoElements: TAction;
    acFilterShowWorkInProgressElements: TAction;
    acFilterShowImplementedElements: TAction;
    acFilterShowDoneElements: TAction;
    acFilterSearch: TAction;
    acFilterClear: TAction;
    acElementsAddSubelement: TAction;
    edSearch: TEdit;
    MainMenu1: TMainMenu;
    MenuItem12: TMenuItem;
    MenuItem13: TMenuItem;
    MenuItem14: TMenuItem;
    MenuItem15: TMenuItem;
    MenuItem16: TMenuItem;
    MenuItem17: TMenuItem;
    MenuItem18: TMenuItem;
    MenuItem19: TMenuItem;
    MenuItem20: TMenuItem;
    MenuItem21: TMenuItem;
    MenuItem22: TMenuItem;
    MenuItem23: TMenuItem;
    MenuItem24: TMenuItem;
    MenuItem25: TMenuItem;
    mFilters: TMenuItem;
    MenuItem2: TMenuItem;
    MenuItem3: TMenuItem;
    MenuItem4: TMenuItem;
    MenuItem5: TMenuItem;
    MenuItem6: TMenuItem;
    MenuItem10: TMenuItem;
    MenuItem11: TMenuItem;
    ToolBar1: TToolBar;
    ToolButton1: TToolButton;
    ImageList1: TImageList;
    ToolButton18: TToolButton;
    ToolButton19: TToolButton;
    ToolButton2: TToolButton;
    ToolButton20: TToolButton;
    ToolButton3: TToolButton;
    ToolButton4: TToolButton;
    ToolButton7: TToolButton;
    ActionList1: TActionList;
    acElementsAdd: TAction;
    mElements: TMenuItem;
    MenuItem1: TMenuItem;
    ToolButton8: TToolButton;
    ApplicationProperties1: TApplicationProperties;
    plSidebar: TPanel;
    tvElements: TTreeView;
    Splitter1: TSplitter;
    Panel1: TPanel;
    Label1: TLabel;
    Label2: TLabel;
    edTitle: TEdit;
    Label3: TLabel;
    cbType: TComboBox;
    cbState: TComboBox;
    Label5: TLabel;
    mDescription: TMemo;
    Shape1: TShape;
    Shape2: TShape;
    Label6: TLabel;
    cbPriority: TComboBox;
    ToolButton5: TToolButton;
    ToolButton6: TToolButton;
    ToolButton9: TToolButton;
    ToolButton10: TToolButton;
    ToolButton11: TToolButton;
    ToolButton12: TToolButton;
    ToolButton13: TToolButton;
    ToolButton14: TToolButton;
    ToolButton15: TToolButton;
    ToolButton16: TToolButton;
    ToolButton17: TToolButton;
    acElementsMoveUnder: TAction;
    MenuItem26: TMenuItem;
    MenuItem27: TMenuItem;
    MenuItem28: TMenuItem;
    ToolButton21: TToolButton;
    mHelp: TMenuItem;
    mHelpAbout: TMenuItem;
    acFileExportHTML: TAction;
    MenuItem7: TMenuItem;
    MenuItem8: TMenuItem;
    sdExportHTML: TSaveDialog;
    acFileNew: TAction;
    acFileOpen: TAction;
    acFileSave: TAction;
    acFileSaveAs: TAction;
    OpenDialog1: TOpenDialog;
    SaveDialog1: TSaveDialog;
    FirstTimeInit1: TFirstTimeInit;
    procedure acElementsAddExecute(Sender: TObject);
    procedure acElementsAddSubelementExecute(Sender: TObject);
    procedure acElementsRemoveExecute(Sender: TObject);
    procedure acFilterClearExecute(Sender: TObject);
    procedure acFilterShowBugsExecute(Sender: TObject);
    procedure acFilterShowDoneElementsExecute(Sender: TObject);
    procedure acFilterShowGroupsExecute(Sender: TObject);
    procedure acFilterShowImplementedElementsExecute(Sender: TObject);
    procedure acFilterShowRequestsExecute(Sender: TObject);
    procedure acFilterShowTasksExecute(Sender: TObject);
    procedure acFilterShowToDoElementsExecute(Sender: TObject);
    procedure acFilterShowTopicsExecute(Sender: TObject);
    procedure acFilterShowWorkInProgressElementsExecute(Sender: TObject);
    procedure MenuItem11Click(Sender: TObject);
    procedure FormShow(Sender: TObject);
    procedure tvElementsSelectionChanged(Sender: TObject);
    procedure edTitleChange(Sender: TObject);
    procedure mDescriptionChange(Sender: TObject);
    procedure cbTypeChange(Sender: TObject);
    procedure cbStateChange(Sender: TObject);
    procedure cbPriorityChange(Sender: TObject);
    procedure edSearchKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState);
    procedure tvElementsExpanded(Sender: TObject; Node: TTreeNode);
    procedure tvElementsCollapsed(Sender: TObject; Node: TTreeNode);
    procedure mHelpAboutClick(Sender: TObject);
    procedure acElementsMoveUnderExecute(Sender: TObject);
    procedure acFileExportHTMLExecute(Sender: TObject);
    procedure acFileNewExecute(Sender: TObject);
    procedure FormCloseQuery(Sender: TObject; var CanClose: boolean);
    procedure acFileSaveExecute(Sender: TObject);
    procedure acFileSaveAsExecute(Sender: TObject);
    procedure acFileOpenExecute(Sender: TObject);
    procedure ApplicationProperties1DropFiles(Sender: TObject;
      const FileNames: array of String);
    procedure FirstTimeInit1Initialize(Sender: TObject);
  private
    FileName: string;
    IgnoreUpdateEvents: Boolean;
    IgnoreCollapseEvents: Boolean;
    SearchText: string;
    procedure UpdateCaption;
    procedure UpdateElementsTree;
    procedure UpdateElementTreeNode(TreeNode: TTreeNode);
    procedure ChangeSelectedElements(Proc: TElementChangeProc);
    procedure ClearData;
    procedure LoadDataFromSelectedElements;
    procedure AddNewElement(ParentNode: TNode);
    procedure SetSearchText(AText: string);
    procedure NewFile;
    procedure SaveToFile(AFileName: string);
    procedure LoadFromFile(AFileName: string);
  public
    procedure BugsModified;
  end;

var
  Main: TMain;

implementation

uses
  LCLType, UState;

{$R *.lfm}

{ TMain }

procedure TMain.MenuItem11Click(Sender: TObject);
begin
  Close;
end;

procedure TMain.FormShow(Sender: TObject);
begin
  UpdateCaption;
  UpdateElementsTree;
end;

procedure TMain.acFilterShowGroupsExecute(Sender: TObject);
begin
  acFilterShowGroups.Checked:=not acFilterShowGroups.Checked;
  UpdateElementsTree;
end;

procedure TMain.acFilterShowImplementedElementsExecute(Sender: TObject);
begin
  acFilterShowImplementedElements.Checked:=not acFilterShowImplementedElements.Checked;
  UpdateElementsTree;
end;

procedure TMain.acFilterShowRequestsExecute(Sender: TObject);
begin
  acFilterShowRequests.Checked:=not acFilterShowRequests.Checked;
  UpdateElementsTree;
end;

procedure TMain.acFilterShowBugsExecute(Sender: TObject);
begin
  acFilterShowBugs.Checked:=not acFilterShowBugs.Checked;
  UpdateElementsTree;
end;

procedure TMain.acFilterClearExecute(Sender: TObject);
begin
  acFilterShowGroups.Checked:=True;
  acFilterShowTopics.Checked:=True;
  acFilterShowTasks.Checked:=True;
  acFilterShowBugs.Checked:=True;
  acFilterShowRequests.Checked:=True;
  acFilterShowToDoElements.Checked:=True;
  acFilterShowWorkInProgressElements.Checked:=True;
  acFilterShowImplementedElements.Checked:=True;
  acFilterShowDoneElements.Checked:=True;
  SetSearchText('');
end;

procedure TMain.acElementsAddExecute(Sender: TObject);
begin
  if Assigned(tvElements.Selected) then
    AddNewElement(TElement(tvElements.Selected.Data).Parent)
  else
    AddNewElement(State.Bugs.Elements);
end;

procedure TMain.acElementsAddSubelementExecute(Sender: TObject);
begin
  if Assigned(tvElements.Selected) then
    AddNewElement(TElement(tvElements.Selected.Data))
  else
    ShowMessage('Select a parent element first');
end;

procedure TMain.acElementsRemoveExecute(Sender: TObject);
var
  Tmp: TNode;
  I: Integer;
begin
  if tvElements.Items.SelectionCount=0 then begin
    ShowMessage('Nothing is selected');
    Exit;
  end;
  if MessageDlg('Delete ' + IntToStr(tvElements.Items.SelectionCount) + ' elements?',
    'Do you want to delete the selected ' + IntToStr(tvElements.Items.SelectionCount) + ' elements? This operation cannot be undone!', mtConfirmation, [mbYes, mbNo], 0) <> mrYes then
      Exit;
  Tmp:=TNode.Create;
  for I:=0 to tvElements.Items.SelectionCount - 1 do
    TElement(tvElements.Items.GetSelections(I).Data).Parent:=Tmp;
  Tmp.Clear;
  Tmp.Free;
  UpdateElementsTree;
end;

procedure TMain.acFilterShowDoneElementsExecute(Sender: TObject);
begin
  acFilterShowDoneElements.Checked:=not acFilterShowDoneElements.Checked;
  UpdateElementsTree;
end;

procedure TMain.acFilterShowTasksExecute(Sender: TObject);
begin
  acFilterShowTasks.Checked:=not acFilterShowTasks.Checked;
  UpdateElementsTree;
end;

procedure TMain.acFilterShowToDoElementsExecute(Sender: TObject);
begin
  acFilterShowToDoElements.Checked:=not acFilterShowToDoElements.Checked;
  UpdateElementsTree;
end;

procedure TMain.acFilterShowTopicsExecute(Sender: TObject);
begin
  acFilterShowTopics.Checked:=not acFilterShowTopics.Checked;
  UpdateElementsTree;
end;

procedure TMain.acFilterShowWorkInProgressElementsExecute(Sender: TObject);
begin
  acFilterShowWorkInProgressElements.Checked:=not acFilterShowWorkInProgressElements.Checked;
  UpdateElementsTree;
end;

procedure TMain.tvElementsSelectionChanged(Sender: TObject);
begin
  LoadDataFromSelectedElements;
end;

procedure TMain.edTitleChange(Sender: TObject);
  procedure Change(Element: TElement);
  begin
    Element.Title:=edTitle.Text;
  end;
begin
  ChangeSelectedElements(@Change);
end;

procedure TMain.mDescriptionChange(Sender: TObject);
  procedure Change(Element: TElement);
  begin
    Element.Description:=mDescription.Text;
  end;
begin
  ChangeSelectedElements(@Change);
end;

procedure TMain.cbTypeChange(Sender: TObject);
  procedure Change(Element: TElement);
  begin
    Element.ElementType:=TElementType(cbType.ItemIndex);
  end;
begin
  if cbType.ItemIndex <> -1 then
    ChangeSelectedElements(@Change);
end;

procedure TMain.cbStateChange(Sender: TObject);
  procedure Change(Element: TElement);
  begin
    Element.State:=TElementState(cbState.ItemIndex);
  end;
begin
  if cbState.ItemIndex <> -1 then
    ChangeSelectedElements(@Change);
end;

procedure TMain.cbPriorityChange(Sender: TObject);
  procedure Change(Element: TElement);
  begin
    Element.Priority:=TElementPriority(cbPriority.ItemIndex);
  end;
begin
  if cbPriority.ItemIndex <> -1 then
    ChangeSelectedElements(@Change);
end;

procedure TMain.edSearchKeyDown(Sender: TObject; var Key: Word;
  Shift: TShiftState);
begin
  if Key=VK_RETURN then SetSearchText(edSearch.Text);
end;

procedure TMain.tvElementsExpanded(Sender: TObject; Node: TTreeNode);
begin
  TElement(Node.Data).Expanded:=True;
end;

procedure TMain.tvElementsCollapsed(Sender: TObject; Node: TTreeNode);
begin
  if IgnoreCollapseEvents then Exit;
  TElement(Node.Data).Expanded:=False;
end;

procedure TMain.mHelpAboutClick(Sender: TObject);
begin
  MessageDlg('About Runtime Bugs', 'Runtime Bugs version 1.0 Copyright (C) 2020 Kostas Michalopoulos', mtInformation, [mbOK], 0);
end;

procedure TMain.acElementsMoveUnderExecute(Sender: TObject);
var
  Target: TElement;
  I: Integer;
begin
  Application.CreateForm(TMoveUnder, MoveUnder);
  MoveUnder.tvElement.Items.Assign(tvElements.Items);
  if MoveUnder.ShowModal=mrOK then begin
    if MoveUnder.tvElement.Items.SelectionCount < 1 then begin
      ShowMessage('No element was selected to move under');
      Exit;
    end;
    Target:=TElement(MoveUnder.tvElement.Items.GetSelections(0).Data);
    for I:=0 to tvElements.Items.SelectionCount - 1 do
      if Target.IsDescendantOf(TElement(tvElements.Items.GetSelections(I).Data)) then begin
        ShowMessage('Cannot move "' + TElement(tvElements.Items.GetSelections(I).Data).Title + '" under "' + Target.Title + '" as the latter is a descendant of it');
        Exit;
      end;
    for I:=0 to tvElements.Items.SelectionCount - 1 do
      TElement(tvElements.Items.GetSelections(I).Data).Parent:=Target;
    UpdateElementsTree;
  end;
  FreeAndNil(MoveUnder);
end;

procedure TMain.acFileExportHTMLExecute(Sender: TObject);
var
  Code: TStringList;
  Count: Integer = 0;

  function Escape(s: string): string;
  begin
    Result:=StringReplace(s, '&', '&amp;', [rfReplaceAll]);
    Result:=StringReplace(Result, '<', '&lt;', [rfReplaceAll]);
    Result:=StringReplace(Result, '>', '&gt;', [rfReplaceAll]);
  end;

  procedure ExportElement(Element: TElement);
  var
    I: Integer;
    S, E: string;
  begin
    E:='';
    S:='';
    case Element.ElementType of
      etGroup: S:='<TT><U>GROUP</U> </TT>';
      etTopic: S:='<TT><U>TOPIC</U> </TT>';
      etTask: S:='<TT><U>TASK</U> </TT>';
      etBug: S:='<TT><U>BUG</U> </TT>';
      etRequest: S:='<TT><U>REQUEST</U> </TT>';
    end;
    case Element.State of
      esToDo: begin end;
      esWIP: begin S:=S + ' [WIP] '; end;
      esImplemented: begin S += '<U>'; E:='</U>' + E; end;
      esDone: begin S += '<FONT COLOR=GRAY>'; E:='</FONT>' + E; end;
    end;
    if Element.State <> esDone then begin
      case Element.Priority of
        epBlocker: begin S:=S + '<FONT COLOR=RED><B>'; E:='</B></FONT>' + E; end;
        epHigh: begin S:=S + '<FONT COLOR=GREEN>'; E:='</FONT>' + E; end;
        epNormal: begin end;
        epLow: begin S:=S + '<FONT COLOR=BLUE>'; E:='</FONT>' + E; end;
      end;
    end;
    Code.Add('<LI><A HREF="#E' + IntToStr(Count) + '">' + S + Escape(Trim(Element.Title)) + E + '</A>');
    Inc(Count);
    for I:=0 to Element.ChildCount - 1 do
      if Element.Children[I] is TElement then begin
        Code.Add('<UL>');
        ExportElement(TElement(Element.Children[I]));
        Code.Add('</UL>');
      end;
    Code.Add('</LI>');
  end;

  procedure ExportElementDesc(Element: TElement);
  var
    I: Integer;
    S: string;
  begin
    Code.Add('<A NAME=E' + IntToStr(Count) + '><H3>' + Escape(Trim(Element.Title)) + '</H3>');
    if Element.Parent is TElement then
      Code.Add('<FONT SIZE=+1>Subelement of <U>' + Escape(Trim(TElement(Element.Parent).Title)) + '</U></FONT><BR>');
    Inc(Count);
    S:='<B>Type:</B> <I>';
    case Element.ElementType of
      etGroup: S += 'Group';
      etTopic: S += 'Topic';
      etTask: S += 'Task';
      etBug: S += 'Bug';
      etRequest: S += 'Request';
    end;
    S += '</I> <B>State:</B> <I>';
    case Element.State of
      esToDo: S += 'To-Do';
      esWIP: S += 'Work in Progress';
      esImplemented: S += 'Implemented';
      esDone: S += 'Done';
    end;
    S += '</I> <B>Priority:</B> <I>';
    case Element.Priority of
      epBlocker: S += '<FONT COLOR=RED>Blocker</FONT>';
      epHigh: S += '<FONT COLOR=GREEN>High</FONT>';
      epNormal: S += 'Normal';
      epLow: S += '<FONT COLOR=BLUE>Low</FONT>';
    end;
    Code.Add(S + '</I><BR><B>Description:</B><BLOCKQUOTE>' + StringReplace(Escape(Trim(Element.Description)), #10, '<BR>', [rfReplaceAll]) + '</BLOCKQUOTE><BR>');
    for I:=0 to Element.ChildCount - 1 do
      if Element.Children[I] is TElement then
        ExportElementDesc(TElement(Element.Children[I]));
  end;

var
  I: Integer;
begin
  if not sdExportHTML.Execute then Exit;
  Code:=TStringList.Create;
  Code.Add('<HTML><HEAD><TITLE>Bug Export</TITLE><HEAD><BODY><H1>Bugs</H1>');
  Code.Add('<H2>Bug List</H2><UL>');
  for I:=0 to tvElements.Items.TopLvlCount - 1 do
    ExportElement(TElement(tvElements.Items.TopLvlItems[I].Data));
  Code.Add('</UL>' + IntToStr(Count) + ' elements.<HR><H2>Bug Descriptions</H2>');
  Count:=0;
  for I:=0 to tvElements.Items.TopLvlCount - 1 do
    ExportElementDesc(TElement(tvElements.Items.TopLvlItems[I].Data));
  Code.Add('</BODY></HTML>');
  try
    Code.SaveToFile(sdExportHTML.FileName);
  except
    MessageDlg('Error Exporting', 'Failed to save ' + sdExportHTML.FileName, mtError, [mbOK], 0);
  end;
  Code.Free;
end;

procedure TMain.acFileNewExecute(Sender: TObject);
begin
  if State.Bugs.Modified then
    if MessageDlg('Modified File', 'The bugs file has been modified. If you proceed you will lose these modifications. Do you want to continue?', mtConfirmation, mbYesNo, 0) <> mrYes then Exit;
  NewFile;
end;

procedure TMain.FormCloseQuery(Sender: TObject; var CanClose: boolean);
begin
  if State.Bugs.Modified then
    if MessageDlg('Modified File', 'The bugs file has been modified. If you proceed you will lose these modifications. Do you want to continue?', mtConfirmation, mbYesNo, 0) <> mrYes then
      CanClose:=False;
end;

procedure TMain.acFileSaveExecute(Sender: TObject);
begin
  if FileName='' then begin
    acFileSaveAs.Execute;
    Exit;
  end;
  SaveToFile(FileName);
end;

procedure TMain.acFileSaveAsExecute(Sender: TObject);
begin
  SaveDialog1.FileName:=FileName;
  if SaveDialog1.Execute then SaveToFile(SaveDialog1.FileName);
end;

procedure TMain.acFileOpenExecute(Sender: TObject);
begin
  OpenDialog1.FileName:=FileName;
  if OpenDialog1.Execute then LoadFromFile(OpenDialog1.FileName);
end;

procedure TMain.ApplicationProperties1DropFiles(Sender: TObject;
  const FileNames: array of String);
begin
  if Length(FileNames) < 1 then Exit;
  if Length(FileNames) <> 1 then begin
    ShowMessage('Only a single file can be opened at once');
  end;
  LoadFromFile(FileNames[0]);
end;

procedure TMain.FirstTimeInit1Initialize(Sender: TObject);
begin
  {$IFNDEF MacOSX}
  if (ParamCount > 0) and FileExists(ParamStr(1)) then
    LoadFromFile(ParamStr(1));
  {$ENDIF}
end;

procedure TMain.UpdateCaption;
var
  NewCaption: string;
begin
  NewCaption:=FileName;
  if State.Bugs.Modified then begin
    if NewCaption='' then
      NewCaption:='Untitled*'
    else
      NewCaption += '*';
  end;
  if NewCaption <> '' then NewCaption += ' - ';
  NewCaption += 'Runtime Bugs';
  Caption:=NewCaption;
end;

procedure TMain.UpdateElementsTree;

  procedure AddElementsOf(ParentTreeNode: TTreeNode; ParentElement: TNode);
  var
    I: Integer;

    function FilterElement(Element: TElement): Boolean;
    begin
      Result:=True;
      if not Element.SearchFiltered then Exit(False);
      if (Element.ElementType=etGroup) and not acFilterShowGroups.Checked then Exit(False);
      if (Element.ElementType=etTopic) and not acFilterShowTopics.Checked then Exit(False);
      if (Element.ElementType=etTask) and not acFilterShowTasks.Checked then Exit(False);
      if (Element.ElementType=etBug) and not acFilterShowBugs.Checked then Exit(False);
      if (Element.ElementType=etRequest) and not acFilterShowRequests.Checked then Exit(False);
      if (Element.State=esToDo) and not acFilterShowToDoElements.Checked then Exit(False);
      if (Element.State=esWIP) and not acFilterShowWorkInProgressElements.Checked then Exit(False);
      if (Element.State=esImplemented) and not acFilterShowImplementedElements.Checked then Exit(False);
      if (Element.State=esDone) and not acFilterShowDoneElements.Checked then Exit(False);
    end;

    procedure AddElement(Element: TElement);
    var
      TreeNode: TTreeNode;
    begin
      TreeNode:=tvElements.Items.AddChild(ParentTreeNode, '');
      TreeNode.Data:=Element;
      if Element.HasChildren then AddElementsOf(TreeNode, Element);
      UpdateElementTreeNode(TreeNode);
      TreeNode.Expanded:=Element.Expanded;
    end;

  begin
    for I:=0 to ParentElement.ChildCount - 1 do
      if ParentElement.Children[I] is TElement then
        if FilterElement(TElement(ParentElement.Children[I])) then
          AddElement(TElement(ParentElement.Children[I]));
  end;

begin
  try
    tvElements.BeginUpdate;
    IgnoreCollapseEvents:=True;
    tvElements.Items.Clear;
    AddElementsOf(nil, State.Bugs.Elements);
  finally
    tvElements.EndUpdate;
    IgnoreCollapseEvents:=False;
  end;
end;

procedure TMain.UpdateElementTreeNode(TreeNode: TTreeNode);
var
  Element: TElement;
begin
  Element:=TElement(TreeNode.Data);
  if Assigned(Element) then begin
    TreeNode.Text:=Element.Title + ' (' + cbPriority.Items[Ord(Element.Priority)] + ')';
    TreeNode.StateIndex:=6 + Ord(Element.ElementType);
    TreeNode.ImageIndex:=11 + Ord(Element.State);
    TreeNode.SelectedIndex:=11 + Ord(Element.State);
  end;
end;

procedure TMain.ChangeSelectedElements(Proc: TElementChangeProc);
var
  I: Integer;
  Element: TElement;
begin
  if IgnoreUpdateEvents then Exit;
  for I:=0 to tvElements.SelectionCount - 1 do begin
    Element:=TElement(tvElements.Selections[I].Data);
    if Element=nil then Continue;
    Proc(Element);
    UpdateElementTreeNode(tvElements.Selections[I]);
  end;
end;

procedure TMain.ClearData;
begin
  IgnoreUpdateEvents:=True;
  edTitle.Text:='';
  mDescription.Text:='';
  cbType.ItemIndex:=-1;
  cbState.ItemIndex:=-1;
  cbPriority.ItemIndex:=-1;
  IgnoreUpdateEvents:=False;
end;

procedure TMain.LoadDataFromSelectedElements;
var
  I: Integer;
  First, Element: TElement;
begin
  ClearData;
  try
    IgnoreUpdateEvents:=True;
    First:=nil;
    for I:=0 to tvElements.SelectionCount -1 do begin
      Element:=TElement(tvElements.Selections[I].Data);
      if Element=nil then Continue;
      if First=nil then begin
        First:=Element;
        edTitle.Text:=First.Title;
        mDescription.Text:=First.Description;
        cbType.ItemIndex:=Ord(First.ElementType);
        cbState.ItemIndex:=Ord(First.State);
        cbPriority.ItemIndex:=Ord(First.Priority);
      end else begin
        if First.Title <> Element.Title then edTitle.Text:='(different titles)';
        if First.Description <> Element.Description then mDescription.Text:='(different descriptions)';
        if First.ElementType <> Element.ElementType then cbType.ItemIndex:=-1;
        if First.State <> Element.State then cbState.ItemIndex:=-1;
        if First.Priority <> Element.Priority then cbPriority.ItemIndex:=-1;
      end;
    end;
  finally
    IgnoreUpdateEvents:=False;
  end;
end;

procedure TMain.AddNewElement(ParentNode: TNode);
var
  NewElement: TElement;
begin
  NewElement:=TElement.Create;
  if cbType.ItemIndex=-1 then
    NewElement.ElementType:=etBug
  else
    NewElement.ElementType:=TElementType(cbType.ItemIndex);
  NewElement.State:=esToDo;
  if cbPriority.ItemIndex=-1 then
    NewElement.Priority:=epNormal
  else
    NewElement.Priority:=TElementPriority(cbPriority.ItemIndex);
  NewElement.Title:='New Entry';
  NewElement.Description:='';
  ParentNode.Add(NewElement);
  tvElements.ClearSelection;
  UpdateElementsTree;
  tvElements.Items.FindNodeWithData(NewElement).Selected:=True;
  LoadDataFromSelectedElements;
  edTitle.SelectAll;
  edTitle.SetFocus;
end;

procedure TMain.SetSearchText(AText: string);

  procedure FilterUp(ANode: TNode);
  begin
    while ANode is TElement do begin
      TElement(ANode).SearchFiltered:=True;
      ANode:=ANode.Parent;
    end;
  end;

  procedure FilterDown(AElement: TElement);
  var
    I: Integer;
  begin
    AElement.SearchFiltered:=True;
    for I:=0 to AElement.ChildCount - 1 do
      if AElement.Children[I] is TElement then
        FilterDown(TElement(AElement.Children[I]));
  end;

  procedure FilterNode(ANode: TNode);
  var
    I: Integer;
  begin
    if ANode is TElement then with TElement(ANode) do begin
      if SearchText='' then
        SearchFiltered:=True
      else begin
        SearchFiltered:=(Pos(SearchText, LowerCase(Title)) <> 0) or (Pos(SearchText, LowerCase(Description)) <> 0);
        if SearchFiltered then begin
          FilterUp(ANode);
          FilterDown(TElement(ANode));
          Exit;
        end;
      end;
    end;
    for I:=0 to ANode.ChildCount - 1 do
      FilterNode(ANode.Children[I]);
  end;

var
  I: Integer;
begin
  SearchText:=LowerCase(Trim(AText));
  FilterNode(State.Bugs.Elements);
  UpdateElementsTree;
  if SearchText <> '' then begin
    for I:=0 to tvElements.Items.Count - 1 do
      tvElements.Items[I].Expanded:=True;
  end;
end;

procedure TMain.NewFile;
begin
  ClearData;
  tvElements.Items.Clear;
  FileName:='';
  State.Bugs.Clear;
  State.Bugs.Modified:=False;
  UpdateCaption;
  edSearch.Text:='';
  UpdateElementsTree;
end;

procedure TMain.SaveToFile(AFileName: string);
begin
  try
    State.Bugs.SaveToFile(AFileName);
    FileName:=AFileName;
    State.Bugs.Modified:=False;
  except
    MessageDlg('Error Saving', 'Failed to save ' + AFileName + ': ' + Exception(ExceptObject).Message, mtError, [mbOK], 0);
  end;
  UpdateCaption;
end;

procedure TMain.LoadFromFile(AFileName: string);
begin
  try
    tvElements.Items.Clear;
    State.Bugs.LoadFromFile(AFileName);
    FileName:=AFileName;
  except
    MessageDlg('Error Opening', 'Failed to open ' + AFileName + ': ' + Exception(ExceptObject).Message, mtError, [mbOK], 0);
    FileName:='';
    State.Bugs.Clear;
  end;
  State.Bugs.Modified:=False;
  UpdateElementsTree;
  UpdateCaption;
  ClearData;
  edSearch.Text:='';
end;

procedure TMain.BugsModified;
begin
  UpdateCaption;
end;

end.

