unit UBugs;

{$mode objfpc}{$H+}

interface

uses
  Classes, SysUtils, Nodes, TreeText;

type
  TBugs = class;
  TElement = class;
  TElementArray = array of TElement;
  TElementType = (etGroup, etTopic, etTask, etBug, etRequest);
  TElementState = (esToDo, esWIP, esImplemented, esDone);
  TElementPriority = (epBlocker, epHigh, epNormal, epLow);

  { TBugsNode }

  TBugsNode = class(TNode)
  private
    FBugs: TBugs;
  protected
    procedure ParentChanged; override;
    procedure ChildrenChanged; override;
    procedure ModifyChanged; override;
  end;

  { TElement }

  TElement = class(TNode)
  private
    FElementType: TElementType;
    FTitle: string;
    FDescription: string;
    FState: TElementState;
    FPriority: TElementPriority;
    procedure SetElementType(AValue: TElementType);
    procedure SetTitle(AValue: string);
    procedure SetDescription(AValue: string);
    procedure SetState(AValue: TElementState);
    procedure SetPriority(AValue: TElementPriority);
  protected
    procedure ParentChanged; override;
    procedure ChildrenChanged; override;
  public
    SearchFiltered: Boolean;
    Expanded: Boolean;
    constructor Create; override;
    function GetSubElements: TElementArray;
    function GetSubElementsOfType(AType: TElementType): TElementArray;
    function CanHaveSubElements: Boolean;
    procedure SaveToTextNode(ANode: TTNode);
    procedure LoadFromTextNode(ANode: TTNode);
    property ElementType: TElementType read FElementType write SetElementType;
    property State: TElementState read FState write SetState;
    property Priority: TElementPriority read FPriority write SetPriority;
    property Title: string read FTitle write SetTitle;
    property Description: string read FDescription write SetDescription;
  end;

  { TElements }

  TElements = class(TBugsNode)
  public
  end;

  { TBugs }

  TBugs = class(TComponent)
  private
    FElements: TElements;
    FOnModified: TNotifyEvent;
    function GetModified: Boolean;
    procedure SetModified(AValue: Boolean);
    procedure NodesModified;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure Clear;
    procedure SaveToTextNode(ANode: TTNode);
    procedure LoadFromTextNode(ANode: TTNode);
    procedure SaveToFile(AFileName: string);
    procedure LoadFromFile(AFileName: string);
    property Elements: TElements read FElements;
    property Modified: Boolean read GetModified write SetModified;
    property OnModified: TNotifyEvent read FOnModified write FOnModified;
  end;

implementation

uses
  MiscUtils;

{ TBugsNode }

procedure TBugsNode.ParentChanged;
begin
  MarkModified;
end;

procedure TBugsNode.ChildrenChanged;
begin
  MarkModified;
end;

procedure TBugsNode.ModifyChanged;
begin
  if Assigned(FBugs) then FBugs.NodesModified;
end;

{ TElement }

procedure TElement.SetDescription(AValue: string);
begin
  if FDescription=AValue then Exit;
  FDescription:=AValue;
  MarkModified;
end;

procedure TElement.SetElementType(AValue: TElementType);
begin
  if FElementType=AValue then Exit;
  FElementType:=AValue;
  MarkModified;
end;

procedure TElement.SetState(AValue: TElementState);
begin
  if FState=AValue then Exit;
  FState:=AValue;
  MarkModified;
end;

procedure TElement.SetPriority(AValue: TElementPriority);
begin
  if FPriority=AValue then Exit;
  FPriority:=AValue;
  MarkModified;
end;

procedure TElement.ParentChanged;
begin
  MarkModified;
end;

procedure TElement.ChildrenChanged;
begin
  MarkModified;
end;

constructor TElement.Create;
begin
  inherited Create;
  SearchFiltered:=True;
end;

procedure TElement.SetTitle(AValue: string);
begin
  if FTitle=AValue then Exit;
  FTitle:=AValue;
  MarkModified;
end;

function TElement.GetSubElements: TElementArray;
var
  I: Integer;
begin
  Result:=nil;
  for I:=0 to ChildCount - 1 do
    if Children[I] is TElement then begin
      SetLength(Result, Length(Result) + 1);
      Result[High(Result)]:=TElement(Children[I]);
    end;
end;

function TElement.GetSubElementsOfType(AType: TElementType): TElementArray;
var
  I: Integer;
begin
  Result:=nil;
  for I:=0 to ChildCount - 1 do
    if (Children[I] is TElement) and (TElement(Children[I]).ElementType=AType) then begin
      SetLength(Result, Length(Result) + 1);
      Result[High(Result)]:=TElement(Children[I]);
    end;
end;

function TElement.CanHaveSubElements: Boolean;
begin
  Result:=not (ElementType in [etBug, etRequest]);
end;

procedure TElement.SaveToTextNode(ANode: TTNode);
var
  I: Integer;
  ElementsNode: TTNode;
begin
  ANode['Title']:=Trim(Title);
  case ElementType of
    etGroup: ANode['Type']:='Group';
    etTopic: ANode['Type']:='Topic';
    etTask: ANode['Type']:='Task';
    etBug: ANode['Type']:='Bug';
    etRequest: ANode['Type']:='Request';
  end;
  case State of
    esToDo: ANode['State']:='ToDo';
    esWIP: ANode['State']:='WIP';
    esImplemented: ANode['State']:='Implemented';
    esDone: ANode['State']:='Done';
  end;
  case Priority of
    epBlocker: ANode['Priority']:='Blocker';
    epHigh: ANode['Priority']:='High';
    epNormal: ANode['Priority']:='Normal';
    epLow: ANode['Priority']:='Low';
  end;
  ANode['Description']:=StringReplace(TrimRight(Description), #13#10, #10, [rfReplaceAll]);
  ElementsNode:=TTNode.Create;
  ElementsNode.Name:='SubElements';
  for I:=0 to ChildCount - 1 do
    if Children[I] is TElement then
      TElement(Children[I]).SaveToTextNode(ElementsNode.CreateChild('Element'));
  if ElementsNode.ChildCount > 0 then
    ANode.AddChild(ElementsNode)
  else
    ElementsNode.Free;
end;

procedure TElement.LoadFromTextNode(ANode: TTNode);
var
  SubElements: TTNode;
  NewElement: TElement;
  I: Integer;
begin
  Title:=ANode['Title'];
  ElementType:=TElementType(IndexOfWord(ANode['Type'], ['Group', 'Topic', 'Task', 'Bug', 'Request'], 3));
  State:=TElementState(IndexOfWord(ANode['State'], ['ToDo', 'WIP', 'Implemented', 'Done'], 0));
  Priority:=TElementPriority(IndexOfWord(ANode['Priority'], ['Blocker', 'High', 'Normal', 'Low'], 2));
  Description:=StringReplace(ANode['Description'], #10, LineEnding, [rfReplaceAll]);
  SubElements:=ANode.FindChild('SubElements');
  if Assigned(SubElements) then begin
    for I:=0 to SubElements.ChildCount - 1 do begin
      if SubElements.Children[I].Name <> 'Element' then Continue;
      NewElement:=TElement.Create;
      NewElement.LoadFromTextNode(SubElements.Children[I]);
      Add(NewElement);
    end;
  end;
end;

{ TBugs }

function TBugs.GetModified: Boolean;
begin
  Result:=Elements.IsModified;
end;

procedure TBugs.SetModified(AValue: Boolean);
begin
  if AValue then Elements.MarkModified else Elements.ClearModified;
end;

procedure TBugs.NodesModified;
begin
  if Assigned(FOnModified) then FOnModified(Self);
end;

constructor TBugs.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FElements:=TElements.Create;
  FElements.FBugs:=Self;
end;

destructor TBugs.Destroy;
begin
  FreeAndNil(FElements);
  inherited Destroy;
end;

procedure TBugs.Clear;
begin
  Elements.Clear;
end;

procedure TBugs.SaveToTextNode(ANode: TTNode);
var
  I: Integer;
  Nodes: TTNode;
begin
  Nodes:=ANode.CreateChild('Elements');
  for I:=0 to Elements.ChildCount - 1 do
    if Elements.Children[I] is TElement then
      TElement(Elements.Children[I]).SaveToTextNode(Nodes.CreateChild('Element'));
end;

procedure TBugs.LoadFromTextNode(ANode: TTNode);
var
  I: Integer;
  Nodes: TTNode;
  NewElement: TElement;
begin
  Clear;
  Nodes:=ANode.FindChild('Elements');
  if Assigned(Nodes) then begin
    for I:=0 to Nodes.ChildCount - 1 do begin
      if Nodes.Children[I].Name <> 'Element' then Continue;
      NewElement:=TElement.Create;
      NewElement.LoadFromTextNode(Nodes.Children[I]);
      Elements.Add(NewElement);
    end;
  end;
end;

procedure TBugs.SaveToFile(AFileName: string);
var
  TextNode: TTNode;
begin
  TextNode:=TTNode.Create;
  TextNode.CreateChild('Comment').Value:='This file is meant to be used with Runtime Bugs';
  TextNode.CreateChild('Comment').Value:='Check http://runtimeterror.com/tools/bugs';
  try
    SaveToTextNode(TextNode);
    if not SetTextFileContents(AFileName, TextNode.Code) then
      raise EInOutError.Create('Failed to save ' + AFileName);
  finally
    TextNode.Free;
  end;
end;

procedure TBugs.LoadFromFile(AFileName: string);
var
  TextNode: TTNode;
begin
  Clear;
  TextNode:=TTNode.Create;
  try
    TextNode.Code:=GetTextFileContents(AFileName);
    LoadFromTextNode(TextNode);
  finally
    TextNode.Free;
  end;
end;

end.

