#include "x11winsys.h"
#include <ctype.h>

static ff_font_t standard_font, bold_font, italic_font, mono_font, big_font, small_font;
static ff_font_t* fonts;
static size_t font_count;
static int disable_doublebuff;

#ifndef LFORMS_USE_XFT
#define STRING_FREE_THRESHOLD 4096
static char* utf8buff;

static void free_utf8buff(void)
{
    free(utf8buff);
}

static const char* utf8ize(const char* s)
{
    const unsigned char* utf8 = (const unsigned char*)s;
    unsigned code;
    size_t i = 0, size = STRING_FREE_THRESHOLD;
    if (!utf8buff) {
        utf8buff = malloc(STRING_FREE_THRESHOLD);
        ff_atexit(free_utf8buff);
    }
    while (*utf8 && (utf8 = ff_dec8(utf8, &code))) {
        if (code > 127) code = '?';
        if (i + 2 >= STRING_FREE_THRESHOLD) {
            size += STRING_FREE_THRESHOLD;
            utf8buff = realloc(utf8buff, size);
        }
        utf8buff[i++] = (char)code;
    }
    utf8buff[i] = 0;
    return utf8buff;
}
#endif

static void ff_font_free(ff_font_t font);

static ff_font_t init_default_font(int size, unsigned flags)
{
    ff_font_t default_font = NULL;
    if (!default_font) {
        /* This may look counterintuitive, but it generally tends to
           match the default font that Window Maker uses by default */
#ifdef LFORMS_USE_XFT
        if (flags & FF_MONOSPACE)
            default_font = ff_font_search("Monospace", size, flags|FF_EXACT_FONT);
        else
            default_font = ff_font_search("Sans Serif", size, flags|FF_EXACT_FONT);
#else
        if (flags & FF_MONOSPACE) {
            default_font = ff_font_search("LucidaTypeWriter", size, flags|FF_EXACT_FONT);
            if (!default_font)
                default_font = ff_font_search("Courier", size, flags|FF_EXACT_FONT);
        }
        if (!default_font)
            default_font = ff_font_search("Helvetica", size, flags);
        if (!default_font)
            default_font = ff_font_search("Fixed", size + 1, flags);
#endif
        if (!default_font) {
            if (flags & FF_MONOSPACE)
                default_font = ff_font_search("Monospace", size, flags);
            else
                default_font = ff_font_search("Sans Serif", size, flags);
        }
        if (!default_font)
            default_font = ff_font_search(NULL, size, flags);
    }
    if (!default_font)
        default_font = ff_font_search(NULL, 0, flags);
    if (!default_font)
        default_font = ff_font_search(NULL, 0, FF_NOFLAGS);
    return default_font;
}

void ff_ws_gfx_init(void)
{
    const char *dbstr = getenv("LFORMS_DOUBLEBUFFER");
    int dbval = dbstr ? atoi(dbstr) : 1;
    if (dbval == 0) disable_doublebuff = 1;
    if (!standard_font) {
        standard_font = init_default_font(12, FF_PROPORTIONAL);
        bold_font = init_default_font(12, FF_PROPORTIONAL|FF_BOLD);
        italic_font = init_default_font(12, FF_PROPORTIONAL|FF_ITALIC);
        mono_font = init_default_font(12, FF_MONOSPACE);
        big_font = init_default_font(24, FF_BOLD|FF_PROPORTIONAL);
        small_font = init_default_font(10, FF_PROPORTIONAL);
    }
}

void ff_ws_gfx_shutdown(void)
{
    size_t i;
    for (i=0; i < font_count; i++) ff_font_free(fonts[i]);
    free(fonts);
    font_count = 0;
}

static void setup_gc(ff_gc_t gc, Drawable d, wswin_t* win)
{
    gc->d = d;
    gc->font = NULL;
    ff_font(gc, win && win->ffwindow ? ff_font_of(win->ffwindow) : standard_font);
}

ff_gc_t ff_ws_create_gc(Window w, wswin_t* win)
{
    ff_gc_t gc = calloc(1, sizeof(struct _ff_gc_t));
    gc->w = w;
    if (!win->width || !win->height)
        ff_ws_area(win, NULL, NULL, &win->width, &win->height);
    gc->dx2 = win->width - 1;
    gc->dy2 = win->height - 1;
    if (!disable_doublebuff && win->flags & FF_DOUBLEBUFFER) {
        if (!win->db || win->dbw != win->width || win->dbh != win->height) {
            win->dbw = win->width;
            win->dbh = win->height;
            if (win->db) XFreePixmap(DISPLAY, win->db);
            win->db = XCreatePixmap(DISPLAY, DefaultRootWindow(DISPLAY), (unsigned)win->width, (unsigned)win->height, (unsigned)DefaultDepth(DISPLAY, DefaultScreen(DISPLAY)));
            if (!win->db) {
                free(gc);
                return NULL;
            }
        }
        gc->gc = XCreateGC(DISPLAY, win->db, 0, NULL);
        if (!gc->gc) {
            free(gc);
            return NULL;
        }
        setup_gc(gc, win->db, win);
    } else {
        gc->gc = XCreateGC(DISPLAY, w, 0, NULL);
        if (!gc->gc) {
            free(gc);
            return NULL;
        }
        setup_gc(gc, w, win);
    }

    return gc;
}

void ff_ws_free_gc(ff_gc_t gc, wswin_t* win)
{
    if (gc) {
        if (!disable_doublebuff && win->flags & FF_DOUBLEBUFFER)
            XCopyArea(DISPLAY, win->db, gc->w, gc->gc, 0, 0, (unsigned)win->dbw, (unsigned)win->dbh, 0, 0);
        XFreeGC(DISPLAY, gc->gc);
        free(gc);
    }
}

void ff_color(ff_gc_t gc, ff_color_t color)
{
    if (!gc) return;
    gc->color = color;
    XSetForeground(DISPLAY, gc->gc, (unsigned long)color);
    XSetBackground(DISPLAY, gc->gc, (unsigned long)color);
}

void ff_font(ff_gc_t gc, ff_font_t font)
{
    if (!gc) return;
    if (!font) font = standard_font;
    if (font != gc->font) {
        gc->font = font;
#ifndef LFORMS_USE_XFT
        if (font->font) {
            XGCValues gcv;
            gcv.font = font->font->fid;
            XChangeGC(DISPLAY, gc->gc, GCFont, &gcv);
        }
#endif
    }
}

void ff_fill(ff_gc_t gc, int x1, int y1, int x2, int y2)
{
    if (!gc) return;
    XFillRectangle(DISPLAY, gc->d, gc->gc, x1, y1, (unsigned)(x2 - x1 + 1), (unsigned)(y2 - y1 + 1));
}

void ff_rect(ff_gc_t gc, int x1, int y1, int x2, int y2)
{
    if (!gc) return;
    XDrawRectangle(DISPLAY, gc->d, gc->gc, x1, y1, (unsigned)(x2 - x1), (unsigned)(y2 - y1));
}

void ff_line(ff_gc_t gc, int x1, int y1, int x2, int y2)
{
    if (!gc) return;
    XDrawLine(DISPLAY, gc->d, gc->gc, x1, y1, x2, y2);
}

void ff_oval(ff_gc_t gc, int x1, int y1, int x2, int y2)
{
    if (!gc) return;
    XDrawArc(DISPLAY, gc->d, gc->gc, x1, y1, (unsigned)(x2 - x1), (unsigned)(y2 - y1), 0, 23040);
}

void ff_disc(ff_gc_t gc, int x1, int y1, int x2, int y2)
{
    XRectangle r;
    if (!gc) return;
    /* It seems that XFillArc actually fills the inside of the circular
       area that the bounds specify, however sometimes we get an extra
       pixel over those bounds so it isn't 100% accurate.  To avoid this
       we expand the passed parameters a bit and set up a clipping area
       so that to avoid drawing outside of the specified bounds */
    r.x = 0;
    r.y = 0;
    r.width = (unsigned short)(x2 - x1 + 1);
    r.height = (unsigned short)(y2 - y1 + 1);
    XSetClipRectangles(DISPLAY, gc->gc, x1, y1, &r, 1, Unsorted);
    XFillArc(DISPLAY, gc->d, gc->gc, x1 - 1, y1 - 1, r.width + 1U, r.height + 1U, 0, 23040);
    XSetClipMask(DISPLAY, gc->gc, None);
}

void ff_pixel(ff_gc_t gc, int x, int y)
{
    if (!gc) return;
    XDrawPoint(DISPLAY, gc->d, gc->gc, x, y);
}

void ff_text(ff_gc_t gc, int x, int y, const char* text)
{
#ifdef LFORMS_USE_XFT
    XftDraw* draw = XftDrawCreate(DISPLAY, gc->d, DefaultVisual(DISPLAY, DefaultScreen(DISPLAY)), DefaultColormap(DISPLAY, DefaultScreen(DISPLAY)));
    XftColor color;
    if (!draw) return;
    color.pixel = gc->color;
    color.color.red = (unsigned short)(((gc->color&0xFF0000U)>>16U)<<8U);
    color.color.green = (unsigned short)((gc->color&0x00FF00U));
    color.color.blue = (unsigned short)((gc->color&0x0000FFU)<<8U);
    color.color.alpha = 0xFFFFU;
    y -= gc->font->font->descent;
    XftDrawStringUtf8(draw, &color, gc->font->font, x, y, (unsigned char*)text, (int)strlen(text));
    XftDrawDestroy(draw);
    if ((gc->font->flags & FF_UNDERLINE) ||
        (gc->font->flags & FF_STRIKEOUT)) {
        XGlyphInfo gi;
        XftTextExtentsUtf8(DISPLAY, gc->font->font, (const FcChar8*)text, (int)strlen(text), &gi);
        if (gc->font->flags & FF_UNDERLINE)
            XDrawLine(DISPLAY, gc->d, gc->gc, x, y + 1, x + gi.xOff, y + 1);
        if (gc->font->flags & FF_STRIKEOUT)
            XDrawLine(DISPLAY, gc->d, gc->gc, x, y - gc->font->font->descent + gi.yOff/2, x + gi.xOff, y - gc->font->font->descent + gi.yOff/2);
    }
#else
    int dir, asc, desc;
    XCharStruct cs;
    if (!gc) return;
    text = utf8ize(text);
    XQueryTextExtents(DISPLAY, XGContextFromGC(gc->gc), text, (int)strlen(text), &dir, &asc, &desc, &cs);
    XDrawString(DISPLAY, gc->d, gc->gc, x, y - desc, text, (int)strlen(text));
    if ((gc->font->flags & FF_UNDERLINE) ||
        (gc->font->flags & FF_STRIKEOUT)) {
        if (gc->font->flags & FF_UNDERLINE)
            XDrawLine(DISPLAY, gc->d, gc->gc, x, y - desc + 1, x + cs.width, y - desc + 1);
        if (gc->font->flags & FF_STRIKEOUT)
            XDrawLine(DISPLAY, gc->d, gc->gc, x, y - (desc + asc)/2, x + cs.width, y - (desc + asc)/2);
    }
#endif
}

void ff_text_size(ff_gc_t gc, const char* text, int* width, int* height)
{
#ifdef LFORMS_USE_XFT
    XGlyphInfo gi;
    XftTextExtentsUtf8(DISPLAY, gc->font->font, (const FcChar8*)text, (int)strlen(text), &gi);
    if (width) *width = gi.xOff;
    if (height) *height = gc->font->font->height;
#else
    int dir, asc, desc;
    XCharStruct cs;
    if (!gc) {
        if (width) *width = 0;
        if (height) *height = 0;
        return;
    }
    text = utf8ize(text);
    if (text[0]) XQueryTextExtents(DISPLAY, XGContextFromGC(gc->gc), text, (int)strlen(text), &dir, &asc, &desc, &cs);
    else XQueryTextExtents(DISPLAY, XGContextFromGC(gc->gc), " ", 1, &dir, &asc, &desc, &cs);
    if (width) *width = cs.width;
    if (height) *height = asc + desc;
#endif
}

void ff_draw(ff_gc_t gc, int x, int y, ff_bitmap_t bmp)
{
    if (!gc || !bmp) return;
    if (bmp->mask) {
        XSetClipOrigin(DISPLAY, gc->gc, x, y);
        XSetClipMask(DISPLAY, gc->gc, bmp->mask);
    }
    XCopyArea(DISPLAY, bmp->pixmap, gc->d, gc->gc, 0, 0, (unsigned)bmp->width, (unsigned)bmp->height, x, y);
    if (bmp->mask) XSetClipMask(DISPLAY, gc->gc, None);
}

void ff_draw_part(ff_gc_t gc, int x, int y, ff_bitmap_t bmp, int x1, int y1, int x2, int y2)
{
    if (!gc || !bmp || x2 < x1 || y2 < y1) return;
    if (bmp->mask) {
        XSetClipOrigin(DISPLAY, gc->gc, x, y);
        XSetClipMask(DISPLAY, gc->gc, bmp->mask);
    }
    XCopyArea(DISPLAY, bmp->pixmap, gc->d, gc->gc, x1, y1, (unsigned)(x2 - x1 + 1), (unsigned)(y2 - y1 + 1), x, y);
    if (bmp->mask) XSetClipMask(DISPLAY, gc->gc, None);
}

ff_bitmap_t ff_bitmap(int width, int height, const ff_color_t* pixels)
{
    ff_bitmap_t bmp;
    if (width < 1 || height < 1) return NULL;
    bmp = calloc(1, sizeof(struct _ff_bitmap_t));
    ff_refobj_init(FF_REFOBJ(bmp), (ff_refobj_dtor_proc_t)ff_bitmap_free);
    bmp->width = width;
    bmp->height = height;
    bmp->pixmap = XCreatePixmap(DISPLAY, DefaultRootWindow(DISPLAY), (unsigned)width, (unsigned)height, (unsigned)DefaultDepth(DISPLAY, DefaultScreen(DISPLAY)));
    if (!bmp->pixmap) {
        free(bmp);
        return NULL;
    }
    if (pixels) ff_bitmap_pixels(bmp, pixels);
    return bmp;
}

void ff_bitmap_free(ff_bitmap_t bmp)
{
    if (!bmp) return;
    if (bmp->gc) {
        if (bmp->gc->gc) XFreeGC(DISPLAY, bmp->gc->gc);
        free(bmp->gc);
    }
    if (bmp->mask) XFreePixmap(DISPLAY, bmp->mask);
    XFreePixmap(DISPLAY, bmp->pixmap);
    free(bmp);
}

void ff_bitmap_pixels(ff_bitmap_t bmp, const ff_color_t* pixels)
{
    XImage* img;
    int x, y, hastc = 0;
    ff_gc_t gc;
    const ff_color_t* p = pixels;
    if (!bmp || !pixels) return;
    img = XGetImage(DISPLAY, bmp->pixmap, 0, 0, (unsigned)bmp->width, (unsigned)bmp->height, AllPlanes, ZPixmap);
    for (y=0; y<bmp->height; y++)
        for (x=0; x<bmp->width; x++,pixels++) {
            XPutPixel(img, x, y, *pixels);
            if (*pixels == 0xFF00FF) hastc = 1;
        }
    gc = ff_bitmap_gc(bmp);
    if (gc)
        XPutImage(DISPLAY, bmp->pixmap, gc->gc, img, 0, 0, 0, 0, (unsigned)bmp->width, (unsigned)bmp->height);
    if (!hastc && bmp->mask) {
        XFreePixmap(DISPLAY, bmp->mask);
        bmp->mask = 0;
    } else if (hastc) {
        if (!bmp->mask) bmp->mask = XCreatePixmap(DISPLAY, DefaultRootWindow(DISPLAY), (unsigned)bmp->width, (unsigned)bmp->height, 1);
        if (bmp->mask) {
            GC gc = XCreateGC(DISPLAY, bmp->mask, 0, NULL);
            if (gc) {
                pixels = p;
                XSetForeground(DISPLAY, gc, 1);
                XFillRectangle(DISPLAY, bmp->mask, gc, 0, 0, (unsigned)bmp->width, (unsigned)bmp->height);
                XSetForeground(DISPLAY, gc, 0);
                for (y=0; y<bmp->height; y++)
                    for (x=0; x<bmp->width; x++,pixels++)
                        if (*pixels == FF_RGB(255, 0, 255))
                            XDrawPoint(DISPLAY, bmp->mask, gc, x, y);
                XFreeGC(DISPLAY, gc);
            } else {
                XFreePixmap(DISPLAY, bmp->mask);
                bmp->mask = 0;
            }
        }
    }
    XDestroyImage(img);
}

void ff_bitmap_size(ff_bitmap_t bmp, int* width, int* height)
{
    if (width) *width = bmp ? bmp->width : 0;
    if (height) *height = bmp ? bmp->height : 0;
}

ff_gc_t ff_bitmap_gc(ff_bitmap_t bmp)
{
    if (!bmp) return NULL;
    if (!bmp->gc) {
        bmp->gc = calloc(1, sizeof(struct _ff_gc_t));
        bmp->gc->gc = XCreateGC(DISPLAY, bmp->pixmap, 0, NULL);
        if (!bmp->gc->gc) {
            free(bmp->gc);
            return NULL;
        }
        bmp->gc->dx2 = bmp->width - 1;
        bmp->gc->dy2 = bmp->height - 1;
        setup_gc(bmp->gc, bmp->pixmap, NULL);
    }
    return bmp->gc;
}

static ff_font_t find_exact_font(const char* name, int size, unsigned flags)
{
    size_t i;
    flags &= ~FF_EXACT_FONT;
    for (i=0; i < font_count; i++)
        if ((!name || ff_streq_nocase(name, fonts[i]->name)) &&
            size == fonts[i]->size && flags == fonts[i]->flags)
            return fonts[i];
    return NULL;
}

ff_font_t ff_font_search(const char* name, int size, unsigned flags)
{
    ff_font_t font = find_exact_font(name, size, flags);
    if (font) return font;
    font = calloc(1, sizeof(struct _ff_font_t));
#ifdef LFORMS_USE_XFT
    /* This gigantic ugly tree of if-else is there because Xft doesn't
       provide a way to compose dynamically the attributes to pass to
       XftFontOpen :-( - chances are there will be bugs */
    if (name) {
        if (flags & FF_MONOSPACE) {
            if (size > 0) {
                font->font = XftFontOpen(DISPLAY, DefaultScreen(DISPLAY),
                                        XFT_FAMILY, XftTypeString, name,
                                        XFT_SLANT, XftTypeInteger, (flags & FF_ITALIC) ? XFT_SLANT_ITALIC : XFT_SLANT_ROMAN,
                                        XFT_WEIGHT, XftTypeInteger, (flags & FF_BOLD) ? XFT_WEIGHT_BOLD : XFT_WEIGHT_MEDIUM,
                                        XFT_SPACING, XftTypeInteger, XFT_MONO,
                                        XFT_PIXEL_SIZE, XftTypeDouble, (double)size,
                                        NULL);
                if (!font->font)
                    font->font = XftFontOpen(DISPLAY, DefaultScreen(DISPLAY),
                                            XFT_FAMILY, XftTypeString, name,
                                            XFT_SLANT, XftTypeInteger, (flags & FF_ITALIC) ? XFT_SLANT_OBLIQUE : XFT_SLANT_ROMAN,
                                            XFT_WEIGHT, XftTypeInteger, (flags & FF_BOLD) ? XFT_WEIGHT_BOLD : XFT_WEIGHT_MEDIUM,
                                            XFT_SPACING, XftTypeInteger, XFT_MONO,
                                            XFT_PIXEL_SIZE, XftTypeDouble, (double)size,
                                            NULL);
                if (!font->font && (flags & FF_EXACT_FONT)) {
                    free(font);
                    return NULL;
                }
                if (!font->font)
                    font->font = XftFontOpen(DISPLAY, DefaultScreen(DISPLAY),
                                            XFT_FAMILY, XftTypeString, name,
                                            XFT_WEIGHT, XftTypeInteger, (flags & FF_BOLD) ? XFT_WEIGHT_BOLD : XFT_WEIGHT_MEDIUM,
                                            XFT_SPACING, XftTypeInteger, XFT_MONO,
                                            XFT_PIXEL_SIZE, XftTypeDouble, (double)size,
                                            NULL);
                if (!font->font)
                    font->font = XftFontOpen(DISPLAY, DefaultScreen(DISPLAY),
                                            XFT_FAMILY, XftTypeString, name,
                                            XFT_SPACING, XftTypeInteger, XFT_MONO,
                                            XFT_PIXEL_SIZE, XftTypeDouble, (double)size,
                                            NULL);
                if (!font->font)
                    font->font = XftFontOpen(DISPLAY, DefaultScreen(DISPLAY),
                                            XFT_FAMILY, XftTypeString, name,
                                            XFT_SLANT, XftTypeInteger, (flags & FF_ITALIC) ? XFT_SLANT_ITALIC : XFT_SLANT_ROMAN,
                                            XFT_WEIGHT, XftTypeInteger, (flags & FF_BOLD) ? XFT_WEIGHT_BOLD : XFT_WEIGHT_MEDIUM,
                                            XFT_SPACING, XftTypeInteger, XFT_MONO,
                                            NULL);
                if (!font->font)
                    font->font = XftFontOpen(DISPLAY, DefaultScreen(DISPLAY),
                                            XFT_FAMILY, XftTypeString, name,
                                            XFT_SLANT, XftTypeInteger, (flags & FF_ITALIC) ? XFT_SLANT_OBLIQUE : XFT_SLANT_ROMAN,
                                            XFT_WEIGHT, XftTypeInteger, (flags & FF_BOLD) ? XFT_WEIGHT_BOLD : XFT_WEIGHT_MEDIUM,
                                            XFT_SPACING, XftTypeInteger, XFT_MONO,
                                            NULL);
                if (!font->font)
                    font->font = XftFontOpen(DISPLAY, DefaultScreen(DISPLAY),
                                            XFT_FAMILY, XftTypeString, name,
                                            XFT_WEIGHT, XftTypeInteger, (flags & FF_BOLD) ? XFT_WEIGHT_BOLD : XFT_WEIGHT_MEDIUM,
                                            XFT_SPACING, XftTypeInteger, XFT_MONO,
                                            NULL);
                if (!font->font)
                    font->font = XftFontOpen(DISPLAY, DefaultScreen(DISPLAY),
                                            XFT_FAMILY, XftTypeString, name,
                                            XFT_SPACING, XftTypeInteger, XFT_MONO,
                                            NULL);
            } else {
                font->font = XftFontOpen(DISPLAY, DefaultScreen(DISPLAY),
                                        XFT_FAMILY, XftTypeString, name,
                                        XFT_SLANT, XftTypeInteger, (flags & FF_ITALIC) ? XFT_SLANT_ITALIC : XFT_SLANT_ROMAN,
                                        XFT_WEIGHT, XftTypeInteger, (flags & FF_BOLD) ? XFT_WEIGHT_BOLD : XFT_WEIGHT_MEDIUM,
                                        XFT_SPACING, XftTypeInteger, XFT_MONO,
                                        NULL);
                if (!font->font)
                    font->font = XftFontOpen(DISPLAY, DefaultScreen(DISPLAY),
                                            XFT_FAMILY, XftTypeString, name,
                                            XFT_SLANT, XftTypeInteger, (flags & FF_ITALIC) ? XFT_SLANT_OBLIQUE : XFT_SLANT_ROMAN,
                                            XFT_WEIGHT, XftTypeInteger, (flags & FF_BOLD) ? XFT_WEIGHT_BOLD : XFT_WEIGHT_MEDIUM,
                                            XFT_SPACING, XftTypeInteger, XFT_MONO,
                                            NULL);
                if (!font->font && (flags & FF_EXACT_FONT)) {
                    free(font);
                    return NULL;
                }
                if (!font->font)
                    font->font = XftFontOpen(DISPLAY, DefaultScreen(DISPLAY),
                                            XFT_FAMILY, XftTypeString, name,
                                            XFT_WEIGHT, XftTypeInteger, (flags & FF_BOLD) ? XFT_WEIGHT_BOLD : XFT_WEIGHT_MEDIUM,
                                            XFT_SPACING, XftTypeInteger, XFT_MONO,
                                            NULL);
                if (!font->font)
                    font->font = XftFontOpen(DISPLAY, DefaultScreen(DISPLAY),
                                            XFT_FAMILY, XftTypeString, name,
                                            XFT_SPACING, XftTypeInteger, XFT_MONO,
                                            NULL);
            }
        } else if (flags & FF_PROPORTIONAL) {
            if (size > 0) {
                font->font = XftFontOpen(DISPLAY, DefaultScreen(DISPLAY),
                                        XFT_FAMILY, XftTypeString, name,
                                        XFT_SLANT, XftTypeInteger, (flags & FF_ITALIC) ? XFT_SLANT_ITALIC : XFT_SLANT_ROMAN,
                                        XFT_WEIGHT, XftTypeInteger, (flags & FF_BOLD) ? XFT_WEIGHT_BOLD : XFT_WEIGHT_MEDIUM,
                                        XFT_SPACING, XftTypeInteger, XFT_PROPORTIONAL,
                                        XFT_PIXEL_SIZE, XftTypeDouble, (double)size,
                                        NULL);
                if (!font->font)
                    font->font = XftFontOpen(DISPLAY, DefaultScreen(DISPLAY),
                                            XFT_FAMILY, XftTypeString, name,
                                            XFT_SLANT, XftTypeInteger, (flags & FF_ITALIC) ? XFT_SLANT_OBLIQUE : XFT_SLANT_ROMAN,
                                            XFT_WEIGHT, XftTypeInteger, (flags & FF_BOLD) ? XFT_WEIGHT_BOLD : XFT_WEIGHT_MEDIUM,
                                            XFT_SPACING, XftTypeInteger, XFT_PROPORTIONAL,
                                            XFT_PIXEL_SIZE, XftTypeDouble, (double)size,
                                            NULL);
                if (!font->font && (flags & FF_EXACT_FONT)) {
                    free(font);
                    return NULL;
                }
                if (!font->font)
                    font->font = XftFontOpen(DISPLAY, DefaultScreen(DISPLAY),
                                            XFT_FAMILY, XftTypeString, name,
                                            XFT_WEIGHT, XftTypeInteger, (flags & FF_BOLD) ? XFT_WEIGHT_BOLD : XFT_WEIGHT_MEDIUM,
                                            XFT_SPACING, XftTypeInteger, XFT_PROPORTIONAL,
                                            XFT_PIXEL_SIZE, XftTypeDouble, (double)size,
                                            NULL);
                if (!font->font)
                    font->font = XftFontOpen(DISPLAY, DefaultScreen(DISPLAY),
                                            XFT_FAMILY, XftTypeString, name,
                                            XFT_SPACING, XftTypeInteger, XFT_PROPORTIONAL,
                                            XFT_PIXEL_SIZE, XftTypeDouble, (double)size,
                                            NULL);
                if (!font->font)
                    font->font = XftFontOpen(DISPLAY, DefaultScreen(DISPLAY),
                                            XFT_FAMILY, XftTypeString, name,
                                            XFT_SLANT, XftTypeInteger, (flags & FF_ITALIC) ? XFT_SLANT_ITALIC : XFT_SLANT_ROMAN,
                                            XFT_WEIGHT, XftTypeInteger, (flags & FF_BOLD) ? XFT_WEIGHT_BOLD : XFT_WEIGHT_MEDIUM,
                                            XFT_SPACING, XftTypeInteger, XFT_PROPORTIONAL,
                                            NULL);
                if (!font->font)
                    font->font = XftFontOpen(DISPLAY, DefaultScreen(DISPLAY),
                                            XFT_FAMILY, XftTypeString, name,
                                            XFT_SLANT, XftTypeInteger, (flags & FF_ITALIC) ? XFT_SLANT_OBLIQUE : XFT_SLANT_ROMAN,
                                            XFT_WEIGHT, XftTypeInteger, (flags & FF_BOLD) ? XFT_WEIGHT_BOLD : XFT_WEIGHT_MEDIUM,
                                            XFT_SPACING, XftTypeInteger, XFT_PROPORTIONAL,
                                            NULL);
                if (!font->font)
                    font->font = XftFontOpen(DISPLAY, DefaultScreen(DISPLAY),
                                            XFT_FAMILY, XftTypeString, name,
                                            XFT_WEIGHT, XftTypeInteger, (flags & FF_BOLD) ? XFT_WEIGHT_BOLD : XFT_WEIGHT_MEDIUM,
                                            XFT_SPACING, XftTypeInteger, XFT_PROPORTIONAL,
                                            NULL);
                if (!font->font)
                    font->font = XftFontOpen(DISPLAY, DefaultScreen(DISPLAY),
                                            XFT_FAMILY, XftTypeString, name,
                                            XFT_SPACING, XftTypeInteger, XFT_PROPORTIONAL,
                                            NULL);
            } else {
                font->font = XftFontOpen(DISPLAY, DefaultScreen(DISPLAY),
                                        XFT_FAMILY, XftTypeString, name,
                                        XFT_SLANT, XftTypeInteger, (flags & FF_ITALIC) ? XFT_SLANT_ITALIC : XFT_SLANT_ROMAN,
                                        XFT_WEIGHT, XftTypeInteger, (flags & FF_BOLD) ? XFT_WEIGHT_BOLD : XFT_WEIGHT_MEDIUM,
                                        XFT_SPACING, XftTypeInteger, XFT_PROPORTIONAL,
                                        NULL);
                if (!font->font)
                    font->font = XftFontOpen(DISPLAY, DefaultScreen(DISPLAY),
                                            XFT_FAMILY, XftTypeString, name,
                                            XFT_SLANT, XftTypeInteger, (flags & FF_ITALIC) ? XFT_SLANT_OBLIQUE : XFT_SLANT_ROMAN,
                                            XFT_WEIGHT, XftTypeInteger, (flags & FF_BOLD) ? XFT_WEIGHT_BOLD : XFT_WEIGHT_MEDIUM,
                                            XFT_SPACING, XftTypeInteger, XFT_PROPORTIONAL,
                                            NULL);
                if (!font->font && (flags & FF_EXACT_FONT)) {
                    free(font);
                    return NULL;
                }
                if (!font->font)
                    font->font = XftFontOpen(DISPLAY, DefaultScreen(DISPLAY),
                                            XFT_FAMILY, XftTypeString, name,
                                            XFT_WEIGHT, XftTypeInteger, (flags & FF_BOLD) ? XFT_WEIGHT_BOLD : XFT_WEIGHT_MEDIUM,
                                            XFT_SPACING, XftTypeInteger, XFT_PROPORTIONAL,
                                            NULL);
                if (!font->font)
                    font->font = XftFontOpen(DISPLAY, DefaultScreen(DISPLAY),
                                            XFT_FAMILY, XftTypeString, name,
                                            XFT_SPACING, XftTypeInteger, XFT_PROPORTIONAL,
                                            NULL);
            }
        } else {
            if (size > 0) {
                font->font = XftFontOpen(DISPLAY, DefaultScreen(DISPLAY),
                                        XFT_FAMILY, XftTypeString, name,
                                        XFT_SLANT, XftTypeInteger, (flags & FF_ITALIC) ? XFT_SLANT_ITALIC : XFT_SLANT_ROMAN,
                                        XFT_WEIGHT, XftTypeInteger, (flags & FF_BOLD) ? XFT_WEIGHT_BOLD : XFT_WEIGHT_MEDIUM,
                                        XFT_PIXEL_SIZE, XftTypeDouble, (double)size,
                                        NULL);
                if (!font->font)
                    font->font = XftFontOpen(DISPLAY, DefaultScreen(DISPLAY),
                                            XFT_FAMILY, XftTypeString, name,
                                            XFT_SLANT, XftTypeInteger, (flags & FF_ITALIC) ? XFT_SLANT_OBLIQUE : XFT_SLANT_ROMAN,
                                            XFT_WEIGHT, XftTypeInteger, (flags & FF_BOLD) ? XFT_WEIGHT_BOLD : XFT_WEIGHT_MEDIUM,
                                            XFT_PIXEL_SIZE, XftTypeDouble, (double)size,
                                            NULL);
                if (!font->font && (flags & FF_EXACT_FONT)) {
                    free(font);
                    return NULL;
                }
                if (!font->font)
                    font->font = XftFontOpen(DISPLAY, DefaultScreen(DISPLAY),
                                            XFT_FAMILY, XftTypeString, name,
                                            XFT_SLANT, XftTypeInteger, (flags & FF_ITALIC) ? XFT_SLANT_OBLIQUE : XFT_SLANT_ROMAN,
                                            XFT_WEIGHT, XftTypeInteger, (flags & FF_BOLD) ? XFT_WEIGHT_BOLD : XFT_WEIGHT_MEDIUM,
                                            XFT_PIXEL_SIZE, XftTypeDouble, (double)size,
                                            NULL);
                if (!font->font)
                    font->font = XftFontOpen(DISPLAY, DefaultScreen(DISPLAY),
                                            XFT_FAMILY, XftTypeString, name,
                                            XFT_WEIGHT, XftTypeInteger, (flags & FF_BOLD) ? XFT_WEIGHT_BOLD : XFT_WEIGHT_MEDIUM,
                                            XFT_PIXEL_SIZE, XftTypeDouble, (double)size,
                                            NULL);
                if (!font->font)
                    font->font = XftFontOpen(DISPLAY, DefaultScreen(DISPLAY),
                                            XFT_FAMILY, XftTypeString, name,
                                            XFT_PIXEL_SIZE, XftTypeDouble, (double)size,
                                            NULL);
                if (!font->font)
                    font->font = XftFontOpen(DISPLAY, DefaultScreen(DISPLAY),
                                            XFT_FAMILY, XftTypeString, name,
                                            XFT_SLANT, XftTypeInteger, (flags & FF_ITALIC) ? XFT_SLANT_ITALIC : XFT_SLANT_ROMAN,
                                            XFT_WEIGHT, XftTypeInteger, (flags & FF_BOLD) ? XFT_WEIGHT_BOLD : XFT_WEIGHT_MEDIUM,
                                            NULL);
                if (!font->font)
                    font->font = XftFontOpen(DISPLAY, DefaultScreen(DISPLAY),
                                            XFT_FAMILY, XftTypeString, name,
                                            XFT_SLANT, XftTypeInteger, (flags & FF_ITALIC) ? XFT_SLANT_OBLIQUE : XFT_SLANT_ROMAN,
                                            XFT_WEIGHT, XftTypeInteger, (flags & FF_BOLD) ? XFT_WEIGHT_BOLD : XFT_WEIGHT_MEDIUM,
                                            NULL);
                if (!font->font)
                    font->font = XftFontOpen(DISPLAY, DefaultScreen(DISPLAY),
                                            XFT_FAMILY, XftTypeString, name,
                                            XFT_WEIGHT, XftTypeInteger, (flags & FF_BOLD) ? XFT_WEIGHT_BOLD : XFT_WEIGHT_MEDIUM,
                                            NULL);
                if (!font->font)
                    font->font = XftFontOpen(DISPLAY, DefaultScreen(DISPLAY),
                                            XFT_FAMILY, XftTypeString, name,
                                            NULL);
            } else {
                font->font = XftFontOpen(DISPLAY, DefaultScreen(DISPLAY),
                                        XFT_FAMILY, XftTypeString, name,
                                        XFT_SLANT, XftTypeInteger, (flags & FF_ITALIC) ? XFT_SLANT_ITALIC : XFT_SLANT_ROMAN,
                                        XFT_WEIGHT, XftTypeInteger, (flags & FF_BOLD) ? XFT_WEIGHT_BOLD : XFT_WEIGHT_MEDIUM,
                                        NULL);
                if (!font->font)
                    font->font = XftFontOpen(DISPLAY, DefaultScreen(DISPLAY),
                                            XFT_FAMILY, XftTypeString, name,
                                            XFT_SLANT, XftTypeInteger, (flags & FF_ITALIC) ? XFT_SLANT_OBLIQUE : XFT_SLANT_ROMAN,
                                            XFT_WEIGHT, XftTypeInteger, (flags & FF_BOLD) ? XFT_WEIGHT_BOLD : XFT_WEIGHT_MEDIUM,
                                            NULL);
                if (!font->font && (flags & FF_EXACT_FONT)) {
                    free(font);
                    return NULL;
                }
                if (!font->font)
                    font->font = XftFontOpen(DISPLAY, DefaultScreen(DISPLAY),
                                            XFT_FAMILY, XftTypeString, name,
                                            XFT_SLANT, XftTypeInteger, (flags & FF_ITALIC) ? XFT_SLANT_OBLIQUE : XFT_SLANT_ROMAN,
                                            XFT_WEIGHT, XftTypeInteger, (flags & FF_BOLD) ? XFT_WEIGHT_BOLD : XFT_WEIGHT_MEDIUM,
                                            NULL);
                if (!font->font)
                    font->font = XftFontOpen(DISPLAY, DefaultScreen(DISPLAY),
                                            XFT_FAMILY, XftTypeString, name,
                                            XFT_WEIGHT, XftTypeInteger, (flags & FF_BOLD) ? XFT_WEIGHT_BOLD : XFT_WEIGHT_MEDIUM,
                                            NULL);
                if (!font->font)
                    font->font = XftFontOpen(DISPLAY, DefaultScreen(DISPLAY),
                                            XFT_FAMILY, XftTypeString, name,
                                            NULL);
            }
        }
    }
    if (!name || !font->font) {
        if (flags & FF_MONOSPACE) {
            if (size > 0) {
                font->font = XftFontOpen(DISPLAY, DefaultScreen(DISPLAY),
                                        XFT_SLANT, XftTypeInteger, (flags & FF_ITALIC) ? XFT_SLANT_ITALIC : XFT_SLANT_ROMAN,
                                        XFT_WEIGHT, XftTypeInteger, (flags & FF_BOLD) ? XFT_WEIGHT_BOLD : XFT_WEIGHT_MEDIUM,
                                        XFT_SPACING, XftTypeInteger, XFT_MONO,
                                        XFT_PIXEL_SIZE, XftTypeDouble, (double)size,
                                        NULL);
                if (!font->font)
                    font->font = XftFontOpen(DISPLAY, DefaultScreen(DISPLAY),
                                            XFT_SLANT, XftTypeInteger, (flags & FF_ITALIC) ? XFT_SLANT_OBLIQUE : XFT_SLANT_ROMAN,
                                            XFT_WEIGHT, XftTypeInteger, (flags & FF_BOLD) ? XFT_WEIGHT_BOLD : XFT_WEIGHT_MEDIUM,
                                            XFT_SPACING, XftTypeInteger, XFT_MONO,
                                            XFT_PIXEL_SIZE, XftTypeDouble, (double)size,
                                            NULL);
                if (!font->font && (flags & FF_EXACT_FONT)) {
                    free(font);
                    return NULL;
                }
                if (!font->font)
                    font->font = XftFontOpen(DISPLAY, DefaultScreen(DISPLAY),
                                            XFT_WEIGHT, XftTypeInteger, (flags & FF_BOLD) ? XFT_WEIGHT_BOLD : XFT_WEIGHT_MEDIUM,
                                            XFT_SPACING, XftTypeInteger, XFT_MONO,
                                            XFT_PIXEL_SIZE, XftTypeDouble, (double)size,
                                            NULL);
                if (!font->font)
                    font->font = XftFontOpen(DISPLAY, DefaultScreen(DISPLAY),
                                            XFT_SPACING, XftTypeInteger, XFT_MONO,
                                            XFT_PIXEL_SIZE, XftTypeDouble, (double)size,
                                            NULL);
                if (!font->font)
                    font->font = XftFontOpen(DISPLAY, DefaultScreen(DISPLAY),
                                            XFT_SLANT, XftTypeInteger, (flags & FF_ITALIC) ? XFT_SLANT_ITALIC : XFT_SLANT_ROMAN,
                                            XFT_WEIGHT, XftTypeInteger, (flags & FF_BOLD) ? XFT_WEIGHT_BOLD : XFT_WEIGHT_MEDIUM,
                                            XFT_SPACING, XftTypeInteger, XFT_MONO,
                                            NULL);
                if (!font->font)
                    font->font = XftFontOpen(DISPLAY, DefaultScreen(DISPLAY),
                                            XFT_SLANT, XftTypeInteger, (flags & FF_ITALIC) ? XFT_SLANT_OBLIQUE : XFT_SLANT_ROMAN,
                                            XFT_WEIGHT, XftTypeInteger, (flags & FF_BOLD) ? XFT_WEIGHT_BOLD : XFT_WEIGHT_MEDIUM,
                                            XFT_SPACING, XftTypeInteger, XFT_MONO,
                                            NULL);
                if (!font->font)
                    font->font = XftFontOpen(DISPLAY, DefaultScreen(DISPLAY),
                                            XFT_WEIGHT, XftTypeInteger, (flags & FF_BOLD) ? XFT_WEIGHT_BOLD : XFT_WEIGHT_MEDIUM,
                                            XFT_SPACING, XftTypeInteger, XFT_MONO,
                                            NULL);
                if (!font->font)
                    font->font = XftFontOpen(DISPLAY, DefaultScreen(DISPLAY),
                                            XFT_SPACING, XftTypeInteger, XFT_MONO,
                                            NULL);
            } else {
                font->font = XftFontOpen(DISPLAY, DefaultScreen(DISPLAY),
                                        XFT_SLANT, XftTypeInteger, (flags & FF_ITALIC) ? XFT_SLANT_ITALIC : XFT_SLANT_ROMAN,
                                        XFT_WEIGHT, XftTypeInteger, (flags & FF_BOLD) ? XFT_WEIGHT_BOLD : XFT_WEIGHT_MEDIUM,
                                        XFT_SPACING, XftTypeInteger, XFT_MONO,
                                        NULL);
                if (!font->font)
                    font->font = XftFontOpen(DISPLAY, DefaultScreen(DISPLAY),
                                            XFT_SLANT, XftTypeInteger, (flags & FF_ITALIC) ? XFT_SLANT_OBLIQUE : XFT_SLANT_ROMAN,
                                            XFT_WEIGHT, XftTypeInteger, (flags & FF_BOLD) ? XFT_WEIGHT_BOLD : XFT_WEIGHT_MEDIUM,
                                            XFT_SPACING, XftTypeInteger, XFT_MONO,
                                            NULL);
                if (!font->font && (flags & FF_EXACT_FONT)) {
                    free(font);
                    return NULL;
                }
                if (!font->font)
                    font->font = XftFontOpen(DISPLAY, DefaultScreen(DISPLAY),
                                            XFT_WEIGHT, XftTypeInteger, (flags & FF_BOLD) ? XFT_WEIGHT_BOLD : XFT_WEIGHT_MEDIUM,
                                            XFT_SPACING, XftTypeInteger, XFT_MONO,
                                            NULL);
                if (!font->font)
                    font->font = XftFontOpen(DISPLAY, DefaultScreen(DISPLAY),
                                            XFT_SPACING, XftTypeInteger, XFT_MONO,
                                            NULL);
            }
        } else if (flags & FF_PROPORTIONAL) {
            if (size > 0) {
                font->font = XftFontOpen(DISPLAY, DefaultScreen(DISPLAY),
                                        XFT_SLANT, XftTypeInteger, (flags & FF_ITALIC) ? XFT_SLANT_ITALIC : XFT_SLANT_ROMAN,
                                        XFT_WEIGHT, XftTypeInteger, (flags & FF_BOLD) ? XFT_WEIGHT_BOLD : XFT_WEIGHT_MEDIUM,
                                        XFT_SPACING, XftTypeInteger, XFT_PROPORTIONAL,
                                        XFT_PIXEL_SIZE, XftTypeDouble, (double)size,
                                        NULL);
                if (!font->font)
                    font->font = XftFontOpen(DISPLAY, DefaultScreen(DISPLAY),
                                            XFT_SLANT, XftTypeInteger, (flags & FF_ITALIC) ? XFT_SLANT_OBLIQUE : XFT_SLANT_ROMAN,
                                            XFT_WEIGHT, XftTypeInteger, (flags & FF_BOLD) ? XFT_WEIGHT_BOLD : XFT_WEIGHT_MEDIUM,
                                            XFT_SPACING, XftTypeInteger, XFT_PROPORTIONAL,
                                            XFT_PIXEL_SIZE, XftTypeDouble, (double)size,
                                            NULL);
                if (!font->font && (flags & FF_EXACT_FONT)) {
                    free(font);
                    return NULL;
                }
                if (!font->font)
                    font->font = XftFontOpen(DISPLAY, DefaultScreen(DISPLAY),
                                            XFT_WEIGHT, XftTypeInteger, (flags & FF_BOLD) ? XFT_WEIGHT_BOLD : XFT_WEIGHT_MEDIUM,
                                            XFT_SPACING, XftTypeInteger, XFT_PROPORTIONAL,
                                            XFT_PIXEL_SIZE, XftTypeDouble, (double)size,
                                            NULL);
                if (!font->font)
                    font->font = XftFontOpen(DISPLAY, DefaultScreen(DISPLAY),
                                            XFT_SPACING, XftTypeInteger, XFT_PROPORTIONAL,
                                            XFT_PIXEL_SIZE, XftTypeDouble, (double)size,
                                            NULL);
                if (!font->font)
                    font->font = XftFontOpen(DISPLAY, DefaultScreen(DISPLAY),
                                            XFT_SLANT, XftTypeInteger, (flags & FF_ITALIC) ? XFT_SLANT_ITALIC : XFT_SLANT_ROMAN,
                                            XFT_WEIGHT, XftTypeInteger, (flags & FF_BOLD) ? XFT_WEIGHT_BOLD : XFT_WEIGHT_MEDIUM,
                                            XFT_SPACING, XftTypeInteger, XFT_PROPORTIONAL,
                                            NULL);
                if (!font->font)
                    font->font = XftFontOpen(DISPLAY, DefaultScreen(DISPLAY),
                                            XFT_SLANT, XftTypeInteger, (flags & FF_ITALIC) ? XFT_SLANT_OBLIQUE : XFT_SLANT_ROMAN,
                                            XFT_WEIGHT, XftTypeInteger, (flags & FF_BOLD) ? XFT_WEIGHT_BOLD : XFT_WEIGHT_MEDIUM,
                                            XFT_SPACING, XftTypeInteger, XFT_PROPORTIONAL,
                                            NULL);
                if (!font->font)
                    font->font = XftFontOpen(DISPLAY, DefaultScreen(DISPLAY),
                                            XFT_WEIGHT, XftTypeInteger, (flags & FF_BOLD) ? XFT_WEIGHT_BOLD : XFT_WEIGHT_MEDIUM,
                                            XFT_SPACING, XftTypeInteger, XFT_PROPORTIONAL,
                                            NULL);
                if (!font->font)
                    font->font = XftFontOpen(DISPLAY, DefaultScreen(DISPLAY),
                                            XFT_SPACING, XftTypeInteger, XFT_PROPORTIONAL,
                                            NULL);
            } else {
                font->font = XftFontOpen(DISPLAY, DefaultScreen(DISPLAY),
                                        XFT_SLANT, XftTypeInteger, (flags & FF_ITALIC) ? XFT_SLANT_ITALIC : XFT_SLANT_ROMAN,
                                        XFT_WEIGHT, XftTypeInteger, (flags & FF_BOLD) ? XFT_WEIGHT_BOLD : XFT_WEIGHT_MEDIUM,
                                        XFT_SPACING, XftTypeInteger, XFT_PROPORTIONAL,
                                        NULL);
                if (!font->font)
                    font->font = XftFontOpen(DISPLAY, DefaultScreen(DISPLAY),
                                            XFT_SLANT, XftTypeInteger, (flags & FF_ITALIC) ? XFT_SLANT_OBLIQUE : XFT_SLANT_ROMAN,
                                            XFT_WEIGHT, XftTypeInteger, (flags & FF_BOLD) ? XFT_WEIGHT_BOLD : XFT_WEIGHT_MEDIUM,
                                            XFT_SPACING, XftTypeInteger, XFT_PROPORTIONAL,
                                            NULL);
                if (!font->font && (flags & FF_EXACT_FONT)) {
                    free(font);
                    return NULL;
                }
                if (!font->font)
                    font->font = XftFontOpen(DISPLAY, DefaultScreen(DISPLAY),
                                            XFT_WEIGHT, XftTypeInteger, (flags & FF_BOLD) ? XFT_WEIGHT_BOLD : XFT_WEIGHT_MEDIUM,
                                            XFT_SPACING, XftTypeInteger, XFT_PROPORTIONAL,
                                            NULL);
                if (!font->font)
                    font->font = XftFontOpen(DISPLAY, DefaultScreen(DISPLAY),
                                            XFT_SPACING, XftTypeInteger, XFT_PROPORTIONAL,
                                            NULL);
            }
        } else {
            if (size > 0) {
                font->font = XftFontOpen(DISPLAY, DefaultScreen(DISPLAY),
                                        XFT_SLANT, XftTypeInteger, (flags & FF_ITALIC) ? XFT_SLANT_ITALIC : XFT_SLANT_ROMAN,
                                        XFT_WEIGHT, XftTypeInteger, (flags & FF_BOLD) ? XFT_WEIGHT_BOLD : XFT_WEIGHT_MEDIUM,
                                        XFT_PIXEL_SIZE, XftTypeDouble, (double)size,
                                        NULL);
                if (!font->font)
                    font->font = XftFontOpen(DISPLAY, DefaultScreen(DISPLAY),
                                            XFT_SLANT, XftTypeInteger, (flags & FF_ITALIC) ? XFT_SLANT_OBLIQUE : XFT_SLANT_ROMAN,
                                            XFT_WEIGHT, XftTypeInteger, (flags & FF_BOLD) ? XFT_WEIGHT_BOLD : XFT_WEIGHT_MEDIUM,
                                            XFT_PIXEL_SIZE, XftTypeDouble, (double)size,
                                            NULL);
                if (!font->font && (flags & FF_EXACT_FONT)) {
                    free(font);
                    return NULL;
                }
                if (!font->font)
                    font->font = XftFontOpen(DISPLAY, DefaultScreen(DISPLAY),
                                            XFT_SLANT, XftTypeInteger, (flags & FF_ITALIC) ? XFT_SLANT_OBLIQUE : XFT_SLANT_ROMAN,
                                            XFT_WEIGHT, XftTypeInteger, (flags & FF_BOLD) ? XFT_WEIGHT_BOLD : XFT_WEIGHT_MEDIUM,
                                            XFT_PIXEL_SIZE, XftTypeDouble, (double)size,
                                            NULL);
                if (!font->font)
                    font->font = XftFontOpen(DISPLAY, DefaultScreen(DISPLAY),
                                            XFT_WEIGHT, XftTypeInteger, (flags & FF_BOLD) ? XFT_WEIGHT_BOLD : XFT_WEIGHT_MEDIUM,
                                            XFT_PIXEL_SIZE, XftTypeDouble, (double)size,
                                            NULL);
                if (!font->font)
                    font->font = XftFontOpen(DISPLAY, DefaultScreen(DISPLAY),
                                            XFT_PIXEL_SIZE, XftTypeDouble, (double)size,
                                            NULL);
                if (!font->font)
                    font->font = XftFontOpen(DISPLAY, DefaultScreen(DISPLAY),
                                            XFT_SLANT, XftTypeInteger, (flags & FF_ITALIC) ? XFT_SLANT_ITALIC : XFT_SLANT_ROMAN,
                                            XFT_WEIGHT, XftTypeInteger, (flags & FF_BOLD) ? XFT_WEIGHT_BOLD : XFT_WEIGHT_MEDIUM,
                                            NULL);
                if (!font->font)
                    font->font = XftFontOpen(DISPLAY, DefaultScreen(DISPLAY),
                                            XFT_SLANT, XftTypeInteger, (flags & FF_ITALIC) ? XFT_SLANT_OBLIQUE : XFT_SLANT_ROMAN,
                                            XFT_WEIGHT, XftTypeInteger, (flags & FF_BOLD) ? XFT_WEIGHT_BOLD : XFT_WEIGHT_MEDIUM,
                                            NULL);
                if (!font->font)
                    font->font = XftFontOpen(DISPLAY, DefaultScreen(DISPLAY),
                                            XFT_WEIGHT, XftTypeInteger, (flags & FF_BOLD) ? XFT_WEIGHT_BOLD : XFT_WEIGHT_MEDIUM,
                                            NULL);
            } else {
                font->font = XftFontOpen(DISPLAY, DefaultScreen(DISPLAY),
                                        XFT_SLANT, XftTypeInteger, (flags & FF_ITALIC) ? XFT_SLANT_ITALIC : XFT_SLANT_ROMAN,
                                        XFT_WEIGHT, XftTypeInteger, (flags & FF_BOLD) ? XFT_WEIGHT_BOLD : XFT_WEIGHT_MEDIUM,
                                        NULL);
                if (!font->font)
                    font->font = XftFontOpen(DISPLAY, DefaultScreen(DISPLAY),
                                            XFT_SLANT, XftTypeInteger, (flags & FF_ITALIC) ? XFT_SLANT_OBLIQUE : XFT_SLANT_ROMAN,
                                            XFT_WEIGHT, XftTypeInteger, (flags & FF_BOLD) ? XFT_WEIGHT_BOLD : XFT_WEIGHT_MEDIUM,
                                            NULL);
                if (!font->font && (flags & FF_EXACT_FONT)) {
                    free(font);
                    return NULL;
                }
                if (!font->font)
                    font->font = XftFontOpen(DISPLAY, DefaultScreen(DISPLAY),
                                            XFT_SLANT, XftTypeInteger, (flags & FF_ITALIC) ? XFT_SLANT_OBLIQUE : XFT_SLANT_ROMAN,
                                            XFT_WEIGHT, XftTypeInteger, (flags & FF_BOLD) ? XFT_WEIGHT_BOLD : XFT_WEIGHT_MEDIUM,
                                            NULL);
                if (!font->font)
                    font->font = XftFontOpen(DISPLAY, DefaultScreen(DISPLAY),
                                            XFT_WEIGHT, XftTypeInteger, (flags & FF_BOLD) ? XFT_WEIGHT_BOLD : XFT_WEIGHT_MEDIUM,
                                            NULL);
            }
        }
    }
    if (!font->font) {
        if (flags & FF_EXACT_FONT) {
            free(font);
            return NULL;
        }
        if (flags & FF_MONOSPACE) {
            font->font = XftFontOpen(DISPLAY, DefaultScreen(DISPLAY),
                                     XFT_SPACING, XftTypeInteger, XFT_MONO,
                                     NULL);
        } else if (flags & FF_PROPORTIONAL) {
            font->font = XftFontOpen(DISPLAY, DefaultScreen(DISPLAY),
                                     XFT_SPACING, XftTypeInteger, XFT_PROPORTIONAL,
                                     NULL);
        }
    }
    if (!font->font)
        font->font = XftFontOpenName(DISPLAY, 0, "");
    if (!font->font)
        fprintf(stderr, "Little Forms: failed to find any fallback Xft font, this will most likely crash\n");
#else
    {
        char* pattern = malloc(strlen(name) + 256);
        const char* weight = "medium";
        char slant = 'r';
        char spacing = '*';
        char sizestr[32];
        if (size > 0) sprintf(sizestr, "%i", size); else strcpy(sizestr, "*");
        if (flags & FF_BOLD) weight = "bold";
        if (flags & FF_ITALIC) slant = 'i';
        if (flags & FF_PROPORTIONAL) spacing = 'p';
        else if (flags & FF_MONOSPACE) spacing = 'm';
        sprintf(pattern, "-*-%s-%s-%c-*-*-%s-*-*-*-%c-*-iso8859-1",
            name ? name : "*",
            weight,
            slant,
            sizestr,
            spacing);
        font->font = XLoadQueryFont(DISPLAY, pattern);
        if (!font->font && slant == 'i') {
            sprintf(pattern, "-*-%s-%s-o-*-*-%s-*-*-*-%c-*-iso8859-1",
                name ? name : "*",
                weight,
                sizestr,
                spacing);
            font->font = XLoadQueryFont(DISPLAY, pattern);
        }
        if (!font->font && !(flags & FF_BOLD)) {
            sprintf(pattern, "-*-%s-normal-%c-*-*-%s-*-*-*-%c-*-iso8859-1",
                name ? name : "*",
                slant,
                sizestr,
                spacing);
            font->font = XLoadQueryFont(DISPLAY, pattern);
            if (!font->font && slant == 'i') {
                sprintf(pattern, "-*-%s-normal-o-*-*-%s-*-*-*-%c-*-iso8859-1",
                    name ? name : "*",
                    sizestr,
                    spacing);
                font->font = XLoadQueryFont(DISPLAY, pattern);
            }
            if (!font->font) {
                sprintf(pattern, "-*-%s-regular-%c-*-*-%s-*-*-*-%c-*-iso8859-1",
                    name ? name : "*",
                    slant,
                    sizestr,
                    spacing);
                font->font = XLoadQueryFont(DISPLAY, pattern);
                if (!font->font && slant == 'i') {
                    sprintf(pattern, "-*-%s-regular-o-*-*-%s-*-*-*-%c-*-iso8859-1",
                        name ? name : "*",
                        sizestr,
                        spacing);
                    font->font = XLoadQueryFont(DISPLAY, pattern);
                }
            }
            if (!font->font) {
                sprintf(pattern, "-*-%s-light-%c-*-*-%s-*-*-*-%c-*-iso8859-1",
                    name ? name : "*",
                    slant,
                    sizestr,
                    spacing);
                font->font = XLoadQueryFont(DISPLAY, pattern);
                if (!font->font && slant == 'i') {
                    sprintf(pattern, "-*-%s-light-o-*-*-%s-*-*-*-%c-*-iso8859-1",
                        name ? name : "*",
                        sizestr,
                        spacing);
                    font->font = XLoadQueryFont(DISPLAY, pattern);
                }
            }
            if (!font->font) {
                sprintf(pattern, "-*-%s-book-%c-*-*-%s-*-*-*-%c-*-iso8859-1",
                    name ? name : "*",
                    slant,
                    sizestr,
                    spacing);
                font->font = XLoadQueryFont(DISPLAY, pattern);
                if (!font->font && slant == 'i') {
                    sprintf(pattern, "-*-%s-book-o-*-*-%s-*-*-*-%c-*-iso8859-1",
                        name ? name : "*",
                        sizestr,
                        spacing);
                    font->font = XLoadQueryFont(DISPLAY, pattern);
                }
            }
        }
        if (!font->font && (flags & FF_BOLD)) {
            sprintf(pattern, "-*-%s-demi-%c-*-*-%s-*-*-*-%c-*-iso8859-1",
                name ? name : "*",
                slant,
                sizestr,
                spacing);
            font->font = XLoadQueryFont(DISPLAY, pattern);
            if (!font->font && slant == 'i') {
                sprintf(pattern, "-*-%s-demi-o-*-*-%s-*-*-*-%c-*-iso8859-1",
                    name ? name : "*",
                    sizestr,
                    spacing);
                font->font = XLoadQueryFont(DISPLAY, pattern);
            }
            if (!font->font) {
                sprintf(pattern, "-*-%s-demi bold-%c-*-*-%s-*-*-*-%c-*-iso8859-1",
                    name ? name : "*",
                    slant,
                    sizestr,
                    spacing);
                font->font = XLoadQueryFont(DISPLAY, pattern);
                if (!font->font && slant == 'i') {
                    sprintf(pattern, "-*-%s-demi bold-o-*-*-%s-*-*-*-%c-*-iso8859-1",
                        name ? name : "*",
                        sizestr,
                        spacing);
                    font->font = XLoadQueryFont(DISPLAY, pattern);
                }
            }
            if (!font->font) {
                sprintf(pattern, "-*-%s-demibold-%c-*-*-%s-*-*-*-%c-*-iso8859-1",
                    name ? name : "*",
                    slant,
                    sizestr,
                    spacing);
                font->font = XLoadQueryFont(DISPLAY, pattern);
                if (!font->font && slant == 'i') {
                    sprintf(pattern, "-*-%s-demibold-o-*-*-%s-*-*-*-%c-*-iso8859-1",
                        name ? name : "*",
                        sizestr,
                        spacing);
                    font->font = XLoadQueryFont(DISPLAY, pattern);
                }
            }
        }
        if (!font->font && (flags & FF_EXACT_FONT)) {
            free(pattern);
            free(font);
            return NULL;
        }
        if (!font->font) {
            sprintf(pattern, "-*-%s-%s-*-*-*-%s-*-*-*-%c-*-iso8859-1",
                name ? name : "*",
                weight,
                sizestr,
                spacing);
            font->font = XLoadQueryFont(DISPLAY, pattern);
        }
        if (!font->font) {
            sprintf(pattern, "-*-%s-*-*-*-*-%s-*-*-*-%c-*-iso8859-1",
                name ? name : "*",
                sizestr,
                spacing);
            font->font = XLoadQueryFont(DISPLAY, pattern);
        }
        if (!font->font) {
            sprintf(pattern, "-*-%s-%s-*-*-*-*-*-*-*-%c-*-iso8859-1",
                name ? name : "*",
                weight,
                spacing);
            font->font = XLoadQueryFont(DISPLAY, pattern);
        }
        if (!font->font) {
            sprintf(pattern, "-*-*-%s-%c-*-*-%s-*-*-*-%c-*-iso8859-1",
                weight,
                slant,
                sizestr,
                spacing);
            font->font = XLoadQueryFont(DISPLAY, pattern);
        }
        if (!font->font && slant == 'i') {
            sprintf(pattern, "-*-*-%s-o-*-*-%s-*-*-*-%c-*-iso8859-1",
                weight,
                sizestr,
                spacing);
            font->font = XLoadQueryFont(DISPLAY, pattern);
        }
        if (!font->font) {
            sprintf(pattern, "-*-*-%s-*-*-*-%s-*-*-*-%c-*-iso8859-1",
                weight,
                sizestr,
                spacing);
            font->font = XLoadQueryFont(DISPLAY, pattern);
        }
        if (!font->font) {
            sprintf(pattern, "-*-*-*-*-*-*-%s-*-*-*-%c-*-iso8859-1",
                sizestr,
                spacing);
            font->font = XLoadQueryFont(DISPLAY, pattern);
        }
        if (!font->font) {
            sprintf(pattern, "-*-*-%s-*-*-*-*-*-*-*-%c-*-iso8859-1",
                weight,
                spacing);
            font->font = XLoadQueryFont(DISPLAY, pattern);
        }
        if (!font->font) {
            sprintf(pattern, "-*-*-*-*-*-*-%s-*-*-*-r-*-iso8859-1", sizestr);
            font->font = XLoadQueryFont(DISPLAY, pattern);
        }
        if (!font->font) {
            sprintf(pattern, "-*-*-*-*-*-*-%s-*-*-*-*-*-iso8859-1", sizestr);
            font->font = XLoadQueryFont(DISPLAY, pattern);
        }
        if (!font->font)
            font->font = XLoadQueryFont(DISPLAY, "-*-*-*-*-*-*-*-*-*-*-*-*-iso8859-1");
        if (!font->font)
            font->font = XLoadQueryFont(DISPLAY, "-*-*-*-*-*-*-*-*-*-*-*-*-*-*");
        free(pattern);
        if (!font->font)
            fprintf(stderr, "Little Forms: failed to find any fallback X11 font, this will most likely crash\n");
    }
#endif
    if (!font->font) {
        free(font);
        font = NULL;
    } else {
        font->name = ff_strdup(name);
        font->size = size;
        font->flags = flags & (~FF_EXACT_FONT);
        fonts = realloc(fonts, sizeof(ff_font_t)*(font_count + 1));
        fonts[font_count++] = font;
    }
    return font;
}

ff_font_t ff_default_font(int which)
{
    switch (which) {
    case FF_BOLD_FONT: return bold_font;
    case FF_ITALIC_FONT: return italic_font;
    case FF_MONOSPACE_FONT: return mono_font;
    case FF_BIG_FONT: return big_font;
    case FF_SMALL_FONT: return small_font;
    }
    return standard_font;
}

static void ff_font_free(ff_font_t font)
{
    if (font) {
        free(font->name);
#ifdef LFORMS_USE_XFT
        if (font->font) XftFontClose(DISPLAY, font->font);
#else
        if (font->font) XFreeFont(DISPLAY, font->font);
#endif
        free(font);
    }
}

const char* ff_font_name(ff_font_t font)
{
    return font ? font->name : standard_font->name;
}

int ff_font_size(ff_font_t font)
{
    return font ? font->size : standard_font->size;
}

unsigned ff_font_flags(ff_font_t font)
{
    return font ? font->flags : standard_font->flags;
}

#ifndef LFORMS_USE_XFT
struct enumfont
{
    char* name;
    int* size;
    int sizes;
    unsigned flags;
};

static int enumfont_compare(const void* a, const void* b)
{
    return strcmp(((struct enumfont*)a)->name,
                  ((struct enumfont*)b)->name);
}

#endif

void ff_font_enum(ff_font_enum_proc proc, void* data, unsigned enum_flags)
{
    const int const_sizes[] = {6, 8, 9, 10, 11, 12, 13, 14, 15, 16, 18,
        20, 22, 24, 28, 32, 36, 48, 64, 72, 80, 108, 110, 120, 135, 160};
#ifdef LFORMS_USE_XFT
    XftFontSet* fs = XftListFonts(DISPLAY, SCREEN,
        XFT_SCALABLE, XftTypeBool, 1, NULL, XFT_FAMILY, NULL);
    int i;
    char** name;
    char* lastname = "";
    if (!fs->nfont) {
        XftFontSetDestroy(fs);
        return;
    }
    name = malloc(sizeof(char*)*(size_t)fs->nfont);
    for (i=0; i < fs->nfont; i++) {
        /* note that we do not check the result, if this fails it will
         * use the last value in "thename" and in a later step any
         * duplicate values will be ignored (we take advantage of the
         * fact that we'll need to do cleanup for the duplicates that
         * Xft does report anyway) */
        XftPatternGetString(fs->fonts[i], XFT_FAMILY, 0, &lastname);
        name[i] = lastname;
    }
    qsort(name, (size_t)fs->nfont, sizeof(char*), ff_qsort_strcmp);
    for (i=0; i < fs->nfont; i++) if (name[i][0] && strcmp(lastname, name[i])) {
        XftFontSet* subfs = XftListFonts(DISPLAY, SCREEN,
            XFT_FAMILY, XftTypeString, name[i], NULL,
            XFT_SLANT, XFT_WEIGHT, XFT_SPACING, NULL);
        unsigned flags = FF_NOFLAGS;
        int j;
        for (j=0; j < subfs->nfont; j++) {
            int weight, slant, spacing;
            if (XftPatternGetInteger(subfs->fonts[j], XFT_WEIGHT, 0, &weight) == XftResultMatch)
                if (weight >= 200) flags |= FF_BOLD;
            if (XftPatternGetInteger(subfs->fonts[j], XFT_SLANT, 0, &slant) == XftResultMatch)
                if (slant >= 100) flags |= FF_ITALIC;
            if (XftPatternGetInteger(subfs->fonts[j], XFT_SPACING, 0, &spacing) == XftResultMatch)
                if (spacing == 100) flags |= FF_MONOSPACE;
        }
        XftFontSetDestroy(subfs);
        proc(name[i], const_sizes, sizeof(const_sizes)/sizeof(int), flags, data);
        lastname = name[i];
    }
    free(name);
    XftFontSetDestroy(fs);
#else
    int i, ii, count, sizes, sz, efonts = 0;
    size_t h, hh, l, ll, tc;
    char** fonts = XListFonts(DISPLAY, "*", 2147483647, &count);
    char tmp[256];
    struct enumfont* efont = malloc(sizeof(struct enumfont)*(size_t)count);
    if (fonts) {
        int* size = malloc((size_t)count*sizeof(int));
        for (i=0; i < count; i++) {
            unsigned flags = FF_NOFLAGS;
            char* name = NULL;
            /* Ignore names outside of XLFD */
            if (fonts[i][0] != '-') goto next;
            l = strlen(fonts[i]);
            if (l < 5 || l > 256) goto next;
            tc = 0;
            /* Foundry */
            h = 1;
            while (h < l && fonts[i][h] != '-') h++;
            if (fonts[i][h] != '-') goto next;
            /* Family */
            h++;
            while (h < l && fonts[i][h] != '-') tmp[tc++] = fonts[i][h++];
            if (fonts[i][h] != '-') goto next;
            tmp[tc] = 0;
            name = ff_strdup(tmp);
            /* Weight */
            tc = 0;
            h++;
            while (h < l && fonts[i][h] != '-') tmp[tc++] = fonts[i][h++];
            if (fonts[i][h] != '-') goto next;
            tmp[tc] = 0;
            flags |= ff_streq_nocase(tmp, "bold") ? FF_BOLD : 0;
            flags |= ff_streq_nocase(tmp, "demi") ? FF_BOLD : 0;
            flags |= ff_streq_nocase(tmp, "demi bold") ? FF_BOLD : 0;
            flags |= ff_streq_nocase(tmp, "demibold") ? FF_BOLD : 0;
            /* Slant */
            tc = 0;
            h++;
            while (h < l && fonts[i][h] != '-') tmp[tc++] = fonts[i][h++];
            if (fonts[i][h] != '-') goto next;
            tmp[tc] = 0;
            flags |= ff_streq_nocase(tmp, "i") ? FF_ITALIC : 0;
            flags |= ff_streq_nocase(tmp, "o") ? FF_ITALIC : 0;
            /* SetWidth */
            h++;
            while (h < l && fonts[i][h] != '-') h++;
            if (fonts[i][h] != '-') goto next;
            /* AddStyle */
            h++;
            while (h < l && fonts[i][h] != '-') h++;
            if (fonts[i][h] != '-') goto next;
            /* PixelSize */
            tc = 0;
            h++;
            if (fonts[i][h] == '[') goto next; /* No matrix support */
            while (h < l && fonts[i][h] != '-') tmp[tc++] = fonts[i][h++];
            tmp[tc] = 0;
            if (strcmp(tmp, "0")) {
                sizes = 1;
                size[0] = atoi(tmp);
                for (ii=i + 1; ii < count; ii++) {
                    /* Ignore names outside of XLFD */
                    if (fonts[ii][0] != '-') continue;
                    ll = strlen(fonts[i]);
                    if (ll < 5 || ll > 256) continue;
                    hh = 1;
                    /* Foundry */
                    while (hh < ll && fonts[ii][hh] != '-') hh++;
                    hh++;
                    /* Family */
                    tc = 0;
                    while (hh < ll && fonts[ii][hh] != '-') tmp[tc++] = fonts[ii][hh++];
                    tmp[tc] = 0;
                    if (!ff_streq_nocase(tmp, name)) break;
                    hh++;
                    /* Weight */
                    while (hh < ll && fonts[ii][hh] != '-') hh++;
                    hh++;
                    /* Slant */
                    while (hh < ll && fonts[ii][hh] != '-') hh++;
                    hh++;
                    /* SetWidth */
                    while (hh < ll && fonts[ii][hh] != '-') hh++;
                    hh++;
                    /* PixelSize */
                    tc = 0;
                    hh++;
                    while (hh < ll && fonts[ii][hh] != '-') tmp[tc++] = fonts[ii][hh++];
                    tmp[tc] = 0;
                    sz = atoi(tmp);
                    if (sz) size[sizes++] = sz;
                }
                i = ii - 1;
            } else {
                sizes = sizeof(const_sizes)/sizeof(const_sizes[0]);
                memcpy(size, const_sizes, sizeof(int)*(size_t)sizes);
            }
            efont[efonts].name = ff_strdup(name);
            efont[efonts].size = malloc((size_t)sizes*sizeof(int));
            memcpy(efont[efonts].size, size, (size_t)sizes*sizeof(int));
            efont[efonts].sizes = sizes;
            efont[efonts].flags = flags;
            efonts++;
          next:
            free(name);
        }
        XFreeFontNames(fonts);
        free(size);
        qsort(efont, (size_t)efonts, sizeof(struct enumfont), enumfont_compare);
        for (i=0; i < efonts; i++) {
            for (ii=i + 1; ii < efonts; ii++) {
                if (strcmp(efont[i].name, efont[ii].name)) break;
                efont[i].flags |= efont[ii].flags;
                efont[i].size = realloc(efont[i].size,
                    sizeof(int)*(size_t)(efont[i].sizes + efont[ii].sizes));
                memcpy(efont[i].size + efont[i].sizes, efont[ii].size,
                    sizeof(int)*(size_t)efont[ii].sizes);
                efont[i].sizes += efont[ii].sizes;
                free(efont[ii].name);
                free(efont[ii].size);
            }
            if (ii > i + 1) {
                memmove(efont + i + 1, efont + ii,
                    sizeof(struct enumfont)*(size_t)(efonts - ii));
                efonts -= (ii - i - 1);
            }
            qsort(efont[i].size, (size_t)efont[i].sizes, sizeof(int), ff_qsort_int);
            efont[i].sizes -= ff_unrepeat(efont[i].size, (size_t)efont[i].sizes, sizeof(int));
        }
        for (i=0; i < efonts; i++) {
            for (ii=0; efont[i].name[ii]; ii++)
                if (!ii || !isalnum(efont[i].name[ii - 1]))
                    efont[i].name[ii] = (char)toupper(efont[i].name[ii]);
            proc(efont[i].name, efont[i].size, efont[i].sizes, efont[i].flags, data);
            free(efont[i].size);
            free(efont[i].name);
        }
        free(efont);
    }
#endif
}
