#include "win32winsys.h"
#include <windowsx.h>

#define TIMER_ID 0x1337

struct dfrev
{
    HWND wnd;
    UINT msg;
    WPARAM wp;
    LPARAM lp;
};

static int exit_requested;
static struct dfrev* dfrev;
static size_t dfrevs, dfrevc;
static HWND last_click_win;
static int last_click_x, last_click_y;
static unsigned last_click_btn;
static unsigned long last_click_ticks;
static struct modal* gmodal;

static int send_event(wswin_t* win, unsigned event, int x, int y, int z, void* p, int flags)
{
    if (!win || !win->ffwindow) return 0;
    return ff_send(win->ffwindow, event, x, y, z, p, flags);
}

static void focus_modal(struct modal* mdl)
{
    while (mdl->window->modal) mdl = mdl->window->modal;
    BringWindowToTop(mdl->window->w);
}

static void focus_disabled(wswin_t* win)
{
    if (gmodal) {
        focus_modal(gmodal);
        return;
    }
    if (win->modal) {
        focus_modal(win->modal);
        return;
    }
}

static int send_disableable_event(wswin_t* win, unsigned event, int x, int y, int z, void* p, unsigned flags, int translatexy)
{
    if (!win || !win->ffwindow) return 0;
    if (!ff_ws_enabled(win)) {
        focus_disabled(win);
        if (win->parent && !ff_ispopup(win->parent->ffwindow)) {
            if (translatexy) ff_ws_translate(&x, &y, win, win->parent);
            return send_disableable_event(win->parent, event, x, y, z, p, flags, FF_YES);
        }
        return 0;
    }
    return ff_send(win->ffwindow, event, x, y, z, p, flags);
}

static void defermsg(HWND wnd, UINT msg, WPARAM wp, LPARAM lp)
{
    if (dfrevc == dfrevs) {
        dfrevc += 64;
        dfrev = realloc(dfrev, sizeof(struct dfrev)*dfrevc);
    }
    dfrev[dfrevs].wnd = wnd;
    dfrev[dfrevs].msg = msg;
    dfrev[dfrevs].wp = wp;
    dfrev[dfrevs].lp = lp;
    dfrevs++;
}

static int keycode(int msgcode)
{
    int fromtable[] = {
        VK_BACK, VK_TAB, VK_OEM_CLEAR, VK_CLEAR, VK_RETURN, VK_PAUSE,
        VK_ESCAPE, VK_DELETE, VK_HOME, VK_LEFT, VK_RIGHT, VK_UP,
        VK_DOWN, VK_PRIOR, VK_NEXT, VK_END, VK_INSERT, VK_NUMLOCK,
        VK_SCROLL, VK_CAPITAL, VK_F1, VK_F2, VK_F3, VK_F4, VK_F5, VK_F6,
        VK_F7, VK_F8, VK_F9, VK_F10, VK_F11, VK_F12, VK_F13, VK_F14,
        VK_SHIFT, VK_SHIFT, VK_CONTROL, VK_CONTROL, VK_MENU, VK_MENU,
        VK_LWIN, VK_RWIN, VK_SPACE, VK_OEM_8, VK_OEM_7,
        0xBB, 0xBC, 0xBD, 0xBE,
        VK_OEM_2, 0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39,
        VK_OEM_1, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48, 0x49, 0x4A,
        0x4B, 0x4C, 0x4D, 0x4E, 0x4F, 0x50, 0x51, 0x52, 0x53, 0x54, 0x55,
        0x56, 0x57, 0x58, 0x59, 0x5A, VK_OEM_4, VK_OEM_5,
        VK_OEM_6, VK_OEM_3, 0
    };
    int totable[] = {
        FFK_BACKSPACE, FFK_TAB, FFK_CLEAR, FFK_CLEAR, FFK_RETURN, FFK_PAUSE,
        FFK_ESCAPE, FFK_DELETE, FFK_HOME, FFK_LEFT, FFK_RIGHT, FFK_UP,
        FFK_DOWN, FFK_PAGEUP, FFK_PAGEDOWN, FFK_END, FFK_INSERT, FFK_NUM_LOCK,
        FFK_SCROLL_LOCK, FFK_CAPS_LOCK, FFK_F1, FFK_F2, FFK_F3, FFK_F4, FFK_F5, FFK_F6,
        FFK_F7, FFK_F8, FFK_F9, FFK_F10, FFK_F11, FFK_F12, FFK_F13, FFK_F14,
        FFK_LSHIFT, FFK_RSHIFT, FFK_LCONTROL, FFK_RCONTROL, FFK_LALT, FFK_RALT,
        FFK_LSUPER, FFK_RSUPER, FFK_SPACE, FFK_EXCLAM, FFK_QUOTEDBL,
        FFK_PLUS, FFK_COMMA, FFK_MINUS, FFK_PERIOD,
        FFK_SLASH, FFK_0, FFK_1, FFK_2, FFK_3, FFK_4, FFK_5, FFK_6, FFK_7, FFK_8, FFK_9,
        FFK_SEMICOLON, FFK_A, FFK_B, FFK_C, FFK_D, FFK_E, FFK_F, FFK_G, FFK_H, FFK_I, FFK_J,
        FFK_K, FFK_L, FFK_M, FFK_N, FFK_O, FFK_P, FFK_Q, FFK_R, FFK_S, FFK_T, FFK_U,
        FFK_V, FFK_W, FFK_X, FFK_Y, FFK_Z, FFK_BRACKETLEFT, FFK_BACKSLASH,
        FFK_BRACKETRIGHT, 0
    };
    int i;

    if (msgcode == VK_CONTROL || msgcode == VK_MENU || msgcode == VK_SHIFT) {
        switch (msgcode) {
        case VK_SHIFT:
            if (GetKeyState(VK_RSHIFT)&0x8000)
                return FFK_RSHIFT;
            return FFK_LSHIFT;
        case VK_MENU:
            if (GetKeyState(VK_RMENU)&0x8000)
                return FFK_RALT;
            return FFK_LALT;
        case VK_CONTROL:
            if (GetKeyState(VK_RCONTROL)&0x8000)
                return FFK_RCONTROL;
            return FFK_LCONTROL;
        }
    }

    for (i=0; fromtable[i]; i++)
        if (fromtable[i] == msgcode)
            return totable[i];

    return msgcode + 100000;
}

static int keyshift(void)
{
    int r = 0;
    if (GetKeyState(VK_SHIFT)&0x8000) r |= FF_SHIFT;
    if (GetKeyState(VK_CONTROL)&0x8000) r |= FF_CONTROL;
    if (GetKeyState(VK_MENU)&0x8000) r |= FF_ALT;
    return r;
}

static void send_press_event(HWND wnd, int x, int y, int btn)
{
    unsigned long ticks;
    send_disableable_event(ff_ws_wswin(wnd), FF_PRESS, x, y, btn, NULL, FF_BUBBLE, FF_YES);
    ticks = ff_ticks();
    if (last_click_btn == btn &&
        last_click_win == wnd &&
        ticks - last_click_ticks < 400 &&
        abs(last_click_x - x) < 16 &&
        abs(last_click_y - y) < 16) {
        last_click_ticks = 0;
        send_disableable_event(ff_ws_wswin(wnd), FF_DBLCLICK, x, y, btn, NULL, FF_BUBBLE, FF_YES);
    } else last_click_ticks = ticks;
    last_click_btn = btn;
    last_click_x = x;
    last_click_y = y;
    last_click_win = wnd;
}

static void process_msg(HWND wnd, UINT msg, WPARAM wp, LPARAM lp)
{
    wswin_t* wswin;
    ff_window_t win;
    RECT r;

    switch (msg) {
    case WM_SHOWWINDOW:
        if (!wp) return;
        wswin = ff_ws_wswin(wnd);
        win = wswin ? wswin->ffwindow : NULL;
        if (win) ff_relayout(win);
        return;
    case WM_LBUTTONDOWN:
        SetCapture(wnd);
        send_press_event(wnd, GET_X_LPARAM(lp), GET_Y_LPARAM(lp), 1);
        return;
    case WM_LBUTTONUP:
        ReleaseCapture();
        send_disableable_event(ff_ws_wswin(wnd), FF_RELEASE, GET_X_LPARAM(lp), GET_Y_LPARAM(lp), 1, NULL, FF_BUBBLE, FF_YES);
        return;
    case WM_MBUTTONDOWN:
        SetCapture(wnd);
        send_press_event(wnd, GET_X_LPARAM(lp), GET_Y_LPARAM(lp), 2);
        return;
    case WM_MBUTTONUP:
        ReleaseCapture();
        send_disableable_event(ff_ws_wswin(wnd), FF_RELEASE, GET_X_LPARAM(lp), GET_Y_LPARAM(lp), 2, NULL, FF_BUBBLE, FF_YES);
        return;
    case WM_RBUTTONDOWN:
        SetCapture(wnd);
        send_press_event(wnd, GET_X_LPARAM(lp), GET_Y_LPARAM(lp), 3);
        return;
    case WM_RBUTTONUP:
        ReleaseCapture();
        send_disableable_event(ff_ws_wswin(wnd), FF_RELEASE, GET_X_LPARAM(lp), GET_Y_LPARAM(lp), 3, NULL, FF_BUBBLE, FF_YES);
        return;
    case WM_MOUSEMOVE:
        send_disableable_event(ff_ws_wswin(wnd), FF_MOTION, GET_X_LPARAM(lp), GET_Y_LPARAM(lp), 0, NULL, FF_BUBBLE, FF_YES);
        return;
    case WM_MOUSEWHEEL:
        {
            POINT p;
            HWND cwin;
            wswin_t* wswin;
            p.x = GET_X_LPARAM(lp);
            p.y = GET_Y_LPARAM(lp);
            cwin = WindowFromPoint(p);
            wswin = ff_ws_wswin(cwin);
            if (wswin) {
                send_disableable_event(wswin, FF_PRESS, p.x, p.y, GET_WHEEL_DELTA_WPARAM(wp) > 0 ? 4 : 5, NULL, FF_BUBBLE, FF_YES);
                send_disableable_event(wswin, FF_RELEASE, p.x, p.y, GET_WHEEL_DELTA_WPARAM(wp) > 0 ? 4 : 5, NULL, FF_BUBBLE, FF_YES);
            }
        }
        return;
    case WM_KEYDOWN:
    case WM_KEYUP:
        send_disableable_event(ff_ws_wswin(wnd), msg == WM_KEYDOWN ? FF_KEYDOWN : FF_KEYUP, keycode(wp), keyshift(), 0, NULL, FF_BUBBLE, FF_NO);
        return;
    case WM_CHAR:
#ifdef UNICODE
        {
            unsigned char utf8[5];
            memset(utf8, 0, 5);
            wswin = ff_ws_wswin(wnd);
            if (wp >= 0xD800 && wp <= 0xDBFF) {
                wswin->leadsurr = wp;
                return;
            } else if (wp >= 0xDC00 && wp <= 0xDFFF) {
                unsigned short utf16[3];
                utf16[0] = wswin->leadsurr;
                utf16[1] = wp;
                utf16[2] = 0;
                wswin->leadsurr = 0;
                if (!ff_dec16(utf16, &wp)) return;
            }
            wswin->leadsurr = 0;
            if (ff_enc8(wp, utf8))
                send_disableable_event(wswin, FF_TEXT, 0, keyshift(), 0, utf8, FF_BUBBLE, FF_NO);
        }
#else
        {
            char ch[2];
            if (wp > 127) return;
            ch[0] = (char)wp;
            ch[1] = 0;
            send_disableable_event(ff_ws_wswin(wnd), FF_TEXT, 0, keyshift(), 0, ch, FF_BUBBLE, FF_NO);
        }
#endif
        return;
    case WM_SETFOCUS:
        send_event(ff_ws_wswin(wnd), FF_FOCUS, 0, 0, 0, NULL, FF_BUBBLE);
        return;
    case WM_KILLFOCUS:
        send_event(ff_ws_wswin(wnd), FF_BLUR, 0, 0, 0, NULL, FF_BUBBLE);
        return;
    case WM_SIZE:
        wswin = ff_ws_wswin(wnd);
        if (!wswin) break;
        GetClientRect(wnd, &r);
        wswin->law = r.right;
        wswin->lah = r.bottom;
        send_event(wswin, FF_AREA, r.right, r.bottom, 0, NULL, FF_NOFLAGS);
        return;
    case WM_CLOSE:
        wswin = ff_ws_wswin(wnd);
        if (!wswin) {
            DestroyWindow(wnd);
            return;
        }
        if (wswin->wsflags & FF_WSFLAG_DISABLED) {
            focus_disabled(wswin);
            return;
        }
        if (!send_event(wswin, FF_CLOSE, 0, 0, 0, NULL, FF_NOFLAGS))
            ff_ws_destroy_window(wswin);
        return;
    }
}

static LRESULT CALLBACK window_proc(HWND wnd, UINT msg, WPARAM wp, LPARAM lp)
{
    wswin_t* wswin;
    ff_window_t win;
    ff_gc_t gc;
    size_t i;
    RECT r;

    switch (msg) {
    case WM_ERASEBKGND:
        return 0;
    case WM_PAINT:
    case WM_NCPAINT:
        wswin = ff_ws_wswin(wnd);
        win = wswin ? wswin->ffwindow : NULL;
        if (wswin) {
            GetClientRect(wnd, &r);
            if (r.right != wswin->law || r.bottom != wswin->lah) {
                wswin->law = r.right;
                wswin->lah = r.bottom;
                send_event(wswin, FF_AREA, r.right, r.bottom, 0, NULL, FF_NOFLAGS);
            }
        }
        if (wswin->flags & FF_OPENGL) {
            if (wglMakeCurrent(wswin->dbdc, wswin->rc)) {
                ff__sendpaint(wswin->ffwindow, NULL);
                SwapBuffers(wswin->dbdc);
            }
        } else {
            gc = win ? ff_ws_create_gc(wnd, wswin) : NULL;
            if (gc) {
                ff_gc_t prevgc = wswin->gc;
                wswin->gc = gc;
                ff__sendpaint(wswin->ffwindow, gc);
                ff_ws_free_gc(gc, wswin);
                wswin->gc = prevgc;
            }
        }
        break;
    case WM_SETCURSOR:
        {
            POINT p;
            HWND cwin;
            wswin_t* wswin;
            GetCursorPos(&p);
            cwin = WindowFromPoint(p);
            wswin = ff_ws_wswin(cwin);
            if (wswin && wswin->cursor)
                SetClassLongPtr(wnd, GCL_HCURSOR, (LONG_PTR)wswin->cursor);
        }
        break;
    case WM_SHOWWINDOW:
        wswin = ff_ws_wswin(wnd);
        if (wswin && (wswin->flags & FF_CENTERED) && !GetParent(wnd)) {
            RECT dr;
            GetWindowRect(wnd, &r);
            GetClientRect(GetDesktopWindow(), &dr);
            wswin->flags &= ~FF_CENTERED;
            ff_ws_move(wswin, (dr.right - (r.right - r.left)) / 2,
                              (dr.bottom - (r.bottom - r.top)) / 2);
        }
        defermsg(wnd, msg, wp, lp);
        break;
    case WM_LBUTTONDOWN:
    case WM_LBUTTONUP:
    case WM_MBUTTONDOWN:
    case WM_MBUTTONUP:
    case WM_RBUTTONDOWN:
    case WM_RBUTTONUP:
    case WM_MOUSEMOVE:
    case WM_MOUSEWHEEL:
    case WM_KEYDOWN:
    case WM_KEYUP:
    case WM_CHAR:
    case WM_SETFOCUS:
    case WM_KILLFOCUS:
    case WM_SIZE:
    case WM_CLOSE:
        defermsg(wnd, msg, wp, lp);
        return 0;
    case WM_DESTROY:
        wswin = ff_ws_wswin(wnd);
        if (wswin) {
            if (wswin->ffwindow) ff__destroyed(wswin->ffwindow);
            ff_ws_unregwin(wswin);
            free(wswin);
        }
        for (i=0; i < dfrevs; i++)
            if (dfrev[i].wnd == wnd)
                dfrev[i].wnd = 0;
        break;
    case WM_TIMER:
        if (wp == TIMER_ID) {
            unsigned long now = ff_ticks();
            send_event(ff_ws_wswin(wnd), FF_TIMER, 0, 0, (int)now, NULL, FF_NOFLAGS);
            return 0;
        }
        break;
    }
    return DefWindowProc(wnd, msg, wp, lp);
}

static void process_dfrs(void)
{
    static int lock;
    size_t i;
    if (lock) return;
    lock++;
    for (i=0; i < dfrevs; i++) {
        if (dfrev[i].wnd)
            process_msg(dfrev[i].wnd, dfrev[i].msg, dfrev[i].wp, dfrev[i].lp);
    }
    dfrevs = 0;
    lock--;
}

int ff_ws_init(int argc, char** argv)
{
    WNDCLASS wc;
    memset(&wc, 0, sizeof(WNDCLASS));
    wc.style = CS_HREDRAW|CS_VREDRAW;
    wc.lpfnWndProc = window_proc;
    wc.hInstance = GetModuleHandle(NULL);
    wc.hIcon = LoadIcon(NULL, IDI_APPLICATION);
    wc.hCursor = LoadCursor(NULL, IDC_ARROW);
    wc.lpszClassName = TEXT("FFWSWIN");
    RegisterClass(&wc);

    memset(&wc, 0, sizeof(WNDCLASS));
    wc.style = CS_HREDRAW|CS_VREDRAW|CS_OWNDC;
    wc.lpfnWndProc = window_proc;
    wc.hInstance = GetModuleHandle(NULL);
    wc.hIcon = LoadIcon(NULL, IDI_APPLICATION);
    wc.hCursor = LoadCursor(NULL, IDC_ARROW);
    wc.lpszClassName = TEXT("FFWSWINGL");
    RegisterClass(&wc);

    ff_ws_common_init();
    return 1;
}

void ff_ws_shutdown(void)
{
    while (ff_ws_has_events()) ff_ws_pump_events();
    UnregisterClass(TEXT("FFWSWINGL"), GetModuleHandle(NULL));
    UnregisterClass(TEXT("FFWSWIN"), GetModuleHandle(NULL));
    ff_ws_common_shutdown();
    free(dfrev);
    dfrevc = dfrevs = 0;
}

void ff_ws_exit(int exitcode)
{
    (void)exitcode;
    exit_requested = FF_YES;
    PostQuitMessage(0);
}

void ff_ws_sync(void)
{
}

int ff_ws_has_events(void)
{
    MSG msg;
    if (dfrevs > 0) return FF_YES;
    return PeekMessage(&msg, 0, 0, 0, PM_NOREMOVE) ? FF_YES : FF_NO;
}

int ff_ws_pump_events(void)
{
    MSG msg;
    int shall_exit = FF_NO;
    while (PeekMessage(&msg, 0, 0, 0, PM_NOREMOVE)) {
        shall_exit = FF_YES;
        if (GetMessage(&msg, 0, 0, 0)) {
            TranslateMessage(&msg);
            DispatchMessage(&msg);
        } else return 0;
    }
    if (exit_requested) {
        process_dfrs();
        /* force exit in case ff_ws_pump_events is called
           from inside deferred event */
        dfrevs = 0;
        return 0;
    }
    if (shall_exit) return 1;
    if (dfrevs > 0) {
        process_dfrs();
        return 1;
    }
    if (GetMessage(&msg, 0, 0, 0)) {
        TranslateMessage(&msg);
        DispatchMessage(&msg);
        process_dfrs();
        return 1;
    }
    return 0;
}

unsigned long ff_ticks(void)
{
    __int64 freq, time;
    if (QueryPerformanceFrequency((LARGE_INTEGER*)&freq)) {
        if (QueryPerformanceCounter((LARGE_INTEGER*)&time)) {
            return (unsigned long)(time*(__int64)1000/freq);
        }
    }
    return GetTickCount();
}

int ff_ws_timer(void* window, int ms)
{
    wswin_t* wswin = window;
    if (wswin->w) {
        if (ms < 0) KillTimer(wswin->w, TIMER_ID);
        else SetTimer(wswin->w, TIMER_ID, ms < 1 ? 1 : ms, NULL);
        return 1;
    }
    return 0;
}

void ff_ws_pushmodal(void* window, void* target)
{
    wswin_t* wswin = window;
    wswin_t* wstgt = target;
    struct modal* mdl;
    if (!wswin || !wswin->w) return;
    if (wstgt) {
        if (!wstgt->w) return;
        mdl = malloc(sizeof(struct modal));
        mdl->window = window;
        mdl->next = wstgt->modal;
        wstgt->modal = mdl;
    } else {
        mdl = malloc(sizeof(struct modal));
        mdl->window = window;
        mdl->next = gmodal;
        gmodal = mdl;
    }
    ff_ws_show(window, FF_YES);
}

void ff_ws_popmodal(void* target)
{
    wswin_t* wswin = target;
    struct modal* mdl = NULL;
    if (target) {
        if (!wswin || !wswin->w) return;
        mdl = wswin->modal;
        if (mdl) wswin->modal = mdl->next;
    } else if (gmodal) {
        mdl = gmodal;
        if (mdl) gmodal = mdl->next;
    }
    if (mdl) free(mdl);
}

void* ff_ws_x11_display(void)
{
    return NULL;
}
