#define LFORMS_SOURCE_CODE
#include <stdlib.h>
#include <string.h>
#include <lforms.h>

static char* onglyph[] = {
    "16 16 5 1",
    "  s none",
    ". s 3d-dark",
    "- s 3d-shadow",
    "@ s 3d-light",
    "* s 3d-face",
    "     -----      ",
    "   --.....--    ",
    "  -..*****-*-   ",
    " -.-**@@@@****  ",
    " -.*@@@@@@@@**  ",
    "-.**@@@@@@@@**@ ",
    "-.*@@@@@@@@@@*@ ",
    "-.*@@@@@@@@@@*@ ",
    "-.*@@@@@@@@@@*@ ",
    "-.*@@@@@@@@@@*@ ",
    " --*@@@@@@@@*@  ",
    " -**@@@@@@@@*@  ",
    "  -***@@@@**@   ",
    "   *@*****@@    ",
    "     @@@@@      ",
    "                "};
static char* offglyph[] = {
    "16 16 5 1",
    "  s none",
    ". s 3d-dark",
    "- s 3d-shadow",
    "@ s 3d-light",
    "* s 3d-face",
    "     -----      ",
    "   --.....--    ",
    "  -..*****-*-   ",
    " -.-**********  ",
    " -.***********  ",
    "-.************@ ",
    "-.************@ ",
    "-.************@ ",
    "-.************@ ",
    "-.************@ ",
    " --**********@  ",
    " -***********@  ",
    "  -*********@   ",
    "   *@*****@@    ",
    "     @@@@@      ",
    "                "};

typedef struct _state_t
{
    char* caption;
    void* value;
    unsigned selected:1;
    unsigned autoselect:1;
    unsigned hasvalue:1;
} state_t;

static ff_bitmap_t onbmp, offbmp;

static void free_bitmaps(void)
{
    if (!onbmp) return;
    ff_bitmap_free(onbmp);
    ff_bitmap_free(offbmp);
}

static void update_prefsize(ff_window_t rb)
{
    state_t* state = ff_get(rb, "-state");
    int tw, th;
    ff_font_text_size(ff_font_of(rb), state->caption, &tw, &th);
    ff_prefsize(rb, tw + 19, th > 16 ? th : 16);
}

static int set_handler(void* rb, ff_event_t* ev, void* data)
{
    ff_namevalue_t* nv = ev->p;
    if (!strcmp(nv->name, "caption")) {
        state_t* state = ff_get(rb, "-state");
        free(state->caption);
        state->caption = ff_strdup(nv->value);
        update_prefsize(rb);
        ff_paint(rb);
        return FF_YES;
    } else if (!strcmp(nv->name, "selected")) {
        state_t* state = ff_get(rb, "-state");
        unsigned newval = nv->value ? FF_YES : FF_NO;
        if (newval != state->selected) {
            state->selected = newval&1;
            if (state->selected) {
                ff_window_t parent = ff_parent(rb);
                ff_window_t* children = ff_children(parent);
                if (children) {
                    int i;
                    for (i=0; children[i]; i++) if (children[i] != rb) {
                        const char* class = ff_get(children[i], "class");
                        if (class && !strcmp(class, "radio"))
                            ff_seti(children[i], "selected", FF_NO);
                    }
                }
                ff_free_children(children);
                if (state->hasvalue) {
                    ff_set(parent, "radio-value", state->value);
                    ff_post(parent, FF_CHANGE, 0, 0, 0, rb, FF_BUBBLE);
                }
            }
            ff_paint(rb);
            ff_post(rb, FF_CHANGE, state->selected, 0, 0, NULL, FF_BUBBLE);
        }
        return FF_YES;
    } else if (!strcmp(nv->name, "autoselect")) {
        state_t* state = ff_get(rb, "-state");
        unsigned newval = nv->value ? FF_YES : FF_NO;
        if (newval != state->autoselect) {
            state->autoselect = newval&1;
            ff_paint(rb);
        }
        return FF_YES;
    } else if (!strcmp(nv->name, "value")) {
        state_t* state = ff_get(rb, "-state");
        if (nv->value != state->value || !state->hasvalue) {
            state->value = nv->value;
            state->hasvalue = FF_YES;
            if (state->selected) {
                ff_window_t parent = ff_parent(rb);
                ff_set(parent, "radio-value", state->value);
                ff_post(parent, FF_CHANGE, 0, 0, 0, rb, FF_BUBBLE);
            }
        }
        return FF_YES;
    } else if (!strcmp(nv->name, "has-value")) {
        state_t* state = ff_get(rb, "-state");
        unsigned newval = nv->value ? FF_YES : FF_NO;
        if (newval != state->hasvalue) {
            state->hasvalue = newval&1;
            if (state->selected && newval) {
                ff_window_t parent = ff_parent(rb);
                ff_set(parent, "radio-value", state->value);
                ff_post(parent, FF_CHANGE, 0, 0, 0, rb, FF_BUBBLE);
            }
        }
        return FF_YES;
    }
    return FF_NO;
}

static int get_handler(void* rb, ff_event_t* ev, void* data)
{
    ff_namevalue_t* nv = ev->p;
    if (!strcmp(nv->name, "caption")) {
        state_t* state = ff_get(rb, "-state");
        nv->value = state->caption;
        return FF_YES;
    } else if (!strcmp(nv->name, "selected")) {
        state_t* state = ff_get(rb, "-state");
        nv->value = (void*)(ff_intptr_t)state->selected;
        return FF_YES;
    } else if (!strcmp(nv->name, "autoselect")) {
        state_t* state = ff_get(rb, "-state");
        nv->value = (void*)(ff_intptr_t)state->autoselect;
        return FF_YES;
    } else if (!strcmp(nv->name, "value")) {
        state_t* state = ff_get(rb, "-state");
        nv->value = state->value;
        return FF_YES;
    } else if (!strcmp(nv->name, "has-value")) {
        state_t* state = ff_get(rb, "-state");
        nv->value = (void*)(ff_intptr_t)state->hasvalue;
        return FF_YES;
    }
    return FF_NO;
}

static int destroy_handler(void* rb, ff_event_t* ev, void* data)
{
    state_t* state = ff_get(rb, "-state");
    free(state->caption);
    free(state);
    return FF_NO;
}

static int newfont_handler(void* rb, ff_event_t* ev, void* data)
{
    update_prefsize(rb);
    return FF_NO;
}

static int paint_handler(void* rb, ff_event_t* ev, void* data)
{
    ff_gc_t gc = ev->p;
    int w, h, tw, th, cy;
    state_t* state = ff_get(rb, "-state");
    ff_area(rb, NULL, NULL, &w, &h);
    cy = (h - 15)/2;
    ff_color_attr(gc, rb, "background", FF_3DFACE_COLOR);
    ff_fill(gc, 0, 0, w, h);
    ff_color_attr(gc, rb, "foreground",
        ff_enabled(rb) ? FF_3DTEXT_COLOR : FF_3DDISABLEDTEXT_COLOR);
    ff_draw(gc, 0, cy, state->selected ? onbmp : offbmp);
    ff_text_size(gc, state->caption, &tw, &th);
    ff_text(gc, 19, ff_round_int(((float)h + (float)th)/2.0f), state->caption);
    return FF_YES;
}

static int press_handler(void* rb, ff_event_t* ev, void* data)
{
    if (ev->z == 1) {
        state_t* state = ff_get(rb, "-state");
        int old = state->selected;
        if (state->autoselect)
            ff_seti(rb, "selected", FF_YES);
        if (state->selected && !old)
            ff_post(rb, FF_ACTION, 0, 0, 0, NULL, FF_BUBBLE);
        return FF_YES;
    }
    return FF_NO;
}

ff_window_t ff_radio(ff_window_t parent, const char* caption, int selected)
{
    ff_window_t rb = ff_window(parent, 0, 0, 100, 25, FF_NOFLAGS);
    state_t* state = calloc(1, sizeof(state_t));
    if (!onbmp) {
        onbmp = ff_xpm(onglyph, NULL, NULL);
        offbmp = ff_xpm(offglyph, NULL, NULL);
        ff_atexit(free_bitmaps);
    }
    state->caption = ff_strdup(caption);
    state->autoselect = FF_YES;
    ff_set(rb, "-state", state);
    ff_setcs(rb, "class", "radio");
    ff_link(rb, FF_SET, set_handler, NULL);
    ff_link(rb, FF_GET, get_handler, NULL);
    if (selected) ff_seti(rb, "selected", FF_YES);
    ff_link(rb, FF_DESTROY, destroy_handler, NULL);
    ff_link(rb, FF_NEWFONT, newfont_handler, NULL);
    ff_link(rb, FF_PAINT, paint_handler, NULL);
    ff_link(rb, FF_PRESS, press_handler, NULL);
    update_prefsize(rb);
    return rb;
}
