#define LFORMS_SOURCE_CODE
#include <stdlib.h>
#include <string.h>
#include <lforms.h>

typedef struct _state_t
{
    char* caption;
    unsigned down:1;
    unsigned checked:1;
    unsigned autocheck:1;
} state_t;

static void update_prefsize(ff_window_t cb)
{
    state_t* state = ff_get(cb, "-state");
    int tw, th;
    ff_font_text_size(ff_font_of(cb), state->caption, &tw, &th);
    ff_prefsize(cb, tw + 19, th > 16 ? th : 16);
}

static int set_handler(void* cb, ff_event_t* ev, void* data)
{
    ff_namevalue_t* nv = ev->p;
    state_t* state = ff_get(cb, "-state");
    if (!strcmp(nv->name, "caption")) {
        free(state->caption);
        state->caption = ff_strdup(nv->value);
        update_prefsize(cb);
        ff_paint(cb);
        return FF_YES;
    } else if (!strcmp(nv->name, "checked")) {
        state_t* state = ff_get(cb, "-state");
        unsigned newval = nv->value ? FF_YES : FF_NO;
        if (newval != state->checked) {
            state->checked = newval&1;
            ff_paint(cb);
            ff_post(cb, FF_CHANGE, state->checked, 0, 0, NULL, FF_BUBBLE);
        }
        return FF_YES;
    } else if (!strcmp(nv->name, "autocheck")) {
        state_t* state = ff_get(cb, "-state");
        unsigned newval = nv->value ? FF_YES : FF_NO;
        if (newval != state->autocheck) {
            state->autocheck = newval&1;
            ff_paint(cb);
        }
        return FF_YES;
    }
    return FF_NO;
}

static int get_handler(void* cb, ff_event_t* ev, void* data)
{
    ff_namevalue_t* nv = ev->p;
    if (!strcmp(nv->name, "caption")) {
        state_t* state = ff_get(cb, "-state");
        nv->value = state->caption;
        return FF_YES;
    } else if (!strcmp(nv->name, "checked")) {
        state_t* state = ff_get(cb, "-state");
        nv->value = (void*)(ff_intptr_t)state->checked;
        return FF_YES;
    } else if (!strcmp(nv->name, "autocheck")) {
        state_t* state = ff_get(cb, "-state");
        nv->value = (void*)(ff_intptr_t)state->autocheck;
        return FF_YES;
    }
    return FF_NO;
}

static int destroy_handler(void* cb, ff_event_t* ev, void* data)
{
    state_t* state = ff_get(cb, "-state");
    free(state->caption);
    free(state);
    return FF_NO;
}

static int newfont_handler(void* cb, ff_event_t* ev, void* data)
{
    update_prefsize(cb);
    return FF_NO;
}

static int paint_handler(void* cb, ff_event_t* ev, void* data)
{
    ff_gc_t gc = ev->p;
    int w, h, tw, th, cy;
    state_t* state = ff_get(cb, "-state");
    ff_area(cb, NULL, NULL, &w, &h);
    cy = (h - 15)/2;
    ff_color_attr(gc, cb, "background", FF_3DFACE_COLOR);
    ff_fill(gc, 0, 0, w, h);
    if (state->down) {
        ff_color(gc, FF_ACTIVE_COLOR);
        ff_fill(gc, 0, cy + 0, 14, cy + 14);
        ff_color(gc, FF_3DDARK_COLOR);
        ff_line(gc, 0, cy, 13, cy);
        ff_line(gc, 0, cy, 0, cy + 13);
        ff_color(gc, FF_ACTEXT_COLOR);
        if (state->checked) {
            ff_line(gc, 4, cy + 11, 11, cy + 4);
            ff_line(gc, 4, cy + 5, 4, cy + 10);
            ff_line(gc, 3, cy + 11, 11, cy + 3);
            ff_line(gc, 3, cy + 5, 3, cy + 10);
            ff_line(gc, 3, cy + 10, 11, cy + 2);
        }
    } else {
        ff_color(gc, FF_3DLIGHT_COLOR);
        ff_line(gc, 0, cy, 14, cy);
        ff_line(gc, 0, cy, 0, cy + 14);
        ff_color(gc, FF_3DDARK_COLOR);
        ff_line(gc, 14, cy, 14, cy + 14);
        ff_line(gc, 0, cy + 14, 14, cy + 14);
        ff_color(gc, FF_3DSHADOW_COLOR);
        ff_line(gc, 13, cy + 1, 13, cy + 13);
        ff_line(gc, 1, cy + 13, 13, cy + 13);
        if (state->checked) {
            ff_line(gc, 4, cy + 11, 11, cy + 4);
            ff_color(gc, FF_3DDARK_COLOR);
            ff_line(gc, 4, cy + 5, 4, cy + 10);
            ff_line(gc, 3, cy + 11, 11, cy + 3);
            ff_color(gc, FF_3DLIGHT_COLOR);
            ff_line(gc, 3, cy + 5, 3, cy + 10);
            ff_line(gc, 3, cy + 10, 11, cy + 2);
        }
        ff_color_attr(gc, cb, "foreground",
            ff_enabled(cb) ? FF_3DTEXT_COLOR : FF_3DDISABLEDTEXT_COLOR);
    }
    ff_text_size(gc, state->caption, &tw, &th);
    ff_text(gc, 19, ff_round_int(((float)h + (float)th)/2.0f), state->caption);
    return FF_YES;
}

static int press_handler(void* cb, ff_event_t* ev, void* data)
{
    state_t* state = ff_get(cb, "-state");
    if (ev->z == 1) {
        state->down = FF_YES;
        ff_paint(cb);
        return FF_YES;
    }
    return FF_NO;
}

static int release_handler(void* cb, ff_event_t* ev, void* data)
{
    state_t* state = ff_get(cb, "-state");
    if (ev->z == 1 && state->down) {
        int px, py;
        state->down = FF_NO;
        ff_paint(cb);
        ff_pointer(&px, &py);
        if (ff_pick(NULL, px, py) == cb) {
            if (state->autocheck)
                ff_seti(cb, "checked", !ff_geti(cb, "checked"));
            ff_post(cb, FF_ACTION, 0, 0, 0, NULL, FF_BUBBLE);
        }
        return FF_YES;
    }
    return FF_NO;
}

ff_window_t ff_checkbox(ff_window_t parent, const char* caption, int checked)
{
    ff_window_t cb = ff_window(parent, 0, 0, 100, 25, FF_NOFLAGS);
    state_t* state = calloc(1, sizeof(state_t));
    state->caption = ff_strdup(caption);
    state->checked = checked ? FF_YES : FF_NO;
    state->autocheck = FF_YES;
    ff_set(cb, "-state", state);
    ff_setcs(cb, "class", "checkbox");
    ff_link(cb, FF_SET, set_handler, NULL);
    ff_link(cb, FF_GET, get_handler, NULL);
    ff_link(cb, FF_DESTROY, destroy_handler, NULL);
    ff_link(cb, FF_NEWFONT, newfont_handler, NULL);
    ff_link(cb, FF_PAINT, paint_handler, NULL);
    ff_link(cb, FF_PRESS, press_handler, NULL);
    ff_link(cb, FF_RELEASE, release_handler, NULL);
    update_prefsize(cb);
    return cb;
}
