#include <lforms.h>

/* ====================================================================
 * This part implements the "needle" custom control that allows users
 * to move a needle inside a box.  It is an example of how one can
 * create a custom control using Little Forms.  See below for using it.
 * ================================================================== */

/* This will hold the control's own private state.  Strictly speaking
 * we don't need this for this control (we could use private attributes
 * instead) but for more complex controls it is better to avoid the
 * attribute overhead. */
typedef struct _state_t
{
    int position;       /* The position of the needle */
    unsigned moving:1;  /* Bitflag: FF_YES means moving the needle */
} state_t;

/* Called when ff_set() is called on the control to set an attribute */
static int set_handler(void* ndl, ff_event_t* ev, void* data)
{
    /* Get information about the attribute that was set */
    ff_namevalue_t* kv = ev->p;

    /* Check if the requested attribute was the needle's position */
    if (!strcmp(kv->name, "position")) {
        /* Get the control's state data */
        state_t* state = ff_get(ndl, "-state");

        /* Get the widget's width since we want to limit the needle's
         * position to be inside the control's box */
        int position, width;
        ff_area(ndl,                /* The window to get the area of */
                NULL,               /* X coordinate (unused) */
                NULL,               /* Y coordinate (unused) */
                &width,             /* Window width */
                NULL);              /* Window height (unused) */
        position = (int)(ff_intptr_t)kv->value;

        /* Limit the position to the box */
        if (position < 0) position = 0;
        if (position > width - 3) position = width - 3;

        /* Make sure the final position is really different since we
         * do not want to fire unnecessary events */
        if (position != state->position) {
            /* Update the position */
            state->position = position;

            /* Fire an event informing that the position changed.  We
             * use ff_post instead of ff_send so that the event is
             * sent outside the event handling cycle.  This is generally
             * a good idea for most event ssince an event handler might
             * delete this control or -in some events- cause infinite
             * loops. */
            ff_post(ndl,            /* The event sender */
                    FF_CHANGE,      /* Send the FF_CHANGE event */
                    position,       /* The new position */
                    0, 0, NULL,     /* Unused parameters */
                    FF_NOFLAGS);    /* No further event flags */

            /* Request a repaint */
            ff_paint(ndl);
        }

        /* We handled this event and do not need any further processing
         * of this attribute.  This also causes the attribute to not
         * be stored in the generic name:value store that each object
         * has (we could return FF_NO here to allow that, but we already
         * have allocated storage for the needle's position). */
        return FF_YES;
    }

    /* We do not know about the requested attribute, let Little Forms
     * handle it */
    return FF_NO;
}

/* Called when ff_get() is called on the control to retrive an
 * attribute.  This is very similar to set_handler above. */
static int get_handler(void* ndl, ff_event_t* ev, void* data)
{
    /* Get information about the attribute that was requested */
    ff_namevalue_t* kv = ev->p;

    /* Check if the requested attribute was the needle's position */
    if (!strcmp(kv->name, "position")) {
        /* Get the control's state data */
        state_t* state = ff_get(ndl, "-state");

        /* Store the position to the given structure */
        kv->value = (void*)(ff_intptr_t)state->position;

        /* We handled this event and there is no need for further
         * processing by Little Forms */
        return FF_YES;
    }

    /* We do not know about the requested attribute, let Little Forms
     * handle it */
    return FF_NO;
}

/* Called to paint the control */
static int paint_handler(void* ndl, ff_event_t* ev, void* data)
{
    /* Get the graphics context from the event.  We will use this
     * graphics context with the draw functions. */
    ff_gc_t gc = ev->p;

    /* Get the control's state data */
    state_t* state = ff_get(ndl, "-state");

    /* Obtain the width and height (stored in w and h respectively) of
     * the control */
    int w, h;
    ff_area(ndl, NULL, NULL, &w, &h); /* See set_handler for the args */

    /* Set the current color to the color stored in the "background"
     * attribute in the control or to FF_3DFACE_COLOR if there isn't
     * any attribute with that name set */
    ff_color_attr(gc, ndl, "background", FF_3DFACE_COLOR);

    /* Fill a solid rectangle over the entire window, leaving a single
     * pixel outline */
    ff_fill(gc, 1, 1, w - 2, h - 2);

    /* Draw the needle */
    ff_color(gc, FF_3DDARK_COLOR);
    ff_line(gc, state->position + 1, 4, state->position + 1, h - 2);
    ff_line(gc, state->position, h - 2, state->position + 2, h - 2);

    /* Draw a thin 3D outline around the window */
    ff_color(gc, FF_3DSHADOW_COLOR);
    ff_line(gc, 0, 0, w - 1, 0);
    ff_line(gc, 0, 0, 0, h - 1);
    ff_color(gc, FF_3DLIGHT_COLOR);
    ff_line(gc, 1, h - 1, w - 1, h - 1);
    ff_line(gc, w - 1, 1, w - 1, h - 1);

    /* Done, return FF_YES to indicate that we drew the window */
    return FF_YES;
}

/* Called when any pointer/mouse button is pressed on the control */
static int press_handler(void* ndl, ff_event_t* ev, void* data)
{
    /* The z value is the button number and unless the user has changed
     * the button mapping, they are 1=left, 2=middle, 3=right, 4=wheel
     * up and 5=wheel down */

    /* Was left button pressed? */
    if (ev->z == 1) {
        /* Get the control's state data */
        state_t* state = ff_get(ndl, "-state");

        /* Set the moving bitflag */
        state->moving = FF_YES;

        /* Move the needle to the position that was clicked */
        ff_seti(ndl, "position", ev->x);

        /* We handled the event */
        return FF_YES;
    }

    /* We do not care about this particular event */
    return FF_NO;
}

/* Called when any pointer/mouse button is released that was originally
 * pressed on the control */
static int release_handler(void* ndl, ff_event_t* ev, void* data)
{
    /* Was left button released (see press_handler for details)? */
    if (ev->z == 1) {
        /* Get the control's state data */
        state_t* state = ff_get(ndl, "-state");

        /* Unset the moving bitflag */
        state->moving = FF_NO;

        /* We handled the event */
        return FF_YES;
    }

    /* We do not care about this particular event */
    return FF_NO;
}

/* Called when the pointer/mouse moves.  This is called either when the
 * mouse moves over the control or when it moves while the user has any
 * button down that was previously pressed over the control. */
static int motion_handler(void* ndl, ff_event_t* ev, void* data)
{
    /* Get the control's state data */
    state_t* state = ff_get(ndl, "-state");

    /* If we are moving, update the needle */
    if (state->moving) ff_seti(ndl, "position", ev->x);

    /* We don't need to block any further handling of this event so
     * even though we could have handled it, it doesn't matter.  So
     * just return FF_NO here. */
    return FF_NO;
}

/* Called right before the object is destroyed */
static int destroy_handler(void* ndl, ff_event_t* ev, void* data)
{
    /* Get the control's state data */
    state_t* state = ff_get(ndl, "-state");

    /* Release the data */
    free(state);

    /* We should not block any other linked handlers for this event
     * to execute, otherwise we may end up leaking resources */
    return FF_NO;
}


/* This ff_needle() function will be used to create the control. */
ff_window_t ff_needle(ff_window_t parent)
{
    /* First we create the base window */
    ff_window_t ndl = ff_window(parent,     /* The control's parent */
                                0, 0,       /* X, Y coordinates */
                                100, 16,    /* Width and height */
                                FF_NOFLAGS);/* No flags needed */

    /* We allocate memory for the control's private state */
    state_t* state = calloc(1, sizeof(state_t));
    /* And store it as a private attribute (- prefix make it private) */
    ff_set(ndl, "-state", state);

    /* Also we set the class attribute so that users know what sort of
     * object this control is (this is just a convention and not really
     * necessary, but it often helps when dealing with multiple controls
     * with the same piece of code) */
    ff_setcs(ndl, "class", "needle");

    /* Link handlers for the FF_SET and FF_GET events that will be used
     * to handle the 'position' attribute */
    ff_link(ndl, FF_SET, set_handler, NULL);
    ff_link(ndl, FF_GET, get_handler, NULL);

    /* Link a handler for the FF_PAINT event */
    ff_link(ndl, FF_PAINT, paint_handler, NULL);

    /* Link handlers for the FF_PRESS and FF_RELEASE events which are
     * fired when a pointer/mouse button is pressed/released.  We'll use
     * these to update the moving bitflag */
    ff_link(ndl, FF_PRESS, press_handler, NULL);
    ff_link(ndl, FF_RELEASE, release_handler, NULL);

    /* Link a handler for the FF_MOTION event so that we will update
     * the needle's position when the pointer/mouse moves while the
     * moving bitflag is set */
    ff_link(ndl, FF_MOTION, motion_handler, NULL);

    /* Link a handler for the FF_DESTROY event so that we release the
     * memory that was reserved for the state. */
    ff_link(ndl, FF_DESTROY, destroy_handler, NULL);

    /* Set the preferred size for the control to 100x16 pixels */
    ff_prefsize(ndl, 100, 16);

    /* Done, give back the new control */
    return ndl;
}


/* ====================================================================
 * This part is an example of how the control can be used.  It is not
 * really any different than a control that comes with Little Forms.
 * ================================================================== */

/* Called by the needle custom control when the needle moves */
static int needle_change(void* ndl, ff_event_t* ev, void* data)
{
    /* Find the first label in the parent window (the main window) */
    ff_window_t lbl = ff_find(ff_parent(ndl), "class", FF_ISSTR, "label");

    /* Prepare a message with the new value */
    char tmp[64];
    sprintf(tmp, "Needle moved to %i", (int)ev->x);
    ff_set(lbl, "caption", tmp);

    /* We handled the event */
    return FF_YES;
}

/* Entry point */
int main(int argc, char** argv)
{
    ff_window_t main_frame; /* The main frame (window) */
    ff_window_t needle;     /* The needle custom control we made */

    /* Initialize */
    if (!ff_init(argc, argv, "Custom Little Forms Control"))
        return 1;

    /* Create main frame */
    main_frame = ff_mainframe("Custom Little Forms Control", NULL,
                              320, 320,
                              FF_INVISIBLE);    /* Start invisible */

    /* Set the layout to a vertical stack */
    ff_layout(main_frame, ff_stack);
    ff_seti(main_frame, "padding-around", 15);
    ff_seti(main_frame, "stack-spacing", 6);

    /* Create a label control */
    ff_label(main_frame, "Use the left button to move the needle:");

    /* Create the needle control */
    needle = ff_needle(main_frame);

    /* Set a mellow color, overriding the default FF_3DFACE_COLOR */
    ff_set(needle, "background", FF_RGB_ATTR(248, 222, 126));

    /* Link the FF_CHANGE event to the needle_change which will be
     * called by the custom control when the needle moves (see the code
     * in set_handler for this) */
    ff_link(needle, FF_CHANGE, needle_change, NULL);

    /* Pack the window both horizontally and vertically to fit
     * the preferred size of its chilren */
    ff_pack(main_frame, FF_BOTH);

    /* Show the main frame */
    ff_show(main_frame, FF_YES);

    /* Enter main loop */
    return ff_run();
}
