unit RBCpl;
{$MODE OBJFPC}{$H+}
interface
uses
//  HeapTrc,
  SysUtils, RBOwn, RBError, RBProg, RBCode, RBSrc, RBScan, RBNative;

type
  ECompileError = class(Exception)
  public
    Source: TSource;
    Col, Row: Integer;
    constructor New(ASource: TSource; ARow, ACol: Integer; AMessage: string);
  end;

  { TCompiler }

  TCompiler = class(TOwnable)
  private
    FSources: array of TSource;
    FProg: TProgram;
    Source: TSource;
    Scanner: TScanner;
    Code: TCode;
    Vars: PVarList;
    CurMethod: TMethod;
    PrePass: Boolean;
    InsideIf, InsideFor, InsideWhile: Boolean;
    procedure RaiseError(const AToken: TToken; AMessage: string);
    procedure RaiseError(AMessage: string);
    procedure ScanOrDie;
    function IsEndOf(Special: TTokenType): Boolean;
    function FindVar(AName: string; out Global: Boolean): Integer;
    function TempVar: Integer;
    function EndOfStatement: Boolean;
    procedure CompileExpression;
    procedure CompilePrintStatement;
    procedure CompileAssignment(ObjectAssignment: Boolean);
    procedure CompileCall(ObjectMethod: Boolean);
    procedure CompileObjectUse;
    procedure CompileIdentifierUse;
    procedure CompileDeclaration;
    procedure CompileDim;
    procedure CompileMethod;
    procedure CompileIf;
    procedure CompileFor;
    procedure CompileWhile;
    procedure CompileStatement;
    procedure CompileBlock;
    procedure CompileSource(ASource: TSource);
  public
    procedure Reset;
    procedure AddSource(ASource: TSource);
    procedure Compile;
    property Prog: TProgram read FProg write FProg;
  end;

implementation

{ ECompileError }
constructor ECompileError.New(ASource: TSource; ARow, ACol: Integer; AMessage: string);
begin
  Source:=ASource;
  Col:=ACol;
  Row:=ARow;
  inherited Create(AMessage);
end;

{ TCompiler }
procedure TCompiler.RaiseError(const AToken: TToken; AMessage: string);
var
  Row, Col: Integer;
begin
  Source.GetRowAndColumn(AToken.Position, Row, Col);
  raise ECompileError.New(Source, Row, Col, AMessage);
end;

procedure TCompiler.RaiseError(AMessage: string);
begin
  RaiseError(Scanner.Token, AMessage);
end;

procedure TCompiler.ScanOrDie;
begin
  if not Scanner.Scan then RaiseError(emUnexpectedEndOfCode);
end;

function TCompiler.IsEndOf(Special: TTokenType): Boolean;
begin
  Result:=(Scanner.Token.TokenType=ttEnd) and (Scanner.Peek.TokenType=Special);
end;

function TCompiler.FindVar(AName: string; out Global: Boolean): Integer;
begin
  if Assigned(CurMethod) then begin
    Result:=CurMethod.Vars.Find(AName);
    Global:=False;
    if Result <> -1 then Exit;
  end;

  Global:=True;
  Result:=Prog.Vars^.Find(AName);
  if Result <> -1 then Exit;

  if Assigned(CurMethod) then begin
    Global:=False;
    Result:=CurMethod.Vars.Add(AName);
    Exit;
  end;

  Global:=True;
  Result:=Prog.Vars^.Add(AName);
end;

function TCompiler.TempVar: Integer;
var
  VarName: string;
begin
  while True do begin
    VarName:=' t' + IntToStr(Code.Address) + 'v' + IntToStr(Random(492893232));
    Result:=Vars^.Find(VarName);
    if Result=-1 then Break;
  end;
  Result:=Vars^.Add(VarName);
end;

function TCompiler.EndOfStatement: Boolean;
begin
  Result:=(Scanner.Token.TokenType in [ttInvalid, ttEndOfLine, ttEnd]) or
          (InsideIf and (Scanner.Token.TokenType=ttElse)) or
          (InsideFor and (Scanner.Token.TokenType=ttNext)) or
          (InsideWhile and (Scanner.Token.TokenType=ttWend));
end;

procedure TCompiler.CompileExpression;

  procedure DoExpression; forward;

  procedure DoVariable;
  var
    VarPos, Slot, Indices: Integer;
    Global: Boolean;
  begin
    Slot:=FindVar(Scanner.Token.StrValue, Global);
    VarPos:=Scanner.Token.Position;
    if Scanner.NextChar='(' then begin
      ScanOrDie;
      Indices:=0;
      while Scanner.Scan do begin
        if Scanner.Token.TokenType=ttRightParen then Break;
        CompileExpression;
        Inc(Indices);
        if Scanner.Token.TokenType=ttRightParen then Break;
        if Scanner.Token.TokenType <> ttComma then RaiseError(emFmt(emCommaExpectedButXFound, Scanner.Token.ToString));
      end;
      if Scanner.Token.TokenType <> ttRightParen then RaiseError(emClosingParenthesisExpected);
      if (Indices < 1) or (Indices > 6) then RaiseError(emInvalidDimensionCount);
      Code.AddPosition(Source, VarPos);
      if Global then Code.AddOpCode(opDimGRead) else Code.AddOpCode(opDimLRead);
      Code.AddByte(Indices);
      Code.AddCardinal(Cardinal(Slot));
    end else begin
      Code.AddPosition(Source, VarPos);
      if Global then Code.AddOpCode(opGRead) else Code.AddOpCode(opLRead);
      Code.AddCardinal(Cardinal(Slot));
    end;
  end;

  procedure DoMethod(Method: TMethod; Native: TNativeMethodInfo; ObjectMethod: string);
  var
    ArgCount, MetPos, ExpPos: Integer;
  begin
    MetPos:=Scanner.Token.Position;
    if ObjectMethod='' then begin
      if not Assigned(Method) and not Assigned(Native) then Method:=Prog.FindMethod(Scanner.Token.StrValue);
      if Assigned(Method) and not Method.Func then
        RaiseError(emFmt(emCannotUseSubXAsFunction, Method.Name));
      if Assigned(Native) and (Native.MethodType=mtSub) then
        RaiseError(emFmt(emCannotUseSubXAsFunction, Native.MetName));
    end;

    ArgCount:=0;

    if Scanner.Peek.TokenType=ttLeftParen then begin
      Scanner.Scan;
      while Scanner.Scan do begin
        if Scanner.Token.TokenType=ttRightParen then Break;
        ExpPos:=Scanner.Token.Position;
        CompileExpression;
        Inc(ArgCount);
        Code.AddPosition(Source, ExpPos);
        Code.AddOpCode(opPushArg);
        if Scanner.Token.TokenType=ttRightParen then Break;
        if Scanner.Token.TokenType <> ttComma then RaiseError(emFmt(emCommaExpectedButXFound, Scanner.Token.ToString));
      end;
      if Scanner.Token.TokenType <> ttRightParen then RaiseError(emClosingParenthesisExpected);
    end;

    Code.AddPosition(Source, MetPos);
    if ObjectMethod <> '' then begin
      Code.AddOpCode(opObjCall);
      Code.AddCardinal(Cardinal(Prog.StringDB.StringToID(ObjectMethod)));
      Code.AddCardinal(Cardinal(ArgCount));
    end else if Assigned(Method) then begin
      if ArgCount <> Method.ArgCount then RaiseError(emFmt(emInvalidArgumentCountToX, Method.Name));
      Code.AddOpCode(opMetCall);
      Code.AddCardinal(Method.ID);
    end else if Assigned(Native) then begin
      Code.AddOpCode(opNatCall);
      Code.AddCardinal(Cardinal(ArgCount));
      Code.AddCardinal(Cardinal(Native.Index));
    end else begin
      Code.AddOpCode(opMetCall);
      Code.AddCardinal(0);
    end;
  end;

  procedure DoObjectFunction;
  var
    VarPos, Slot, Indices: Integer;
    Global: Boolean;
  begin
    Slot:=FindVar(Scanner.Token.StrValue, Global);
    VarPos:=Scanner.Token.Position;
    Code.AddPosition(Source, VarPos);
    if Global then Code.AddOpCode(opGRead) else Code.AddOpCode(opLRead);
    Code.AddCardinal(Cardinal(Slot));
    ScanOrDie;
    ScanOrDie;
    if Scanner.Token.TokenType <> ttIdentifier then
      RaiseError(emFmt(emIdentifierExpectedButXFound, Scanner.Token.ToString));
    DoMethod(nil, nil, Scanner.Token.StrValue);
  end;

  procedure DoIdentifier;
  var
    Name: string;
    Method: TMethod;
    Native: TNativeMethodInfo;
  begin
    Name:=Scanner.Token.StrValue;

    if Name='TRUE' then begin
      Code.AddPosition(Source, Scanner.Token.Position);
      Code.AddOpCode(opPushInt);
      Code.AddInteger(-1);
      Exit;
    end;
    if Name='FALSE' then begin
      Code.AddPosition(Source, Scanner.Token.Position);
      Code.AddOpCode(opPushZero);
      Exit;
    end;

    if Scanner.Peek.TokenType=ttPeriod then begin
      DoObjectFunction;
      Exit;
    end;

    Prog.NativeInterface.Find(Name, Native);
    Method:=Prog.FindMethod(Name);
    if Assigned(Native) then begin
      DoMethod(nil, Native, '');
    end else if Assigned(Method) then begin
      DoMethod(Method, nil, '');
    end else begin
      if Scanner.Peek.TokenType=ttLeftParen then begin
        if PrePass then
          DoMethod(nil, nil, '')
        else
          DoVariable;
      end else begin
        DoVariable;
      end;
    end;
  end;

  procedure DoValue;
  begin
    Code.AddPosition(Source, Scanner.Token.Position);
    case Scanner.Token.TokenType of
      ttInteger: begin
        Code.AddOpCode(opPushInt);
        Code.AddInteger(Scanner.Token.IntValue);
      end;
      ttReal: begin
        Code.AddOpCode(opPushReal);
        Code.AddReal(Scanner.Token.RealValue);
      end;
      ttString: begin
        Code.AddOpCode(opPushStr);
        Code.AddString(Scanner.Token.StrValue);
      end;
      ttIdentifier: DoIdentifier;
      else RaiseError(emFmt(emExpectedValueButXFound, Scanner.Token.ToString));
    end;
    ScanOrDie;
  end;

  procedure DoParentheses;
  begin
    if Scanner.Token.TokenType=ttLeftParen then begin
      ScanOrDie;
      DoExpression;
      if Scanner.Token.TokenType <> ttRightParen then
        RaiseError(emClosingParenthesisExpected);
      ScanOrDie;
    end else DoValue;
  end;

  procedure DoNegation;
  var
    Pos: Integer;
  begin
    if Scanner.Token.TokenType=ttMinus then begin
      Pos:=Scanner.Token.Position;
      ScanOrDie;
      DoParentheses;
      Code.AddPosition(Source, Pos);
      Code.AddOpCode(opNeg);
    end else DoParentheses;
  end;

  procedure DoPower;
  var
    Pos: Integer;
  begin
    DoNegation;
    while Scanner.Token.TokenType=ttCaret do begin
      ScanOrDie;
      Pos:=Scanner.Token.Position;
      DoNegation;
      Code.AddPosition(Source, Pos);
      Code.AddOpCode(opPow);
    end;
  end;

  procedure DoMulDiv;
  var
    TT: TTokenType;
    Pos: Integer;
  begin
    DoPower;
    while Scanner.Token.TokenType in [ttStar, ttSlash] do begin
      TT:=Scanner.Token.TokenType;
      ScanOrDie;
      Pos:=Scanner.Token.Position;
      DoPower;
      Code.AddPosition(Source, Pos);
      case TT of
        ttStar: Code.AddOpCode(opMul);
        ttSlash: Code.AddOpCode(opDiv);
      end;
    end;
  end;

  procedure DoIntDiv;
  var
    Pos: Integer;
  begin
    DoMulDiv;
    while Scanner.Token.TokenType=ttBackSlash do begin
      ScanOrDie;
      Pos:=Scanner.Token.Position;
      DoMulDiv;
      Code.AddPosition(Source, Pos);
      Code.AddOpCode(opIntDiv);
    end;
  end;

  procedure DoMod;
  var
    Pos: Integer;
  begin
    DoIntDiv;
    while Scanner.Token.TokenType=ttMod do begin
      ScanOrDie;
      Pos:=Scanner.Token.Position;
      DoIntDiv;
      Code.AddPosition(Source, Pos);
      Code.AddOpCode(opMod);
    end;
  end;

  procedure DoAddSub;
  var
    TT: TTokenType;
    Pos: Integer;
  begin
    DoMod;
    while Scanner.Token.TokenType in [ttPlus, ttMinus] do begin
      TT:=Scanner.Token.TokenType;
      Pos:=Scanner.Token.Position;
      ScanOrDie;
      DoMod;
      Code.AddPosition(Source, Pos);
      case TT of
        ttPlus: Code.AddOpCode(opAdd);
        ttMinus: Code.AddOpCode(opSub);
      end;
    end;
  end;

  procedure DoEquality;
  var
    TT: TTokenType;
    Pos: Integer;
  begin
    DoAddSub;
    while Scanner.Token.TokenType in [ttEqual, ttInequal, ttLess, ttGreat, ttLessEqual, ttGreatEqual] do begin
      TT:=Scanner.Token.TokenType;
      Pos:=Scanner.Token.Position;
      ScanOrDie;
      DoAddSub;
      Code.AddPosition(Source, Pos);
      case TT of
        ttEqual: Code.AddOpCode(opEq);
        ttInequal: Code.AddOpCode(opNEq);
        ttLess: Code.AddOpCode(opLess);
        ttGreat: Code.AddOpCode(opGreat);
        ttLessEqual: Code.AddOpCode(opLEqual);
        ttGreatEqual: Code.AddOpCode(opGEqual);
      end;
    end;
  end;

  procedure DoNot;
  var
    Pos: integer;
  begin
    if Scanner.Token.TokenType=ttNot then begin
      ScanOrDie;
      Pos:=Scanner.Token.Position;
      DoEquality;
      Code.AddPosition(Source, Pos);
      Code.AddOpCode(opNot);
    end else DoEquality;
  end;

  procedure DoAnd;
  var
    Pos: Integer;
  begin
    DoNot;
    while Scanner.Token.TokenType=ttAnd do begin
      ScanOrDie;
      Pos:=Scanner.Token.Position;
      DoNot;
      Code.AddPosition(Source, Pos);
      Code.AddOpCode(opAnd);
    end;
  end;

  procedure DoOr;
  var
    Pos: Integer;
  begin
    DoAnd;
    while Scanner.Token.TokenType=ttOr do begin
      ScanOrDie;
      Pos:=Scanner.Token.Position;
      DoAnd;
      Code.AddPosition(Source, Pos);
      Code.AddOpCode(opOr);
    end;
  end;

  procedure DoXOr;
  var
    Pos: Integer;
  begin
    DoOr;
    while Scanner.Token.TokenType=ttXOr do begin
      ScanOrDie;
      Pos:=Scanner.Token.Position;
      DoOr;
      Code.AddPosition(Source, Pos);
      Code.AddOpCode(opXOr);
    end;
  end;

  procedure DoEqv;
  begin
    DoXOr;
  end;

  procedure DoImp;
  begin
    DoEqv;
  end;

  procedure DoExpression;
  begin
    DoImp;
  end;

begin
  DoExpression;
end;

procedure TCompiler.CompilePrintStatement;
var
  AddNewLine: Boolean = True;
  Pos: Integer;
begin
  while Scanner.Scan do begin
    if EndOfStatement then Break;
    AddNewLine:=True;
    Pos:=Scanner.Token.Position;
    CompileExpression;
    Code.AddPosition(Source, Pos);
    Code.AddOpCode(opPrint);
    if EndOfStatement then Break;
    if not (Scanner.Token.TokenType in [ttSemicolon, ttComma]) then
      RaiseError(emFmt(emUnexpectedToken, Scanner.Token.ToString));
    if Scanner.Token.TokenType=ttComma then Code.AddOpCode(opTab) else AddNewLine:=False;
  end;
  if not EndOfStatement then RaiseError(emUnexpectedEndOfCode);
  if AddNewLine then Code.AddOpCode(opNewLine);
end;

procedure TCompiler.CompileAssignment(ObjectAssignment: Boolean);
var
  Pos, Slot, Indices: Integer;
  Global: Boolean;
  PropertyName: string;
begin
  if ObjectAssignment then
    PropertyName:=Scanner.Token.StrValue
  else
    Slot:=FindVar(Scanner.Token.StrValue, Global);
  ScanOrDie;
  if Scanner.Token.TokenType=ttEqual then begin
    ScanOrDie;
    Pos:=Scanner.Token.Position;
    CompileExpression;
    Code.AddPosition(Source, Pos);
    if ObjectAssignment then begin
      Code.AddOpCode(opPWrite);
      Code.AddCardinal(Cardinal(Prog.StringDB.StringToID(PropertyName)));
    end else begin
      if Global then Code.AddOpCode(opGWrite) else Code.AddOpCode(opLWrite);
      Code.AddCardinal(Cardinal(Slot));
    end;
  end else begin // left paren
    Indices:=0;
    while Scanner.Scan do begin
      if Scanner.Token.TokenType=ttRightParen then Break;
      CompileExpression;
      Inc(Indices);
      if Scanner.Token.TokenType=ttRightParen then Break;
      if Scanner.Token.TokenType <> ttComma then RaiseError(emFmt(emCommaExpectedButXFound, Scanner.Token.ToString));
    end;
    if Scanner.Token.TokenType <> ttRightParen then RaiseError(emClosingParenthesisExpected);
    if (Indices < 1) or (Indices > 6) then RaiseError(emInvalidDimensionCount);
    ScanOrDie;
    if Scanner.Token.TokenType <> ttEqual then RaiseError(emEqualExpectedInArrayAssign);
    ScanOrDie;
    Pos:=Scanner.Token.Position;
    CompileExpression;
    Code.AddPosition(Source, Pos);
    if Global then Code.AddOpCode(opDimGWrite) else Code.AddOpCode(opDimLWrite);
    Code.AddByte(Indices);
    Code.AddCardinal(Cardinal(Slot));
  end;
end;

procedure TCompiler.CompileCall(ObjectMethod: Boolean);
var
  Method: TMethod = nil;
  Native: TNativeMethodInfo;
  MetPos, Pos, ArgCount: Integer;
  MetName: string;
begin
  MetPos:=Scanner.Token.Position;
  if not ObjectMethod then begin
    if not Prog.NativeInterface.Find(Scanner.Token.StrValue, Native) then begin
      Method:=Prog.FindMethod(Scanner.Token.StrValue);
      if not Assigned(Method) and not PrePass then
        RaiseError(emFmt(emUnknownMethod, Scanner.Token.StrValue));
    end;
  end else begin
    MetName:=Scanner.Token.StrValue;
  end;
  ArgCount:=0;

  while Scanner.Scan do begin
    if EndOfStatement then Break;
    Pos:=Scanner.Token.Position;
    CompileExpression;
    Inc(ArgCount);
    Code.AddPosition(Source, Pos);
    Code.AddOpCode(opPushArg);
    if EndOfStatement then Break;
    if Scanner.Token.TokenType <> ttComma then RaiseError(emFmt(emCommaExpectedButXFound, Scanner.Token.ToString));
  end;
  if not EndOfStatement then RaiseError(emFmt(emUnexpectedToken, Scanner.Token.ToString));
  Code.AddPosition(Source, MetPos);
  if ObjectMethod then begin
    Code.AddOpCode(opObjCall);
    Code.AddCardinal(Cardinal(Prog.StringDB.StringToID(MetName)));
    Code.AddCardinal(Cardinal(ArgCount));
    Code.AddOpCode(opDrop);
  end else if Assigned(Method) then begin
    if ArgCount <> Method.ArgCount then RaiseError(emFmt(emInvalidArgumentCountToX, Method.Name));
    Code.AddOpCode(opMetCall);
    Code.AddCardinal(Method.ID);
    if Method.Func then Code.AddOpCode(opDrop);
  end else if Assigned(Native) then begin
    Code.AddOpCode(opNatCall);
    Code.AddCardinal(Cardinal(ArgCount));
    Code.AddCardinal(Cardinal(Native.Index));
    if Native.MethodType=mtFunction then Code.AddOpCode(opDrop);
  end else begin
    Code.AddOpCode(opMetCall);
    Code.AddCardinal(0);
  end;
end;

procedure TCompiler.CompileObjectUse;
var
  Slot: Integer;
  Global: Boolean;
begin
  Slot:=FindVar(Scanner.Token.StrValue, Global);
  ScanOrDie;
  Code.AddPosition(Source, Scanner.Token.Position);
  if Global then Code.AddOpCode(opGRead) else Code.AddOpCode(opLRead);
  Code.AddCardinal(Cardinal(Slot));
  ScanOrDie;
  if Scanner.Token.TokenType <> ttIdentifier then RaiseError(emFmt(emIdentifierExpectedButXFound, Scanner.Token.ToString));
  if Scanner.Peek.TokenType=ttEqual then
    CompileAssignment(True)
  else
    CompileCall(True);
end;

procedure TCompiler.CompileIdentifierUse;
begin
  if Scanner.Token.StrValue='PRINT' then begin
    CompilePrintStatement;
    Exit;
  end;

  if (Scanner.Peek.TokenType=ttEqual) or (Scanner.NextChar='(') then begin
    CompileAssignment(False);
    Exit;
  end;

  if Scanner.Peek.TokenType=ttPeriod then begin
    CompileObjectUse;
    Exit;
  end;

  CompileCall(False);
end;

procedure TCompiler.CompileDeclaration;
begin
end;

procedure TCompiler.CompileDim;
var
  ReDim: Boolean;
  VarPos, Dimensions, Index: Integer;
  Global: Boolean;
begin
  ReDim:=Scanner.Token.TokenType=ttReDim;
  while Scanner.Scan do begin
    if Scanner.Token.TokenType <> ttIdentifier then RaiseError(emFmt(emIdentifierExpectedButXFound, Scanner.Token.ToString));
    VarPos:=Scanner.Token.Position;
    Index:=FindVar(Scanner.Token.StrValue, Global);
    ScanOrDie;
    if Scanner.Token.TokenType <> ttLeftParen then RaiseError(emOpeningParenthesisExpected);
    Dimensions:=0;
    while Scanner.Scan do begin
      CompileExpression;
      Inc(Dimensions);
      if not (Scanner.Token.TokenType in [ttRightParen, ttComma]) then
        RaiseError(emFmt(emCommaExpectedButXFound, Scanner.Token.ToString));
      if Scanner.Token.TokenType=ttRightParen then Break;
    end;
    if Scanner.Token.TokenType <> ttRightParen then RaiseError(emClosingParenthesisExpected);
    if (Dimensions < 0) or (Dimensions > 6) then RaiseError(emInvalidDimensionCount);
    Code.AddPosition(Source, VarPos);
    if Global then begin
      if ReDim then Code.AddOpCode(opGReDim) else Code.AddOpCode(opGDim);
    end else begin
      if ReDim then Code.AddOpCode(opLReDim) else Code.AddOpCode(opLDim);
    end;
    Code.AddByte(Byte(Dimensions));
    Code.AddCardinal(Cardinal(Index));
    ScanOrDie;
    if (Scanner.Token.TokenType <> ttComma) and not EndOfStatement then
      RaiseError(emFmt(emCommaExpectedButXFound, Scanner.Token.ToString));
    if EndOfStatement then Break;
  end;
end;

procedure TCompiler.CompileMethod;
var
  IsFunction: Boolean;
  NameToken: TToken;

  procedure ParseArguments;
  var
    ArgNames: array of TToken;
    I: Integer;
  begin
    Scanner.Scan;
    SetLength(ArgNames, 0);
    while Scanner.Scan do begin
      if Scanner.Token.TokenType <> ttIdentifier then
        RaiseError(emFmt(emIdentifierExpectedButXFound, Scanner.Token.ToString));
      for I:=0 to High(ArgNames) do
        if ArgNames[I].StrValue=Scanner.Token.StrValue then
          RaiseError(emFmt(emDuplicateDefinitionOfX, Scanner.Token.StrValue));
      SetLength(ArgNames, Length(ArgNames) + 1);
      ArgNames[High(ArgNames)]:=Scanner.Token;
      ScanOrDie;
      if not (Scanner.Token.TokenType in [ttComma, ttRightParen]) then
        RaiseError(emFmt(emCommaExpectedButXFound, Scanner.Token.ToString));
      if Scanner.Token.TokenType=ttRightParen then Break;
    end;
    if Scanner.Token.TokenType <> ttRightParen then RaiseError(emUnexpectedEndOfCode);
    for I:=0 to High(ArgNames) do CurMethod.Vars.Add(ArgNames[I].StrValue);
    CurMethod.ArgCount:=Length(ArgNames);
  end;

begin
  IsFunction:=Scanner.Token.TokenType=ttFunction;
  ScanOrDie;
  if Scanner.Token.TokenType <> ttIdentifier then
    RaiseError(emFmt(emIdentifierExpectedButXFound, Scanner.Token.ToString));
  NameToken:=Scanner.Token;

  try
    CurMethod:=Prog.FindMethod(NameToken.StrValue);
    if PrePass and Assigned(CurMethod) then RaiseError(NameToken, emFmt(emMethodAlreadyDefined, NameToken.StrValue));
    if not Assigned(CurMethod) then begin
      CurMethod:=TMethod.Create;
      Prog.AddMethod(CurMethod);
    end;
    CurMethod.Name:=NameToken.StrValue;
    CurMethod.Func:=IsFunction;
    if Scanner.Peek.TokenType=ttLeftParen then ParseArguments;
    if CurMethod.Func then CurMethod.Vars.Add(CurMethod.Name);
    Code:=CurMethod.Code;
    Vars:=@CurMethod.Vars;
    if IsFunction then begin
      Code.AddOpCode(opPushZero);
      Code.AddOpCode(opLWrite);
      Code.AddCardinal(Cardinal(CurMethod.ArgCount));
    end;
    CompileBlock;
  except
    Code:=Prog.Code;
    Vars:=Prog.Vars;
    if Assigned(CurMethod) then Prog.RemoveMethod(CurMethod);
    raise;
  end;
  Code:=Prog.Code;
  Vars:=Prog.Vars;
  CurMethod:=nil;
end;

procedure TCompiler.CompileIf;
var
  CheckForElse, SaveIf: Boolean;
  IfPos, JumpIfNotAddr, JumpOverElseAddr: Integer;
begin
  IfPos:=Scanner.Token.Position;
  ScanOrDie;
  CompileExpression;
  if Scanner.Token.TokenType <> ttThen then RaiseError(emFmt(emThenExpectedButXFound, Scanner.Token.ToString));
  Code.AddPosition(Source, IfPos);
  Code.AddOpCode(opIfNot);
  JumpIfNotAddr:=Code.Address;
  Code.AddCardinal(0);
  ScanOrDie;
  SaveIf:=InsideIf;
  InsideIf:=True;
  CheckForElse:=True;
  if Scanner.Token.TokenType=ttEndOfLine then begin
    CompileBlock;
    if not ((Scanner.Token.TokenType=ttElse) or IsEndOf(ttIf)) then RaiseError(emMissingEndIf);
    if IsEndOf(ttIf) then begin
      Scanner.Scan;
      CheckForElse:=False;
    end;
  end else CompileStatement;
  if CheckForElse and (Scanner.Token.TokenType=ttElse) then begin
    Code.AddOpCode(opJump);
    JumpOverElseAddr:=Code.Address;
    Code.AddCardinal(0);
    Code.SetCardinal(JumpIfNotAddr, Code.Address);
    ScanOrDie;
    if Scanner.Token.TokenType=ttEndOfLine then begin
      CompileBlock;
      if not IsEndOf(ttIf) then RaiseError(emMissingEndIf);
      Scanner.Scan;
    end else CompileStatement;
    Code.SetCardinal(JumpOverElseAddr, Code.Address);
  end else Code.SetCardinal(JumpIfNotAddr, Code.Address);
  InsideIf:=SaveIf;
end;

procedure TCompiler.CompileFor;
var
  VarToken: TToken;
  ForPos, EnumVar, TargetVar, StepVar, LoopStartAddr, LoopSkipAddr: Integer;
  Global, SaveFor: Boolean;
begin
  ForPos:=Scanner.Token.Position;
  TargetVar:=TempVar;
  StepVar:=TempVar;
  ScanOrDie; // <varname>
  if Scanner.Token.TokenType <> ttIdentifier then RaiseError(emVariableNameExpectedInFor);
  VarToken:=Scanner.Token;
  ScanOrDie; // =
  if Scanner.Token.TokenType <> ttEqual then RaiseError(emEqualExpectedAfterVarInFor);
  ScanOrDie; // <start> To
  CompileExpression;
  EnumVar:=FindVar(VarToken.StrValue, Global);
  Code.AddPosition(Source, ForPos);
  if Global then Code.AddOpCode(opGWrite) else Code.AddOpCode(opLWrite);
  Code.AddCardinal(Cardinal(EnumVar));
  if Scanner.Token.TokenType <> ttTo then RaiseError(emToExpectedAfterExprInFor);
  ScanOrDie; // <end> ...
  CompileExpression;
  Code.AddPosition(Source, ForPos);
  if Vars=Prog.Vars then Code.AddOpCode(opGWrite) else Code.AddOpCode(opLWrite);
  Code.AddCardinal(Cardinal(TargetVar));
  if Scanner.Token.TokenType=ttStep then begin
    ScanOrDie;
    CompileExpression;
  end else begin
    Code.AddOpCode(opPushInt);
    Code.AddInteger(1);
  end;
  Code.AddPosition(Source, ForPos);
  if Vars=Prog.Vars then Code.AddOpCode(opGWrite) else Code.AddOpCode(opLWrite);
  Code.AddCardinal(Cardinal(StepVar));
  if Scanner.Token.TokenType <> ttEndOfLine then RaiseError(emFmt(emNewLineExpectedButXFound, Scanner.Token.ToString));
  SaveFor:=InsideFor;
  InsideFor:=True;
  Code.AddPosition(Source, ForPos);
  if Global then Code.AddOpCode(opGRead) else Code.AddOpCode(opLRead);
  Code.AddCardinal(Cardinal(EnumVar));
  if Vars=Prog.Vars then Code.AddOpCode(opGRead) else Code.AddOpCode(opLRead);
  Code.AddCardinal(Cardinal(StepVar));
  Code.AddOpCode(opSub);
  if Global then Code.AddOpCode(opGWrite) else Code.AddOpCode(opLWrite);
  Code.AddCardinal(Cardinal(EnumVar));
  if Global then Code.AddOpCode(opGRead) else Code.AddOpCode(opLRead);
  Code.AddCardinal(Cardinal(EnumVar));
  if Vars=Prog.Vars then Code.AddOpCode(opGRead) else Code.AddOpCode(opLRead);
  Code.AddCardinal(Cardinal(StepVar));
  if Vars=Prog.Vars then Code.AddOpCode(opGRead) else Code.AddOpCode(opLRead);
  Code.AddCardinal(Cardinal(TargetVar));
  if Global then Code.AddOpCode(opGFor) else Code.AddOpCode(opLFor);
  LoopStartAddr:=Code.Address;
  Code.AddCardinal(Cardinal(0));
  Code.AddCardinal(Cardinal(EnumVar));
  Code.AddPosition(Source, ForPos);
  Code.AddOpCode(opJump);
  LoopSkipAddr:=Code.Address;
  Code.AddCardinal(0);

  Code.SetCardinal(LoopStartAddr, Code.Address);
  LoopStartAddr:=Code.Address;
  CompileBlock;

  if Scanner.Token.TokenType <> ttNext then RaiseError(emFmt(emNextExpectedButXFound, Scanner.Token.ToString));
  Code.AddPosition(Source, ForPos);
  if Global then Code.AddOpCode(opGRead) else Code.AddOpCode(opLRead);
  Code.AddCardinal(Cardinal(EnumVar));
  if Vars=Prog.Vars then Code.AddOpCode(opGRead) else Code.AddOpCode(opLRead);
  Code.AddCardinal(Cardinal(StepVar));
  if Vars=Prog.Vars then Code.AddOpCode(opGRead) else Code.AddOpCode(opLRead);
  Code.AddCardinal(Cardinal(TargetVar));
  if Global then Code.AddOpCode(opGFor) else Code.AddOpCode(opLFor);
  Code.AddCardinal(Cardinal(LoopStartAddr));
  Code.AddCardinal(Cardinal(EnumVar));
  Code.SetCardinal(LoopSkipAddr, Code.Address);
  InsideFor:=SaveFor;
end;

procedure TCompiler.CompileWhile;
var
  SaveWhile: Boolean;
  WhilePos, LoopStartAddr, LoopSkipAddr: Integer;
begin
  WhilePos:=Scanner.Token.Position;
  SaveWhile:=InsideWhile;
  InsideWhile:=True;
  ScanOrDie;
  LoopStartAddr:=Code.Address;
  CompileExpression;
  Code.AddPosition(Source, WhilePos);
  Code.AddOpCode(opIfNot);
  LoopSkipAddr:=Code.Address;
  Code.AddCardinal(0);
  if Scanner.Token.TokenType <> ttEndOfLine then RaiseError(emFmt(emNewLineExpectedButXFound, Scanner.Token.ToString));
  CompileBlock;
  if Scanner.Token.TokenType <> ttWend then RaiseError(emFmt(emWendExpectedButXFound, Scanner.Token.ToString));
  Code.AddPosition(Source, WhilePos);
  Code.AddOpCode(opJump);
  Code.AddCardinal(Cardinal(LoopStartAddr));
  Code.SetCardinal(LoopSkipAddr, Code.Address);
  InsideWhile:=SaveWhile;
end;

procedure TCompiler.CompileStatement;
begin
  case Scanner.Token.TokenType of
    ttSub, ttFunction:
      if Assigned(CurMethod) then
        RaiseError(emCannotDefineMethodInMethod)
      else
        CompileMethod;
    ttGlobal: CompileDeclaration;
    ttDim, ttReDim: CompileDim;
    ttIf: CompileIf;
    ttFor: CompileFor;
    ttWhile: CompileWhile;
    ttIdentifier: CompileIdentifierUse;
    else RaiseError(emFmt(emUnexpectedToken, Scanner.Token.ToString));
  end;
end;

procedure TCompiler.CompileBlock;
var
  TT: TTokenType;
begin
  while Scanner.Scan do begin
    case Scanner.Token.TokenType of
      ttEndOfLine: Continue;
      ttEnd: begin
        TT:=Scanner.Peek.TokenType;
        if TT in [ttSub, ttFunction] then begin
          if not Assigned(CurMethod) then
            RaiseError(emUnexpectedUseOfEnd);
          Scanner.Scan;
          if CurMethod.Func <> (Scanner.Token.TokenType=ttFunction) then
            RaiseError(emInvalidEndUsage);
          if CurMethod.Func then begin
            Code.AddOpCode(opLRead);
            Code.AddCardinal(Cardinal(CurMethod.ArgCount));
          end;
        end else if TT=ttIf then begin
          if not InsideIf then RaiseError(emUnexpectedUseOfEnd);
          Break;
        end;
        Code.AddOpCode(opEnd);
        Break;
      end;
      ttElse: begin
        if not InsideIf then RaiseError(emElseOutsideOfIf);
        Break;
      end;
      ttNext: begin
        if not InsideFor then RaiseError(emNextOutsideOfFor);
        Break;
      end;
      ttWend: begin
        if not InsideWhile then RaiseError(emWendOutsideOFWhile);
        Break;
      end;
      else CompileStatement;
    end;
  end;
end;

procedure TCompiler.CompileSource(ASource: TSource);
begin
  try
    Source:=ASource;
    Scanner:=TScanner.Create(Self);
    Scanner.Code:=ASource.SourceCode;
    Code:=Prog.Code;
    Vars:=Prog.Vars;
    CompileBlock;
  finally
    Scanner.Free;
  end;
end;

procedure TCompiler.Reset;
begin
  SetLength(FSources, 0);
  CurMethod:=nil;
end;

procedure TCompiler.AddSource(ASource: TSource);
begin
  SetLength(FSources, Length(FSources) + 1);
  FSources[High(FSources)]:=ASource;
end;

procedure TCompiler.Compile;

  procedure Pass;
  var
    I: Integer;
    Err: ECompileError;
  begin
    Prog.Code.Reset;
    for I:=0 to High(FSources) do begin
      try
        CompileSource(FSources[I]);
      except
        on EScanError do begin
          Err:=ECompileError.Create(EScanError(ExceptObject).Message);
          Err.Source:=FSources[I];
          Err.Row:=EScanError(ExceptObject).Row;
          Err.Col:=EScanError(ExceptObject).Col;
          raise Err;
        end;
      end;
    end;
  end;

begin
  PrePass:=True;
  Prog.RemoveMethods;
  Pass;

  PrePass:=False;
  Prog.ResetMethods;
  Pass;

  Prog.Code.AddOpCode(opEnd);
end;

end.
