unit RBCode;
{$MODE OBJFPC}{$H+}
interface
uses
  RBOwn, RBStrDB;

type
  { TOpcode }
  TOpcode = (
    opNop,
    opHello,
    opPushZero,
    opPushInt,
    opPushReal,
    opPushStr,
    opDrop,
    opNeg,
    opNot,
    opAnd,
    opOr,
    opXOr,
    opEq,
    opNEq,
    opLess,
    opGreat,
    opLEqual,
    opGEqual,
    opAdd,
    opSub,
    opDiv,
    opIntDiv,
    opMod,
    opMul,
    opPow,
    opJump,
    opCall,
    opRet,
    opIfNot,
    opGFor,
    opLFor,
    opEnd,
    opMetCall,
    opNatCall,
    opObjCall,
    opPushArg,
    opPopArg,
    opLRead,
    opLWrite,
    opDimLRead,
    opDimLWrite,
    opLDim,
    opLReDim,
    opGRead,
    opGWrite,
    opDimGRead,
    opDimGWrite,
    opGDim,
    opGReDim,
    opPWrite,
    opPrint,
    opTab,
    opNewLine
  );

  TCodePosition = record
    Address, Position: integer;
    Source: TObject;
  end;

  { TCode }
  TCode = class(TOwnable)
  private
    FStringDB: TStringDB;
    Bytes: array of Byte;
    FAddress: Integer;
    FMaxAddr: Integer;
    FPositions: array of TCodePosition;
  public
    procedure Reset;
    procedure AddPosition(Source: TObject; P: Integer);
    function FindPosition(Addr: Integer; out CodePosition: TCodePosition): Boolean;
    procedure AddByte(V: Byte);
    procedure AddOpCode(V: TOpCode);
    procedure AddCardinal(V: Cardinal);
    procedure AddInteger(V: Int64);
    procedure AddReal(V: Double);
    procedure AddString(V: string);
    procedure SetCardinal(Addr, V: Cardinal);
    function GetByte: Byte;
    function GetOpCode: TOpCode;
    function GetCardinal: Cardinal;
    function GetInteger: Int64;
    function GetReal: Double;
    function GetString: string;
    function HasMore: Boolean;
    //procedure Encode(Stream: TStream);
    //procedure Decode(Stream: TStream);
    property StringDB: TStringDB read FStringDB write FStringDB;
    property Address: Integer read FAddress write FAddress;
  end;

implementation

{ TCode }
procedure TCode.Reset;
begin
  FAddress:=0;
  FMaxAddr:=0;
  SetLength(FPositions, 0);
end;

procedure TCode.AddPosition(Source: TObject; P: Integer);
begin
  if (Length(FPositions) > 0) and (FPositions[High(FPositions)].Position >= P) then Exit;
  SetLength(FPositions, Length(FPositions) + 1);
  FPositions[High(FPositions)].Address:=Address;
  FPositions[High(FPositions)].Source:=Source;
  FPositions[High(FPositions)].Position:=P;
end;

function TCode.FindPosition(Addr: Integer; out CodePosition: TCodePosition): Boolean;
var
  I: Integer;
begin
  for I:=0 to High(FPositions) - 1 do begin
    if (Addr >= FPositions[I].Address) and (Addr < FPositions[I + 1].Address) then begin
      CodePosition:=FPositions[I];
      Exit(True);
    end;
  end;
  if (Length(FPositions) > 0) and (Address >= FPositions[High(FPositions)].Address) then begin
    CodePosition:=FPositions[High(FPositions)];
    Exit(True);
  end;
  Result:=False;
end;

procedure TCode.AddByte(V: Byte);
begin
  if FAddress >= Length(Bytes) then begin
    SetLength(Bytes, FAddress + 16384);
  end;
  Bytes[FAddress]:=V;
  Inc(FAddress);
  if FMaxAddr < FAddress then FMaxAddr:=FAddress;
end;

procedure TCode.AddOpCode(V: TOpCode);
begin
  AddByte(Ord(V));
end;

procedure TCode.AddCardinal(V: Cardinal);
begin
  AddByte(V and $FF);
  AddByte((V shr 8) and $FF);
  AddByte((V shr 16) and $FF);
  AddByte((V shr 24) and $FF);
end;

procedure TCode.AddInteger(V: Int64);
begin
  AddByte(V and $FF);
  AddByte((V shr 8) and $FF);
  AddByte((V shr 16) and $FF);
  AddByte((V shr 24) and $FF);
  AddByte((V shr 32) and $FF);
  AddByte((V shr 40) and $FF);
  AddByte((V shr 48) and $FF);
  AddByte((V shr 56) and $FF);
end;

procedure TCode.AddReal(V: Double);
var
  I: Int64 absolute V;
begin
  AddInteger(I);
end;

procedure TCode.AddString(V: string);
begin
  AddCardinal(StringDB.StringToID(V));
end;

procedure TCode.SetCardinal(Addr, V: Cardinal);
begin
  Bytes[Addr]:=V and $FF;
  Bytes[Addr + 1]:=(V shr 8) and $FF;
  Bytes[Addr + 2]:=(V shr 16) and $FF;
  Bytes[Addr + 3]:=(V shr 24) and $FF;
end;

function TCode.GetByte: Byte;
begin
  Result:=Bytes[FAddress];
  Inc(FAddress);
end;

function TCode.GetOpCode: TOpCode;
begin
  Result:=TOpCode(GetByte);
end;

function TCode.GetCardinal: Cardinal;
begin
  Result:=Cardinal(GetByte);
  Result:=(Cardinal(GetByte) shl 8) or Result;
  Result:=(Cardinal(GetByte) shl 16) or Result;
  Result:=(Cardinal(GetByte) shl 24) or Result;
end;

function TCode.GetInteger: Int64;
begin
  Result:=Int64(GetByte);
  Result:=(Int64(GetByte) shl 8) or Result;
  Result:=(Int64(GetByte) shl 16) or Result;
  Result:=(Int64(GetByte) shl 24) or Result;
  Result:=(Int64(GetByte) shl 32) or Result;
  Result:=(Int64(GetByte) shl 40) or Result;
  Result:=(Int64(GetByte) shl 48) or Result;
  Result:=(Int64(GetByte) shl 56) or Result;
end;

function TCode.GetReal: Double;
var
  I: Int64;
  R: Double absolute I;
begin
  I:=GetInteger;
  Result:=R;
end;

function TCode.GetString: string;
begin
  Result:=StringDB.IDToString(GetCardinal);
end;

function TCode.HasMore: Boolean;
begin
  Result:=Address < Length(Bytes);
end;
{
procedure TCode.Encode(Stream: TStream);

  procedure EncInt(I: Cardinal);
  begin
    Stream.WriteByte(I and $FF);
    Stream.WriteByte((I shr 8) and $FF);
    Stream.WriteByte((I shr 16) and $FF);
    Stream.WriteByte((I shr 24) and $FF);
  end;

begin
  Stream.WriteByte(42);
  EncInt(FMaxAddr);
  Stream.WriteBuffer(Bytes[0], FMaxAddr);
end;

procedure TCode.Decode(Stream: TStream);
var
  I: Integer;

  function DecInt: Cardinal;
  begin
    Result:=Stream.ReadByte;
    Result:=(Stream.ReadByte shl 8) or Result;
    Result:=(Stream.ReadByte shl 16) or Result;
    Result:=(Stream.ReadByte shl 24) or Result;
  end;

begin
  Reset;
  if Stream.ReadByte <> 42 then Exit;
  FAddress:=Integer(DecInt);
  SetLength(Bytes, FAddress);
  FMaxAddr:=FAddress;
  I:=0;
  while I < FAddress do
    Inc(I, Stream.Read(Bytes[I], FAddress - I));
  FAddress:=0;
end;
}
end.
