unit RBVMStor;
{$MODE OBJFPC}{$H+}
interface

uses
  RBOwn;

type
  PVMArray = ^TVMArray;

  { TVMValue }

  TVMValueType = (vtInvalid, vtInteger, vtReal, vtString, vtArray); PVMValue = ^TVMValue;
  TVMValue = record
    ValueType: TVMValueType;
    StrValue: string;
    case LongInt of
      0: ( IntValue: Int64 );
      1: ( RealValue: Double );
      2: ( ArrValue: PVMArray );
  end;

  { TVMArray }

  TVMArraySize = array [0..5] of Integer;
  TVMArray = object
    Size: TVMArraySize;
    Dims: Integer;
    Refs: Integer;
    Data: PVMValue;
    DLen: Integer;
    VMP: Pointer;
    constructor Init(const ASize: TVMArraySize; ADims: Integer);
    destructor Done;
    function GetIndex(const Indices: TVMArraySize): Integer; inline;
    procedure Write(const Indices: TVMArraySize; const V: TVMValue); inline;
    procedure Read(const Indices: TVMArraySize; out V: TVMValue); inline;
  end;

  { TVMStorage }

  TVMStorage = class(TOwnable)
  private
    Values: array of TVMValue;
    Max: Integer;
    FActive: Boolean;
  public
    constructor Create(AOwner: TOwnable); override;
    destructor Destroy; override;
    procedure Clear;
    procedure Copy(const Source: TVMStorage);
    procedure Write(ASlot: Integer; const AValue: TVMValue);
    procedure Read(ASlot: Integer; var AValue: TVMValue);
    property Active: Boolean read FActive write FActive;
  end;

procedure SetInt(out V: TVMValue; S: Int64); inline;
procedure SetReal(out V: TVMValue; S: Double); inline;
procedure SetString(out V: TVMValue; S: string); inline;
procedure IncRef(Arr: PVMArray); inline;
procedure DecRef(Arr: PVMArray); inline;
procedure CreateArray(VM: Pointer; const Size: TVMArraySize; Dimensions: Integer; var V: TVMValue);
function ValueToString(const V: TVMValue): string;

implementation

uses
  SysUtils, RBVM, RBUtil, RBError;

procedure SetInt(out V: TVMValue; S: Int64);
begin
  V.ValueType:=vtInteger;
  V.IntValue:=S;
end;

procedure SetReal(out V: TVMValue; S: Double);
begin
  V.ValueType:=vtReal;
  V.RealValue:=S;
end;

procedure SetString(out V: TVMValue; S: string);
begin
  V.ValueType:=vtString;
  V.StrValue:=S;
end;

procedure IncRef(Arr: PVMArray);
begin
  Inc(Arr^.Refs);
end;

procedure DecRef(Arr: PVMArray);
begin
  Dec(Arr^.Refs);
  if Arr^.Refs <= 0 then begin
    TVirtualMachine(Arr^.VMP).DisposeArray(Arr);
  end;
end;

procedure CreateArray(VM: Pointer; const Size: TVMArraySize; Dimensions: Integer; var V: TVMValue);
begin
  V.ValueType:=vtArray;
  V.ArrValue:=New(PVMArray, Init(Size, Dimensions));
  V.ArrValue^.VMP:=VM;
end;

function ValueToString(const V: TVMValue): string;
var
  I: Integer;
begin
  case V.ValueType of
    vtInteger: Result:=IntToStr(V.IntValue);
    vtReal: Result:=FloatStr(V.RealValue);
    vtString: Result:=V.StrValue;
    vtArray: begin
      Result:='[';
      for I:=0 to V.ArrValue^.DLen - 1 do begin
        if I > 0 then Result:=Result + ', ';
        Result:=Result + ValueToString(V.ArrValue^.Data[I]);
      end;
      Result:=Result + ']';
    end;
  end;
end;

{ TVMArray }
constructor TVMArray.Init(const ASize: TVMArraySize; ADims: Integer);
var
  I, Total: Integer;
begin
  Size:=ASize;
  Dims:=ADims;
  Refs:=0;
  Total:=1;
  for i:=0 to Dims - 1 do Total:=Total*ASize[I];
  GetMem(Data, SizeOf(TVMValue)*Total);
  DLen:=Total;
  for I:=0 to DLen - 1 do begin
    Initialize(Data[I]);
    Data[I].ValueType:=vtInteger;
    Data[I].IntValue:=0;
  end;
end;

destructor TVMArray.Done;
var
  I: Integer;
begin
  for I:=0 to DLen - 1 do Finalize(Data[I]);
  FreeMem(Data);
end;

function TVMArray.GetIndex(const Indices: TVMArraySize): Integer;
var
  I, J, K: Integer;
begin
  Result:=0;
  for I:=0 to Dims - 1 do begin
    K:=Indices[I] - 1;
    if (K < 0) or (K >= Size[I]) then TVirtualMachine(VMP).RuntimeError(emArrayIndexOutOfBounds);
    for J:=0 to I - 1 do K:=K*Size[J];
    Result:=Result + K;
  end;
end;

procedure TVMArray.Write(const Indices: TVMArraySize; const V: TVMValue);
var
  Index: Integer;
begin
  if (V.ValueType=vtArray) and Assigned(V.ArrValue) then IncRef(V.ArrValue);
  Index:=GetIndex(Indices);
  if Index > DLen then TVirtualMachine(VMP).RuntimeError(emArrayIndexOutOfBounds);
  if (Data[Index].ValueType=vtArray) and Assigned(Data[Index].ArrValue) then DecRef(Data[Index].ArrValue);
  Data[Index]:=V;
end;

procedure TVMArray.Read(const Indices: TVMArraySize; out V: TVMValue);
var
  Index: Integer;
begin
  Index:=GetIndex(Indices);
  if Index >= DLen then TVirtualMachine(VMP).RuntimeError(emArrayIndexOutOfBounds);
  V:=Data[Index];
end;

{ TVMStorage }
constructor TVMStorage.Create(AOwner: TOwnable);
begin
  inherited Create(AOwner);
  Max:=-1;
end;

destructor TVMStorage.Destroy;
begin
  Clear;
  inherited Destroy;
end;

procedure TVMStorage.Clear;
var
  I: Integer;
begin
  for I:=0 to Max do with Values[I] do begin
    if (ValueType=vtArray) and Assigned(ArrValue) then DecRef(ArrValue);
    ValueType:=vtInteger;
    StrValue:='';
    IntValue:=0;
  end;
  Max:=-1;
end;

procedure TVMStorage.Copy(const Source: TVMStorage);
var
  I: Integer;
begin
  Clear;
  SetLength(Values, Source.Max + 1);
  for I:=0 to Source.Max do begin
    Values[I]:=Source.Values[I];
    if (Values[I].ValueType=vtArray) and Assigned(Values[I].ArrValue) then IncRef(Values[I].ArrValue);
  end;
  Max:=Source.Max;
end;

procedure TVMStorage.Write(ASlot: Integer; const AValue: TVMValue);
var
  I, J: Integer;
begin
  if ASlot > High(Values) then begin
    J:=Length(Values);
    SetLength(Values, ASlot + 1);
    for I:=J to ASlot - 1 do with Values[I] do begin
      ValueType:=vtInteger;
      IntValue:=0;
    end;
  end;
  if (AValue.ValueType=vtArray) and Assigned(AValue.ArrValue) then IncRef(AValue.ArrValue);
  if (Values[ASlot].ValueType=vtArray) and Assigned(Values[ASlot].ArrValue) then DecRef(Values[ASlot].ArrValue);
  Values[ASlot]:=AValue;
  if Max < ASlot then Max:=ASlot;
end;

procedure TVMStorage.Read(ASlot: Integer; var AValue: TVMValue);
begin
  if ASlot > High(Values) then with AValue do begin
    ValueType:=vtInteger;
    IntValue:=0;
    Exit;
  end;
  AValue:=Values[ASlot];
end;

end.
