unit RBUIGtk;
{$MODE OBJFPC}{$H+}
interface

uses
  SysUtils, gtk2, gdk2, glib2, pango, Math, RBOwn, RBUI;

type
  TGTK2UserInterface = class(TUserInterface)
  private
    FMainWindow: PGtkWidget;
    FEditorScroll: PGtkWidget;
    FEditorWidget: PGtkWidget;
    LastDirectory: string;
    procedure CreateCodeEditor;
    procedure CreateMainWindow;
    procedure UpdateTitle;
  protected
    procedure FileNameChanged; override;
    function ShowFileDialog(ForSave: Boolean; Description, Extension: string; var SelectedFile: string): Boolean; override;
    procedure ShowMethodsDialog;
  public
    function Initialize: Boolean; override;
    procedure Run; override;
    procedure Quit; override;
    procedure MessageBox(ACaption, AMessage: string; MessageType: TMessageType); override;
    function ConfirmBox(ACaption, AMessage: string; DefValue: Boolean): Boolean; override;
    procedure RefreshEditor; override;
    procedure RefreshEditorLine(Y: Integer); override;
    property MainWindow: PGtkWidget read FMainWindow;
    property EditorWidget: PGtkWidget read FEditorWidget;
  end;

function GTK2UserInterface: TGTK2UserInterface; inline;

implementation

uses
  RBUTF8;

{$PUSH}{$WARN 5024 OFF}
{$PACKRECORDS C}

// Helpers ////////////////////////////////////////////////////////////////////

procedure gtk_marshal_VOID__POINTER_POINTER(c: PGClosure; rv: PGValue; npv: guint; pv: PGValue; ih,d : gpointer); cdecl; external gtklib;

function GTK2UserInterface: TGTK2UserInterface; inline;
begin
  Result:=TGTK2UserInterface(UserInterface);
end;

function Color(r, g, b: Byte): TGdkColor;
begin
  Result.pixel:=gdk_rgb_xpixel_from_rgb((r shl 16) or (g shl 8) or b);
  Result.red:=r shl 8;
  Result.green:=g shl 8;
  Result.blue:=b shl 8;
end;

procedure SetFgColor(GC: PGdkGC; r, g, b: Byte);
var
  Col: TGdkColor;
begin
  Col:=Color(r, g, b);
  gdk_gc_set_foreground(GC, @Col);
end;

procedure SetBgColor(GC: PGdkGC; r, g, b: Byte);
var
  Col: TGdkColor;
begin
  Col:=Color(r, g, b);
  gdk_gc_set_background(GC, @Col);
end;

// Code Editor ////////////////////////////////////////////////////////////////

type
  PRBGtkEditorClass = ^TRBGtkEditorClass;
  TRBGtkEditorClass = record
    ParentClass: TGtkWidgetClass;
  end;

  PRBGtkEditor = ^TRBGtkEditor;
  TRBGtkEditor = record
    Widget: TGtkWidget;
  end;

var
  Ed: TCodeEditor;
  CodeEditorParentClass: PGtkWidgetClass = nil;
  CodeEditorType: TGtkType = 0;
  CodeEditorTypeInfo: TGtkTypeInfo;
  EditorLayout: PPangoLayout;
  ChWidth, ChHeight: Integer;
  LastKnownLineCount: Integer;

procedure MakeSureCursorIsVisible;
var
  VW, VH, VX, VY: gint;
  HA, VA: PGtkAdjustment;
  Widget: PGtkWidget;
  N: Double;
begin
  Widget:=GTK2UserInterface.EditorWidget;
  if LastKnownLineCount <> Ed.LineCount then begin
    LastKnownLineCount:=Ed.LineCount;
    gtk_widget_set_size_request(Widget, 256*ChWidth, Ed.LineCount*ChHeight + ChHeight);
    while gtk_events_pending <> 0 do gtk_main_iteration;
  end;

  VW:=Widget^.parent^.allocation.width;
  VH:=Widget^.parent^.allocation.height;
  HA:=gtk_viewport_get_hadjustment(GTK_VIEWPORT(Widget^.parent));
  VA:=gtk_viewport_get_vadjustment(GTK_VIEWPORT(Widget^.parent));
  gtk_widget_translate_coordinates(Widget, Widget^.parent, Ed.CursorX*ChWidth, Ed.CursorY*ChHeight, @VX, @VY);
  if VX + ChWidth > VW then begin
    N:=gtk_adjustment_get_value(HA) + (VX + ChWidth*12 - VW);
    if N > HA^.upper - HA^.page_size then N:=HA^.upper - HA^.page_size;
    gtk_adjustment_set_value(HA, N);
  end;
  if VX < 0 then begin
    N:=gtk_adjustment_get_value(HA) + VX - ChWidth*12;
    if N < HA^.lower then N:=HA^.lower;
    gtk_adjustment_set_value(HA, N);
  end;
  if VY + ChHeight > VH then begin
    N:=gtk_adjustment_get_value(VA) + (VY + ChHeight - VH);
    if N > VA^.upper - VA^.page_size then N:=VA^.upper - VA^.page_size;
    gtk_adjustment_set_value(VA, N);
  end;
  if VY < 0 then begin
    N:=gtk_adjustment_get_value(VA) + VY;
    if N < VA^.lower then N:=VA^.lower;
    gtk_adjustment_set_value(VA, N);
  end;
end;

procedure CodeEditorDestroy(Obj: PGtkObject); cdecl;
var
  Editor: PRBGtkEditor;
begin
  Editor:=PRBGtkEditor(Obj);
  if Assigned(GTK_OBJECT_CLASS(CodeEditorParentClass)^.destroy) then
    GTK_OBJECT_CLASS(CodeEditorParentClass)^.destroy(Obj);
end;

procedure CodeEditorRealize(Widget: PGtkWidget); cdecl;
var
  Editor: PRBGtkEditor;
  Attributes: TGdkWindowAttr;
  AttributesMask: GInt;
begin
  Editor:=PRBGtkEditor(Widget);
  //gtk_widget_set_realized(Widget, True);
  GTK_WIDGET_SET_FLAGS(Widget, GTK_REALIZED);
  with Attributes do begin
    x:=Widget^.allocation.x;
    y:=Widget^.allocation.y;
    width:=Widget^.allocation.width;
    height:=Widget^.allocation.height;
    wclass:=GDK_INPUT_OUTPUT;
    window_type:=GDK_WINDOW_CHILD;
    event_mask:=GDK_EXPOSURE_MASK or
                GDK_BUTTON_PRESS_MASK or
                GDK_BUTTON_RELEASE_MASK or
                GDK_POINTER_MOTION_MASK or
                GDK_POINTER_MOTION_HINT_MASK or
                GDK_KEY_PRESS_MASK or
                GDK_KEY_RELEASE_MASK;
    visual:=gtk_widget_get_visual(Widget);
    colormap:=gtk_widget_get_colormap(Widget);
  end;
  AttributesMask:=GDK_WA_X or GDK_WA_Y or GDK_WA_VISUAL or GDK_WA_COLORMAP;
  Widget^.window:=gdk_window_new(gtk_widget_get_parent_window(Widget), @Attributes, AttributesMask);
  Widget^.style:=gtk_style_attach(Widget^.style, Widget^.window);
  gdk_window_set_user_data(Widget^.window, Widget);
  gtk_style_set_background(Widget^.style, Widget^.window, GTK_STATE_ACTIVE);
end;

function CodeEditorExpose(Widget: PGtkWidget; Event: PGdkEventExpose): gboolean; cdecl;
var
  Editor: PRBGtkEditor;
  GC: PGdkGC;
  Rect: TGdkRectangle;
  D: PGdkDrawable;
  LY: Integer;

  procedure SetText(Text: string; out Width, Height: Integer); inline;
  var
    InkRect, LogRect: TPangoRectangle;
  begin
    pango_layout_set_text(EditorLayout, PChar(Text), Length(Text));
    pango_layout_get_pixel_extents(EditorLayout, @InkRect, @LogRect);
    Width:=LogRect.width - LogRect.x;
    Height:=LogRect.height - LogRect.y;
  end;

  procedure SetText(Text: string); inline;
  begin
    pango_layout_set_text(EditorLayout, PChar(Text), Length(Text));
  end;

  procedure DrawText(X, Y: Integer; Text: string); inline;
  begin
    pango_layout_set_text(EditorLayout, PChar(Text), Length(Text));
    gdk_draw_layout(D, GC, X, Y, EditorLayout);
  end;

  procedure InitLayout;
  begin
    EditorLayout:=gtk_widget_create_pango_layout(Widget, nil);
    SetText('W', ChWidth, ChHeight);
  end;

  procedure DrawLine;
  var
    Line: PCodeLine;
    Y: Integer;
    X, MinX, MaxX: Integer;
  begin
    if (LY < 0) or (LY >= Ed.LineCount) then Exit;
    Y:=LY*ChHeight;
    Line:=Ed.Lines[LY];
    if Ed.WholeLineInSelection(LY) then begin
      SetFgColor(GC, 180, 192, 250);
      gdk_draw_rectangle(D, GC, 1, Rect.X, Y, Rect.Width, ChHeight);
    end else if Ed.LineInSelection(LY) then begin
      MinX:=MaxInt;
      MaxX:=0;
      for X:=Rect.X div ChWidth to (Rect.X + Rect.Width) div ChWidth + 1 do
        if Ed.CharInSelection(X, LY) then begin
          MinX:=Min(MinX, X*ChWidth);
          MaxX:=Max(MaxX, X*ChWidth + ChWidth);
        end;
      if MaxX > MinX then begin
        SetFgColor(GC, 180, 192, 250);
        gdk_draw_rectangle(D, GC, 1, MinX, Y, MaxX - MinX, ChHeight);
      end;
    end;
    SetFgColor(GC, 0, 0, 0);
    DrawText(0, Y, Line^.Code);
  end;

begin
  Result:=False;
  Editor:=PRBGtkEditor(Widget);
  D:=Widget^.window;
  GC:=gdk_gc_new(D);
  if not Assigned(GC) then Exit;
  if not Assigned(EditorLayout) then InitLayout;
  pango_layout_set_width(EditorLayout, -1);
  Rect:=Event^.Area;
  gdk_gc_set_clip_origin(GC, 0, 0);
  gdk_gc_set_clip_rectangle(GC, @Rect);
  gdk_gc_set_fill(GC, GDK_SOLID);
  SetFgColor(GC, 255, 255, 255);
  gdk_draw_rectangle(D, GC, 1, Rect.x, Rect.y, Rect.width, Rect.height);
  {
  SetFgColor(GC, 255, 0, 0);
  gdk_draw_rectangle(D, GC, 0, Rect.x, Rect.y, Rect.width - 1, Rect.height - 1);
  }
  for LY:=Rect.y div ChHeight to (Rect.y + Rect.height) div ChHeight do DrawLine;
  if gtk_widget_has_focus(Widget) then begin
    SetFgColor(GC, 0, 0, 0);
    gdk_draw_rectangle(D, GC, 1, Ed.CursorX*ChWidth, Ed.CursorY*ChHeight, 2, ChHeight);
  end;
  gdk_gc_unref(GC);
end;

procedure CodeEditorSizeRequest(Widget: PGtkWidget; Requisition: PGtkRequisition); cdecl;
begin
  with Requisition^ do begin
    width:=300;
    height:=200;
  end;
end;

procedure CodeEditorSizeAllocate(Widget: PGtkWidget; Allocation: PGtkAllocation); cdecl;
begin
  Widget^.allocation:=Allocation^;
  if (GTK_WIDGET_FLAGS(Widget) and GTK_REALIZED)=GTK_REALIZED then begin
    gdk_window_move_resize(Widget^.window, Allocation^.x, Allocation^.y, Allocation^.width, Allocation^.height);
  end;
end;

function CodeEditorButtonPress(Widget: PGtkWidget; Event: PGdkEventButton): gboolean; cdecl;
var
  Editor: PRBGtkEditor;
begin
  Result:=False;
  Editor:=PRBGtkEditor(Widget);
end;

function CodeEditorButtonRelease(Widget: PGtkWidget; Event: PGdkEventButton): gboolean; cdecl;
var
  Editor: PRBGtkEditor;
begin
  Result:=False;
  Editor:=PRBGtkEditor(Widget);
end;

function CodeEditorMotionNotify(Widget: PGtkWidget; Event: PGdkEventMotion): gboolean; cdecl;
var
  Editor: PRBGtkEditor;
begin
  Result:=False;
  Editor:=PRBGtkEditor(Widget);
end;

function CodeEditorKeyPress(Widget: PGtkWidget; Event: PGdkEventKey): gboolean; cdecl;
var
  Shift, Control: Boolean;
  LastCursorX, LastCursorY: Integer;
begin
  Result:=True;
  if not gtk_widget_has_focus(GTK2UserInterface.EditorWidget) then Exit;
  Control:=(Event^.state and GDK_CONTROL_MASK)=GDK_CONTROL_MASK;
  Shift:=(Event^.state and GDK_SHIFT_MASK)=GDK_SHIFT_MASK;
  LastCursorX:=Ed.CursorX;
  LastCursorY:=Ed.CursorY;
  case Event^.keyval of
    GDK_KEY_Left: Ed.MoveLeft(Control);
    GDK_KEY_Right: Ed.MoveRight(Control);
    GDK_KEY_Up: Ed.MoveUp(1);
    GDK_KEY_Down: Ed.MoveDown(1);
    GDK_KEY_Page_Up: Ed.MoveUp(GTK2UserInterface.FEditorScroll^.allocation.height div ChHeight);
    GDK_KEY_Page_Down: Ed.MoveDown(GTK2UserInterface.FEditorScroll^.allocation.height div ChHeight);
    GDK_KEY_Home: Ed.MoveHome;
    GDK_KEY_End: Ed.MoveEnd;
    GDK_KEY_Delete: Ed.Delete;
    GDK_KEY_BackSpace: Ed.Erase;
    GDK_KEY_Return: begin
      Ed.BreakLine;
      Ed.IndentLine(Ed.CursorY);
    end;
    GDK_KEY_Tab, 65056 { ??? }: begin
      Ed.ShiftIndent(Shift);
      MakeSureCursorIsVisible;
      Exit;
    end;
    else if ((Event^.state and $FF) in [0, GDK_SHIFT_MASK]) and (Length(Event^._string) > 0) and (Ord(Event^._string[0]) > 31) then begin
      Ed.Insert(Event^._string);
      Shift:=False;
    end else Result:=false;
  end;
  if (Ed.CursorX <> LastCursorX) or (Ed.CursorY <> LastCursorY) then begin
    if Shift then Ed.UpdateSelection(LastCursorX, LastCursorY) else Ed.ClearSelection;
  end;
  if Result then MakeSureCursorIsVisible;
end;

function CodeEditorKeyRelease(Widget: PGtkWidget; Event: PGdkEventKey): gboolean; cdecl;
begin
  Result:=False;
end;

procedure CodeEditorClassInit(Cls: Pointer); cdecl;
var
  ObjectClass: PGtkObjectClass;
  WidgetClass: PGtkWidgetClass;
  EditorClass: PRBGtkEditorClass;
begin
  ObjectClass:=PGtkObjectClass(Cls);
  WidgetClass:=PGtkWidgetClass(Cls);
  EditorClass:=PRBGtkEditorClass(Cls);
  CodeEditorParentClass:=gtk_type_class(gtk_widget_get_type);
  ObjectClass^.Destroy:=@CodeEditorDestroy;
  with WidgetClass^ do begin
    realize:=@CodeEditorRealize;
    expose_event:=@CodeEditorExpose;
    size_request:=@CodeEditorSizeRequest;
    size_allocate:=@CodeEditorSizeAllocate;
    button_press_event:=@CodeEditorButtonPress;
    button_release_event:=@CodeEditorButtonRelease;
  end;
end;

procedure CodeEditorInit(TypeInstance: PGTypeInstance; CodeEditor: Pointer); cdecl;
//var
//Widget: PGtkWidget;
begin
  //Widget:=PGtkWidget(CodeEditor);
end;

function CodeEditorGetType: TGtkType; cdecl;
begin
  if CodeEditorType=0 then begin
    FillChar(CodeEditorTypeInfo, SizeOf(CodeEditorTypeInfo), 0);
    with CodeEditorTypeInfo do begin
      type_name:='RBGtkCodeEditor';
      object_size:=SizeOf(TRBGtkEditor);
      class_size:=SizeOf(TRBGtkEditorClass);
      class_init_func:=@CodeEditorClassInit;
      object_init_func:=@CodeEditorInit;
    end;
    CodeEditorType:=gtk_type_unique(GTK_TYPE_WIDGET, @CodeEditorTypeInfo);
  end;
  Result:=CodeEditorType;
end;

procedure TGTK2UserInterface.CreateCodeEditor;
var
  VP: PGtkWidget;
  FD: TPangoFontDescription;
begin
  FEditorScroll:=gtk_scrolled_window_new(nil, nil);
  VP:=gtk_viewport_new(gtk_scrolled_window_get_hadjustment(GTK_SCROLLED_WINDOW(FEditorScroll)), gtk_scrolled_window_get_vadjustment(GTK_SCROLLED_WINDOW(FEditorScroll)));
  gtk_viewport_set_shadow_type(GTK_VIEWPORT(VP), GTK_SHADOW_NONE);
  gtk_widget_push_visual(gdk_rgb_get_visual);
  gtk_widget_push_colormap(gdk_rgb_get_cmap);
  FEditorWidget:=gtk_type_new(CodeEditorGetType);
  gtk_widget_pop_colormap;
  gtk_widget_pop_visual;
  FD:=pango_font_description_from_string('monospace normal');
  gtk_widget_modify_font(EditorWidget, FD);
  GTK_WIDGET_SET_FLAGS(EditorWidget, GTK_CAN_FOCUS);
  g_signal_connect(G_OBJECT(MainWindow), 'key_press_event', G_CALLBACK(@CodeEditorKeyPress), NULL);
  g_signal_connect(G_OBJECT(MainWindow), 'key_release_event', G_CALLBACK(@CodeEditorKeyRelease), NULL);
  pango_font_description_free(FD);
  gtk_container_add(GTK_CONTAINER(VP), EditorWidget);
  gtk_container_add(GTK_CONTAINER(FEditorScroll), VP);
end;

// Menu Actions ///////////////////////////////////////////////////////////////

var
  Clip: string;

procedure CopyToClipboard;
begin
  Clip:=UserInterface.CodeEditor.GetSelectionText;
end;

procedure MenuFileNew(Widget: PGtkWidget; UserData: gpointer); cdecl;
begin
  UserInterface.NewProgram;
end;

procedure MenuFileOpen(Widget: PGtkWidget; UserData: gpointer); cdecl;
begin
  UserInterface.OpenProgram;
end;

procedure MenuFileSave(Widget: PGtkWidget; UserData: gpointer); cdecl;
begin
  UserInterface.SaveProgram;
end;

procedure MenuFileSaveAs(Widget: PGtkWidget; UserData: gpointer); cdecl;
begin
  UserInterface.SaveProgramAs;
end;

procedure MenuEditCut(Widget: PGtkWidget; UserData: gpointer); cdecl;
begin
  CopyToClipboard;
  UserInterface.CodeEditor.DeleteSelection;
  MakeSureCursorIsVisible;
end;

procedure MenuEditCopy(Widget: PGtkWidget; UserData: gpointer); cdecl;
begin
  CopyToClipboard;
end;

procedure MenuEditPaste(Widget: PGtkWidget; UserData: gpointer); cdecl;
begin
  UserInterface.CodeEditor.SetSelectionText(Clip);
  MakeSureCursorIsVisible;
end;

procedure MenuEditClear(Widget: PGtkWidget; UserData: gpointer); cdecl;
begin
  UserInterface.CodeEditor.DeleteSelection;
  MakeSureCursorIsVisible;
end;

procedure MenuViewSubsAndFunctions(Widget: PGtkWidget; UserData: gpointer); cdecl;
begin
  GTK2UserInterface.ShowMethodsDialog;
end;

procedure MenuRunStart(Widget: PGtkWidget; UserData: gpointer); cdecl;
begin
  UserInterface.RunProgram;
end;

// Main Window ////////////////////////////////////////////////////////////////

function MainWindowDeleteEvent(Widget: PGtkWidget; Event: PGdkEvent; Data: Pointer): Boolean; cdecl;
begin
  UserInterface.Quit;
  Result:=True;
end;

procedure MainWindowDestroy(Widget: PGtkWidget; Data: Pointer); cdecl;
begin
  UserInterface.Quit;
end;

procedure TGTK2UserInterface.CreateMainWindow;
var
  TopBox: PGtkContainer;
  MenuBar: PGtkMenuShell;
  TopMenu: PGtkMenu;
  AccelGroup: PGtkAccelGroup;

  procedure AddTopMenu(Caption: string);
  var
    MenuItem: PGtkMenuItem;
  begin
    MenuItem:=GTK_MENU_ITEM(gtk_menu_item_new_with_mnemonic(PGChar(Caption)));
    TopMenu:=GTK_MENU(gtk_menu_new);
    gtk_menu_item_set_submenu(MenuItem, GTK_WIDGET(TopMenu));
    gtk_menu_shell_append(MenuBar, GTK_WIDGET(MenuItem));
  end;

  procedure AddMenuItem(Caption, AccelKey: string; Handler: Pointer);
  var
    MenuItem: PGtkMenuItem;
    AccKeyOnly: GUint;
    AccKeyMod: TGdkModifierType;
  begin
    MenuItem:=GTK_MENU_ITEM(gtk_menu_item_new_with_mnemonic(PGChar(Caption)));
    if AccelKey <> '' then begin
      gtk_accelerator_parse(PGChar(AccelKey), @AccKeyOnly, @AccKeyMod);
      gtk_widget_add_accelerator(GTK_WIDGET(MenuItem), 'activate', AccelGroup, AccKeyOnly, AccKeyMod, GTK_ACCEL_VISIBLE);
    end;
    if Handler <> nil then g_signal_connect(MenuItem, 'activate', G_CALLBACK(Handler), nil);
    gtk_menu_shell_append(GTK_MENU_SHELL(TopMenu), GTK_WIDGET(MenuItem));
  end;

  procedure AddSeparator;
  var
    Separator: PGtkWidget;
  begin
    Separator:=gtk_separator_menu_item_new;
    gtk_menu_shell_append(GTK_MENU_SHELL(TopMenu), Separator);
  end;

begin
  FMainWindow:=gtk_window_new(GTK_WINDOW_TOPLEVEL);
  AccelGroup:=gtk_accel_group_new();
  gtk_window_add_accel_group(GTK_WINDOW(MainWindow), AccelGroup);
  TopBox:=GTK_CONTAINER(gtk_vbox_new(false, 0));
  gtk_container_add(GTK_CONTAINER(MainWindow), GTK_WIDGET(TopBox));
  MenuBar:=GTK_MENU_SHELL(gtk_menu_bar_new);
  AddTopMenu('_File');
  AddMenuItem('_New', '<Control>n', @MenuFileNew);
  AddMenuItem('_Open...', '<Control>o', @MenuFileOpen);
  AddMenuItem('_Save', '<Control>s', @MenuFileSave);
  AddMenuItem('Save _as...', '<Control><Shift>s', @MenuFileSaveAs);
  AddSeparator;
  AddMenuItem('_Quit', '<Control>q', @MainWindowDestroy);
  AddTopMenu('_Edit');
  AddMenuItem('_Undo', '<Control>z', nil);
  AddMenuItem('_Redo', '<Control><Shift>z', nil);
  AddSeparator;
  AddMenuItem('Cu_t', '<Control>x', @MenuEditCut);
  AddMenuItem('_Copy', '<Control>c', @MenuEditCopy);
  AddMenuItem('_Paste', '<Control>v', @MenuEditPaste);
  AddMenuItem('Cl_ear', 'Delete', @MenuEditClear);
  AddSeparator;
  AddMenuItem('Select _all', '<Control>a', nil);
  AddTopMenu('_View');
  AddMenuItem('_Subs and Functions...', 'F2', @MenuViewSubsAndFunctions);
  AddMenuItem('_Toggle status bar', '', nil);
  AddTopMenu('_Search');
  AddMenuItem('_Find...', '<Control>f', nil);
  AddMenuItem('_Repeat', 'F3', nil);
  AddMenuItem('_Change...', '<Control><Shift>f', nil);
  AddTopMenu('_Run');
  AddMenuItem('_Start', 'F5', @MenuRunStart);
  AddMenuItem('_Pause', '', nil);
  AddMenuItem('_Restart', '', nil);
  AddTopMenu('_Help');
  AddMenuItem('_About...', '', nil);
  gtk_box_pack_start(GTK_BOX(TopBox), GTK_WIDGET(MenuBar), False, True, 0);
  CreateCodeEditor;
  gtk_container_add(TopBox, FEditorScroll);
  g_signal_connect(MainWindow, 'delete-event', G_CALLBACK(@MainWindowDeleteEvent), nil);
  g_signal_connect(MainWindow, 'destroy', G_CALLBACK(@MainWindowDestroy), nil);
  gtk_window_set_default_size(GTK_WINDOW(MainWindow), 600, 440);
  UpdateTitle;
  gtk_container_resize_children(GTK_CONTAINER(MainWindow));
  gtk_widget_show_all(MainWindow);
end;

procedure TGTK2UserInterface.UpdateTitle;
var
  Title: string;
begin
  Title:='Runtime BASIC';
  if FileName <> '' then begin
    if CodeEditor.Modified then Title:=Title+'*';
    Title:=ExtractFileName(FileName) + ' - ' + Title;
  end;
  gtk_window_set_title(GTK_WINDOW(MainWindow), PGChar(Title));
end;

procedure TGTK2UserInterface.FileNameChanged;
begin
  UpdateTitle;
end;

function TGTK2UserInterface.ShowFileDialog(ForSave: Boolean; Description, Extension: string; var SelectedFile: string): Boolean;
var
  Dlg: PGtkWidget;
  FN: PChar;
  Filter: PGtkFileFilter;
begin
  if ForSave then begin
    Dlg:=gtk_file_chooser_dialog_new('Save File As', GTK_WINDOW(MainWindow), GTK_FILE_CHOOSER_ACTION_SAVE, GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL, GTK_STOCK_SAVE, GTK_RESPONSE_ACCEPT, nil);
    gtk_file_chooser_set_do_overwrite_confirmation(GTK_FILE_CHOOSER(Dlg), True);
  end else begin
    Dlg:=gtk_file_chooser_dialog_new('Open File', GTK_WINDOW(MainWindow), GTK_FILE_CHOOSER_ACTION_OPEN, GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL, GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT, nil);
  end;
  Filter:=gtk_file_filter_new;
  gtk_file_filter_set_name(Filter, PChar(Description));
  gtk_file_filter_add_pattern(Filter, PChar('*.' + Extension));
  gtk_file_chooser_add_filter(GTK_FILE_CHOOSER(Dlg), Filter);
  Filter:=gtk_file_filter_new;
  gtk_file_filter_set_name(Filter, 'All files');
  gtk_file_filter_add_pattern(Filter, '*');
  gtk_file_chooser_add_filter(GTK_FILE_CHOOSER(Dlg), Filter);
  if SelectedFile=''then begin
    if LastDirectory='' then LastDirectory:=GetCurrentDir;
    gtk_file_chooser_set_current_folder(GTK_FILE_CHOOSER(Dlg), PChar(LastDirectory));
  end else begin
    gtk_file_chooser_set_filename(GTK_FILE_CHOOSER(Dlg), PChar(SelectedFile));
  end;
  Result:=gtk_dialog_run(GTK_DIALOG(Dlg)) = GTK_RESPONSE_ACCEPT;
  if Result then begin
    FN:=gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(Dlg));
    SelectedFile:=string(FN);
    g_free(FN);
    FN:=gtk_file_chooser_get_current_folder(GTK_FILE_CHOOSER(Dlg));
    LastDirectory:=string(FN);
    g_free(FN);
    if Pos('.', SelectedFile)=0 then SelectedFile:=SelectedFile + '.' + Extension;
  end;
  gtk_widget_destroy(Dlg);
end;

procedure MethodsDialogTreeRowActivated(Tree: PGtkTreeView; Path: PGtkTreePath; Col: PGtkTreeViewColumn; UserData: gpointer); cdecl;
begin
  gtk_dialog_response(GTK_DIALOG(UserData), GTK_RESPONSE_ACCEPT);
end;

procedure TGTK2UserInterface.ShowMethodsDialog;
var
  Dlg, Content: PGtkWidget;
  Model: PGtkListStore;
  Col: PGtkTreeViewColumn;
  List: PGtkWidget;
  Iter: TGtkTreeIter;
  Sel: PGtkTreeSelection;
  Path: PGtkTreePath;
  Scroll: PGtkWidget;
  I, CurrentIndex: Integer;
  Accepted: Boolean;
begin
  Dlg:=gtk_dialog_new_with_buttons('Methods', GTK_WINDOW(MainWindow), GTK_DIALOG_DESTROY_WITH_PARENT or GTK_DIALOG_MODAL, GTK_STOCK_OK, GTK_RESPONSE_ACCEPT, nil);
  gtk_dialog_set_default_response(GTK_DIALOG(Dlg), GTK_RESPONSE_ACCEPT);
  Content:=GTK_DIALOG(Dlg)^.vbox;
  Model:=gtk_list_store_new(1, G_TYPE_STRING);
  CurrentIndex:=-1;
  for I:=0 to MethodCount - 1 do begin
    gtk_list_store_append(Model, @Iter);
    if Methods[I].Name='' then
      gtk_list_store_set(Model, @Iter, 0, 'Main Program', -1)
    else
      gtk_list_store_set(Model, @Iter, 0, PChar(Methods[I].Name), -1);
    if Methods[I]=CurrentMethod then CurrentIndex:=I;
  end;
  if CurrentIndex=-1 then CurrentIndex:=0;
  Scroll:=gtk_scrolled_window_new(nil, nil);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(Scroll), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_container_add(GTK_CONTAINER(Content), Scroll);
  List:=gtk_tree_view_new;
  g_signal_connect(List, 'row-activated', G_CALLBACK(@MethodsDialogTreeRowActivated), Dlg);
  gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(List), False);
  Col:=gtk_tree_view_column_new_with_attributes('Method', gtk_cell_renderer_text_new, 'text', 0, nil);
  gtk_tree_view_append_column(GTK_TREE_VIEW(List), Col);
  gtk_tree_view_set_model(GTK_TREE_VIEW(List), Model);
  Sel:=gtk_tree_view_get_selection(GTK_TREE_VIEW(List));
  Path:=gtk_tree_path_new_from_indices(CurrentIndex, -1);
  gtk_tree_selection_select_path(Sel, Path);
  gtk_tree_path_free(Path);
  gtk_container_add(GTK_CONTAINER(Scroll), List);
  gtk_widget_set_size_request(Dlg, 300, 300);
  gtk_widget_show_all(Dlg);
  gtk_widget_grab_focus(List);
  Accepted:=gtk_dialog_run(GTK_DIALOG(Dlg)) = GTK_RESPONSE_ACCEPT;
  I:=-1;
  if Accepted and gtk_tree_selection_get_selected(Sel, nil, @Iter) then begin
    Path:=gtk_tree_model_get_path(Model, @Iter);
    I:=gtk_tree_path_get_indices(Path)[0];
  end;
  gtk_widget_destroy(Dlg);
  if I <> -1 then CurrentMethod:=Methods[I];
end;

function TGTK2UserInterface.Initialize: Boolean;
begin
  if not gtk_init_check(@argc, @argv) then Exit(False);
  gdk_rgb_init;
  FCodeEditor:=TCodeEditor.Create(Self);
  Ed:=FCodeEditor;
  CreateMainWindow;
  CommonInit;
  Ed.Reset;
  Ed.Refresh;
  Result:=True;
end;

procedure TGTK2UserInterface.Run;
begin
  gtk_widget_grab_focus(EditorWidget);
  gtk_main();
  CommonShutdown;
  if Assigned(EditorLayout) then begin
    g_object_unref(EditorLayout);
    EditorLayout:=nil;
  end;
end;

procedure TGTK2UserInterface.Quit;
begin
  gtk_main_quit();
  gtk_widget_destroy(MainWindow);
end;

procedure TGTK2UserInterface.MessageBox(ACaption, AMessage: string; MessageType: TMessageType);
var
  Dlg: PGtkWidget;
  MT: TGtkMessageType;
begin
  case MessageType of
    mtInfo: MT:=GTK_MESSAGE_INFO;
    mtWarning: MT:=GTK_MESSAGE_WARNING;
    mtError: MT:=GTK_MESSAGE_ERROR;
  end;
  Dlg:=gtk_message_dialog_new(GTK_WINDOW(FMainWindow), GTK_DIALOG_MODAL, MT, GTK_BUTTONS_OK, '%s'#10#10'%s', PChar(ACaption), PChar(AMessage));
  gtk_dialog_run(GTK_DIALOG(Dlg));
  gtk_widget_destroy(Dlg);
end;

function TGTK2UserInterface.ConfirmBox(ACaption, AMessage: string; DefValue: Boolean): Boolean;
var
  Dlg: PGtkWidget;
begin
  Dlg:=gtk_message_dialog_new(GTK_WINDOW(FMainWindow), GTK_DIALOG_MODAL, GTK_MESSAGE_WARNING, GTK_BUTTONS_YES_NO, '%s'#10#10'%s', PChar(ACaption), PChar(AMessage));
  Result:=gtk_dialog_run(GTK_DIALOG(Dlg)) = GTK_RESPONSE_YES;
  gtk_widget_destroy(Dlg);
end;

procedure TGTK2UserInterface.RefreshEditor;
begin
  if Assigned(EditorWidget) then gtk_widget_queue_draw(EditorWidget);
end;

procedure TGTK2UserInterface.RefreshEditorLine(Y: Integer);
var
  Rect: TGdkRectangle;
begin
  if Assigned(EditorWidget) then begin
    Rect.x:=0;
    Rect.y:=Y*ChHeight;
    Rect.width:=FEditorWidget^.allocation.width;
    Rect.height:=ChHeight;
    gdk_window_invalidate_rect(FEditorWidget^.window, @Rect, False);
  end;
end;

{$POP}

end.
