//---------------------------------------------------------------------------
#ifndef WingEdgeH
#define WingEdgeH
//---------------------------------------------------------------------------
#include <string.h>
#include "HashMap.h"
#include "Plane.h"
//---------------------------------------------------------------------------
// Forward class declarations
template <typename T> struct TWEEdge;
//---------------------------------------------------------------------------
// Winged edge mesh vertex
template <typename T> struct TWEVertex
{
    T               X, Y, Z;    // Coordinates
    TWEEdge<T>*     Edge;       // Incident edge

    inline TWEVertex(){}
    inline TWEVertex(T x, T y, T z, TWEEdge<T>* edge=NULL)
        : X(x), Y(y), Z(z), Edge(edge)
    {}
    inline TWEVertex(const TVector<T>& v)
        : X(v.X), Y(v.Y), Z(v.Z), Edge(NULL)
    {}
    inline TWEVertex(const TVector<T>& v, TWEEdge<T>* edge)
        : X(v.X), Y(v.Y), Z(v.Z), Edge(edge)
    {}
    inline operator TVector<T>() const
    {
        return TVector<T>(X, Y, Z);
    }
};
//---------------------------------------------------------------------------
// Widged edge mesh face
template <typename T> struct TWEFace
{
    TWEEdge<T>*     Edge;       // Incident edge

    inline TWEFace(){}
    inline TWEFace(TWEEdge<T>* edge)
        : Edge(edge)
    {}
};
//---------------------------------------------------------------------------
// Widged edge mesh edge
template <typename T> struct TWEEdge
{
    TWEVertex<T>*   First;              // First vertex of the edge
    TWEVertex<T>*   Second;             // Second vertex of the edge
    TWEFace<T>*     Left;               // Left side face
    TWEFace<T>*     Right;              // Right side face
    TWEEdge<T>*     LeftPrevious;       // Previous edge on the left side
    TWEEdge<T>*     LeftNext;           // Next edge on the left side
    TWEEdge<T>*     RightPrevious;      // Previous edge on the right side
    TWEEdge<T>*     RightNext;          // Next edge on the right side
};
//---------------------------------------------------------------------------
// Widged edge mesh
template <typename T> class TWEMesh
{
public:
    inline ~TWEMesh()
    {
        Clear();
    }
    // Create a new vertex in the mesh
    inline TWEVertex<T>* NewVertex(T x, T y, T z)
    {
        TWEVertex<T>* v = new TWEVertex<T>(x, y, z);
        m_vertices.Add(v);
        return v;
    }
    // Create a new face in the mesh
    inline TWEFace<T>* NewFace()
    {
        TWEFace<T>* f = new TWEFace<T>(NULL);
        m_faces.Add(f);
        return f;
    }
    // Create a new edge in the mesh
    inline TWEEdge<T>* NewEdge()
    {
        TWEEdge<T>* e = new TWEEdge<T>();
        ::memset(e, 0, sizeof(TWEEdge<T>));
        m_edges.Add(e);
        return e;
    }
    // Delete the given vertex
    inline void DeleteVertex(TWEVertex<T>* v)
    {
        m_vertices.Remove(v);
        delete v;
    }
    // Delete the given face
    inline void DeleteFace(TWEFace<T>* f)
    {
        m_faces.Remove(f);
        delete f;
    }
    // Delete the given edge
    inline void DeleteEdge(TWEEdge<T>* e)
    {
        m_edges.Remove(e);
        delete e;
    }
    // Clear the mesh
    inline void Clear()
    {
        // Delete vertices
        for (unsigned i=0; i < m_vertices.GetSize(); ++i)
            delete m_vertices[i];
        m_vertices.Clear();
        // Delete faces
        for (unsigned i=0; i < m_faces.GetSize(); ++i)
            delete m_faces[i];
        m_faces.Clear();
        // Delete edges
        for (unsigned i=0; i < m_edges.GetSize(); ++i)
            delete m_edges[i];
        m_edges.Clear();
    }
    // Return the vertices in this mesh
    inline const TDynArray< TWEVertex<T>* >& GetVertices() const { return m_vertices; }
    // Return the faces in this mesh
    inline const TDynArray< TWEFace<T>* >& GetFaces() const { return m_faces; }
    // Return the edges in this mesh
    inline const TDynArray< TWEEdge<T>* >& GetEdges() const { return m_edges; }
private:
    TDynArray< TWEVertex<T>* >  m_vertices; // Vertices in the mesh
    TDynArray< TWEFace<T>* >    m_faces;    // Faces in the mesh
    TDynArray< TWEEdge<T>* >    m_edges;    // Edges in the mesh
};
//---------------------------------------------------------------------------
// Winged edge mesh functions
namespace NWEMesh
{
    // Convert the given triangle soup to a winged mesh structure, returns false
    // if the mesh cannot be converted (no holes and all edges much have exactly
    // two faces they touch)
    template <typename T> inline bool TSoupToWingedMesh(const TVector<T>* vertices, unsigned triangles, TWEMesh<T>& mesh)
    {
        mesh.Clear();        
    }
    // Same as above, but use "bare" XYZ values
    template <typename T> inline bool TSoupToWingedMesh(const T* vertices, unsigned triangles, TWEMesh<T>& mesh)
    {
        // Create vertex array
        TDynArray< TVector<T> > data;
        data.Reserve(triangles*3);
        for (unsigned i=0; i < triangles; ++i,vertices += 3)
        {
            data.Add(TVector<T>(vertices[0], vertices[1], vertices[2])); 
        }
        // Create mesh
        return TSoupToWingedMesh<T>(data.GetData(), triangles, mesh);
    }
};
#endif
