//---------------------------------------------------------------------------
#ifndef VectorH
#define VectorH
//---------------------------------------------------------------------------
#include <math.h>
#include <mem.h>
#include "MathUtils.h"
#include "DynArray.h"
//---------------------------------------------------------------------------
// Major axis
enum EVectorMajorAxis
{
    VMA_X,
    VMA_Y,
    VMA_Z
};
//---------------------------------------------------------------------------
// Vector
template <typename T> struct TVector
{
    typedef TDynArray< TVector<T> >     TAVectorArray;

    T   X, Y, Z, W;

    inline TVector() {}
    inline TVector(T x, T y, T z = 0.0f, T w = 1.0f)
        : X(x), Y(y), Z(z), W(w)
    {}

    // Set components
    inline void Set2D(T x, T y)
    {
        X = x;
        Y = y;
    }
    inline void Set3D(T x, T y, T z)
    {
        X = x;
        Y = y;
        Z = z;
    }
    inline void Set4D(T x, T y, T z, T w)
    {
        X = x;
        Y = y;
        Z = z;
        W = w;
    }

    // Copy partial components
    inline void CopyXY(const TVector<T>& v)
    {
        X = v.X;
        Y = v.Y;
    }
    inline void CopyXYZ(const TVector<T>& v)
    {
        X = v.X;
        Y = v.Y;
        Z = v.Z;
    }

    // Vector length
    inline T GetLength2D() const
    {
        return sqrt(GetLengthSquared2D());
    }
    inline T GetLength3D() const
    {
        return sqrt(GetLengthSquared3D());
    }
    inline T GetLength4D() const
    {
        return sqrt(GetLengthSquared4D());
    }
    inline T GetLengthSquared2D() const
    {
        return X*X + Y*Y;
    }
    inline T GetLengthSquared3D() const
    {
        return X*X + Y*Y + Z*Z;
    }
    inline T GetLengthSquared4D() const
    {
        return X*X + Y*Y + Z*Z + W*W;
    }

    // Normalization
    inline void Normalize2D()
    {
        T len = GetLength2D();
        if (len > 0.0)
        {
            X /= len;
            Y /= len;
        }
    }
    inline void Normalize3D()
    {
        T len = GetLength3D();
        if (len > 0.0)
        {
            X /= len;
            Y /= len;
            Z /= len;
        }
    }
    inline void Normalize4D()
    {
        T len = GetLength4D();
        if (len > 0.0)
        {
            X /= len;
            Y /= len;
            Z /= len;
            W /= len;
        }
    }
    inline TVector<T> GetNormalized2D() const
    {
        T len = GetLength2D();
        return len > 0.0 ? TVector<T>(X / len, Y / len) : *this;
    }
    inline TVector<T> GetNormalized3D() const
    {
        T len = GetLength3D();
        return len > 0.0 ? TVector<T>(X / len, Y / len, Z / len) : *this;
    }
    inline TVector<T> GetNormalized4D() const
    {
        T len = GetLength4D();
        return len > 0.0 ? TVector<T>(X / len, Y / len, Z / len, W / len) : *this;
    }

    // Vector major axis
    inline EVectorMajorAxis GetMajorAxis2D() const
    {
        return fabs(X) > fabs(Y) ? VMA_X : VMA_Y;
    }
    inline EVectorMajorAxis GetMajorAxis3D() const
    {
        return fabs(X) > fabs(Y) ?
            (fabs(X) > fabs(Z) ? VMA_X : VMA_Z) :
            (fabs(Y) > fabs(Z) ? VMA_Y : VMA_Z);
    }

    // Partial vector
    inline TVector<T> XY() const
    {
        return TVector<T>(X, Y, 0, 1);
    }
    inline TVector<T> YX() const
    {
        return TVector<T>(Y, X, 0, 1);
    }
    inline TVector<T> XZ() const
    {
        return TVector<T>(X, Z, 0, 1);
    }
    inline TVector<T> XYZ() const
    {
        return TVector<T>(X, Y, Z, 1);
    }
    inline TVector<T> ZYX() const
    {
        return TVector<T>(Z, Y, X, 1);
    }
    inline TVector<T> WXYZ() const
    {
        return TVector<T>(W, X, Y, Z);
    }
    inline TVector<T> WZYX() const
    {
        return TVector<T>(W, Z, Y, X);
    }

    // Conversion to another vector type
    inline TVector<int> ToInt() const
    {
        return TVector<int>((int)X, (int)Y, (int)Z, (int)W);
    }
    inline TVector<float> ToFloat() const
    {
        return TVector<float>(X, Y, Z, W);
    }
    inline TVector<double> ToDouble() const
    {
        return TVector<double>(X, Y, Z, W);
    }
    inline TVector<long double> ToLongDouble() const
    {
        return TVector<long double>(X, Y, Z, W);
    }

    // Math operators
    inline TVector<T> operator-() const
    {
        return TVector<T>(-X, -Y, -Z, -W);
    }
    inline TVector<T> operator-(const TVector<T>& rhs) const
    {
        return TVector<T>(X - rhs.X, Y - rhs.Y, Z - rhs.Z, W - rhs.W);
    }
    inline TVector<T> operator+(const TVector<T>& rhs) const
    {
        return TVector<T>(X + rhs.X, Y + rhs.Y, Z + rhs.Z, W + rhs.W);
    }
    inline TVector<T> operator*(const TVector<T>& rhs) const
    {
        return TVector<T>(X * rhs.X, Y * rhs.Y, Z * rhs.Z, W * rhs.W);
    }
    inline TVector<T> operator*(T s) const
    {
        return TVector<T>(X * s, Y * s, Z * s, W * s);
    }
    inline TVector<T> operator/(const TVector<T>& rhs) const
    {
        return TVector<T>(X / rhs.X, Y / rhs.Y, Z / rhs.Z, W / rhs.W);
    }
    inline TVector<T> operator/(T s) const
    {
        return TVector<T>(X / s, Y / s, Z / s, W / s);
    }
    inline TVector<T>& operator+=(const TVector<T>& rhs)
    {
        X += rhs.X;
        Y += rhs.Y;
        Z += rhs.Z;
        W += rhs.W;
        return *this;
    }
    inline TVector<T>& operator-=(const TVector<T>& rhs)
    {
        X -= rhs.X;
        Y -= rhs.Y;
        Z -= rhs.Z;
        W -= rhs.W;
        return *this;
    }
    inline TVector<T>& operator*=(const TVector<T>& rhs)
    {
        X *= rhs.X;
        Y *= rhs.Y;
        Z *= rhs.Z;
        W *= rhs.W;
        return *this;
    }
    inline TVector<T>& operator*=(T s)
    {
        X *= s;
        Y *= s;
        Z *= s;
        W *= s;
        return *this;
    }
    inline TVector<T>& operator/=(const TVector<T>& rhs)
    {
        X /= rhs.X;
        Y /= rhs.Y;
        Z /= rhs.Z;
        W /= rhs.W;
        return *this;
    }
    inline TVector<T>& operator/=(T s)
    {
        X /= s;
        Y /= s;
        Z /= s;
        W /= s;
        return *this;
    }

    // Equality operator
    inline bool operator==(const TVector<T>& rhs) const
    {
        return X == rhs.X && Y == rhs.Y && Z == rhs.Z && W == rhs.W;
    }
    inline bool operator!=(const TVector<T>& rhs) const
    {
        return X != rhs.X || Y != rhs.Y || Z != rhs.Z || W != rhs.W;
    }

    // Allows index access
    inline T operator[](unsigned index) const
    {
        DEBUG_ASSERT(index <= 3, "index for operator[] must be 0 to 3");
        return (&X)[index];
    }
    inline T& operator[](unsigned index)
    {
        DEBUG_ASSERT(index <= 3, "index for operator[] must be 0 to 3");
        return (&X)[index];
    }

    // Allows the vector to be used directly as an array of floats
    inline operator const T*() const
    {
        return &X;
    }
};
//---------------------------------------------------------------------------
// Float vector
typedef TVector<float>      AVector;
// Vector array type
typedef TDynArray<AVector>  AVectorArray;
//---------------------------------------------------------------------------
// Vector functions
namespace NVector
{
    // Convert vector types
    template <typename T> inline TVector<float> ToSingle(const TVector<T>& src)
    {
        return TVector<float>(src.X, src.Y, src.Z, src.W);
    }
    template <typename T> inline TVector<double> ToDouble(const TVector<T>& src)
    {
        return TVector<double>(src.X, src.Y, src.Z, src.W);
    }
    // Calculate dot product
    template <typename T> inline T TDot2D(const TVector<T>& a, const TVector<T>& b)
    {
        return a.X*b.X + a.Y*b.Y;
    }
    template <typename T> inline T TDot3D(const TVector<T>& a, const TVector<T>& b)
    {
        return a.X*b.X + a.Y*b.Y + a.Z*b.Z;
    }
    template <typename T> inline T TDot4D(const TVector<T>& a, const TVector<T>& b)
    {
        return a.X*b.X + a.Y*b.Y + a.Z*b.Z + a.W*b.W;
    }

    // Calculate cross product
    template <typename T> inline TVector<T> TCross3D(const TVector<T>& a, const TVector<T>& b)
    {
        return TVector<T>(
            a.Y*b.Z - a.Z*b.Y,
            a.Z*b.X - a.X*b.Z,
            a.X*b.Y - a.Y*b.X
        );
    }

    // Calculate the distance between two vectors
    template <typename T> inline T TDistance2D(const TVector<T>& a, const TVector<T>& b)
    {
        return (b - a).GetLength2D();
    }
    template <typename T> inline T TDistance3D(const TVector<T>& a, const TVector<T>& b)
    {
        return (b - a).GetLength3D();
    }
    template <typename T> inline T TDistance4D(const TVector<T>& a, const TVector<T>& b)
    {
        return (b - a).GetLength4D();
    }
    template <typename T> inline T TDistanceSquared2D(const TVector<T>& a, const TVector<T>& b)
    {
        return (b - a).GetLengthSquared2D();
    }
    template <typename T> inline T TDistanceSquared3D(const TVector<T>& a, const TVector<T>& b)
    {
        return (b - a).GetLengthSquared3D();
    }
    template <typename T> inline T TDistanceSquared4D(const TVector<T>& a, const TVector<T>& b)
    {
        return (b - a).GetLengthSquared4D();
    }

    // Returns true if the given vector is inside the given box
    template <typename T> inline bool TInBox2D(const TVector<T>& v, const TVector<T>& a, const TVector<T>& b)
    {
        return v.X >= a.X && v.Y >= a.Y &&
               v.X <= b.X && v.Y <= b.Y;
    }
    template <typename T> inline bool TInBox3D(const TVector<T>& v, const TVector<T>& a, const TVector<T>& b)
    {
        return v.X >= a.X && v.Y >= a.Y && v.Z >= a.Z &&
               v.X <= b.X && v.Y <= b.Y && v.Z <= b.Z;
    }
    template <typename T> inline bool TInBox4D(const TVector<T>& v, const TVector<T>& a, const TVector<T>& b)
    {
        return v.X >= a.X && v.Y >= a.Y && v.Z >= a.Z && v.W >= a.W &&
               v.X <= b.X && v.Y <= b.Y && v.Z <= b.Z && v.W <= b.W;
    }
    
    // Returns the normal of a triangle defined by three vectors
    template <typename T> inline TVector<T> TCalcTriangleNormal3D(const TVector<T>& a, const TVector<T>& b, const TVector<T>& c)
    {
        return TCross3D<T>(b - a, c - a).GetNormalized3D();
    }
}
//---------------------------------------------------------------------------
// Vector hash function
namespace NHash
{
    template <typename T> inline unsigned Func(const TVector<T>& v)
    {
        return NHash::Raw((const char*)&v, sizeof(v));
    }
}
//---------------------------------------------------------------------------
#endif

