//---------------------------------------------------------------------------
#ifdef __BORLANDC__
#include <vcl\vcl.h>
#pragma hdrstop
#endif
#include "OpenGL.h"
#include "Textures.h"
#include "UtilsUnit.h"
//---------------------------------------------------------------------------
// CTexture Implementation
CTexture::CTexture()
    : m_id(0)
    , m_width(0)
    , m_height(0)
    , m_format(TF_RGB)
    , m_thumbnail(NULL)
{
}
//---------------------------------------------------------------------------
CTexture::CTexture(const AString& name)
    : m_name(name)
    , m_id(0)
    , m_width(0)
    , m_height(0)
    , m_format(TF_RGB)
    , m_thumbnail(NULL)
{
}
//---------------------------------------------------------------------------
CTexture::~CTexture()
{
    DEBUG_ASSERT(m_id == 0, "Destroying an unreleased texture");
}
//---------------------------------------------------------------------------
bool CTexture::LoadImage(const AString& path)
{
    // Release any existing image
    Release();
    
#ifdef __BORLANDC__
    // Try to load a picture
    TPicture* pic = new TPicture();
    try
    {
        pic->LoadFromFile(path.GetText());
    }
    catch(...)
    {
        delete pic;
        return false;
    }

    // Try to access the data
    Graphics::TBitmap* bmp = pic->Bitmap;
    if (!bmp)
    {
        delete pic;
        return false;
    }

    // Set the bitmap
    bool ret = SetFromBitmap(pic->Bitmap);
    if (ret) m_name = NUtilities::ExtractFileNameOnly(path).GetText();
    delete pic;

    return ret;
#else // non-C++ Builder code
    return false;
#endif
}
//---------------------------------------------------------------------------
#ifdef __BORLANDC__
bool CTexture::SetFromBitmap(Graphics::TBitmap* bitmap, int maskColor)
{
    DEBUG_ASSERT(bitmap, "null bitmap was given");

    bool ret = false;

    // Create desktop compatible DC
    HDC ddc = GetDC(0);
    HDC dc = CreateCompatibleDC(ddc);
    ReleaseDC(0, ddc);

    HGDIOBJ oldBmp = SelectObject(dc, bitmap->Handle);

    // Grab the pixels from the bitmap
    BITMAPINFO bmi;
    ZeroMemory(&bmi, sizeof(bmi));
    bmi.bmiHeader.biSize = sizeof(bmi.bmiHeader);
    bmi.bmiHeader.biWidth = bitmap->Width;
    bmi.bmiHeader.biHeight = -bitmap->Height;
    bmi.bmiHeader.biPlanes = 1;
    bmi.bmiHeader.biBitCount = 32;
    bmi.bmiHeader.biCompression = BI_RGB;
    unsigned char* data = (unsigned char*)VirtualAlloc(NULL, bitmap->Width*bitmap->Height*4, MEM_COMMIT, PAGE_READWRITE);
    if (GetDIBits(dc, bitmap->Handle, 0, bitmap->Height, data, &bmi, DIB_RGB_COLORS))
    {
        unsigned char* texels = (unsigned char*)malloc(bitmap->Width*bitmap->Height*(maskColor == -1 ? 3 : 4));
        unsigned char* originalTexels = texels;
        for (int y=0; y < bitmap->Height; ++y)
        {
            const unsigned char* scanline = data + y*bitmap->Width*4;
            for (int x=0; x < bitmap->Width; ++x)
            {
                if (maskColor == (scanline[2]|(scanline[1]<<8)|(scanline[0]<<16)))
                {
                    texels[0] = 64;
                    texels[1] = 100;
                    texels[2] = 128;
                    texels[3] = 0;
                }
                else
                {
                    texels[0] = scanline[2];
                    texels[1] = scanline[1];
                    texels[2] = scanline[0];
                    if (maskColor != -1) texels[3] = 255;
                }
                texels += maskColor == -1 ? 3 : 4;
                scanline += 4;
            }
        }
        SetTexels(bitmap->Width, bitmap->Height, maskColor == -1 ? TF_RGB : TF_RGBA, true, originalTexels);
        free(originalTexels);
        ret = true; // Success
    }

    // Release memory
    VirtualFree(data, 0, MEM_RELEASE);
    SelectObject(dc, oldBmp);
    DeleteDC(dc);

    return ret;
}
#endif
//---------------------------------------------------------------------------
#ifdef __BORLANDC__
bool CTexture::SetFromImageList(Controls::TImageList* il, int index, int maskColor)
{
    DEBUG_ASSERT(il, "null imagelist was given");
    DEBUG_ASSERT(index >= 0 && index < il->Count, "invalid imagelist index");

    // Create bitmap for imagelist
    Graphics::TBitmap* bmp = new Graphics::TBitmap();

    // Load the bitmap data from the imagelist
    il->GetBitmap(index, bmp);
    bool ret = SetFromBitmap(bmp, 0xFF00FF);

    // Destroy the bitmap
    delete bmp;

    return ret;
}
#endif
//---------------------------------------------------------------------------
void CTexture::SetTexels(unsigned width, unsigned height, ETextureFormat format, bool genMips, const void* data)
{
    DEBUG_ASSERT(width > 0, "width cannot be zero");
    DEBUG_ASSERT(height > 0, "height cannot be zero");
    DEBUG_ASSERT(format == TF_RGB || format == TF_RGBA, "only RGB and RGBA texture formats are supported");
    DEBUG_ASSERT(data, "null data was passed");

    // Generate texture ID if necessary
    if (!m_id) GLCALL(glGenTextures(1, &m_id));

    // Upload image
    DEBUG_ASSERT(m_id != 0, "texture id is 0");
    GLCALL(glBindTexture(GL_TEXTURE_2D, m_id));
    GLCALL(glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR));
    GLCALL(glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, genMips ? GL_LINEAR_MIPMAP_LINEAR : GL_LINEAR));
    if (genMips)
    {
        GLCALL(gluBuild2DMipmaps(GL_TEXTURE_2D, format == TF_RGB ? GL_RGB : GL_RGBA, width, height, format == TF_RGB ? GL_RGB : GL_RGBA, GL_UNSIGNED_BYTE, data));
    }
    else
    {
        GLCALL(glTexImage2D(GL_TEXTURE_2D, 0, format == TF_RGB ? GL_RGB : GL_RGBA, width, height, 0, format == TF_RGB ? GL_RGB : GL_RGBA, GL_UNSIGNED_BYTE, data));
    }

    // Store data
    m_width = width;
    m_height = height;
    m_format = format;

    // Generate thumbnail
    delete m_thumbnail;
    m_thumbnail = new Graphics::TBitmap;
    m_thumbnail->Width = 64;
    m_thumbnail->Height = 64;
    int cw = format == TF_RGB ? 3 : 4;
    for (int y=0; y < 64; ++y)
    {
        for (int x=0; x < 64; ++x)
        {
            int tx = x*width / 64;
            int ty = y*height / 64;
            DWORD col = (((unsigned char*)data)[(ty*width + tx)*cw])|
                ((((unsigned char*)data + 1)[(ty*width + tx)*cw]) << 8)|
                ((((unsigned char*)data + 2)[(ty*width + tx)*cw]) << 16);
            m_thumbnail->Canvas->Pixels[x][y] = TColor(col);
        }
    }
}
//---------------------------------------------------------------------------
void CTexture::Release()
{
    if (m_id) glDeleteTextures(1, &m_id);
    m_id = 0;
    delete m_thumbnail;
    m_thumbnail = NULL;
    m_width = m_height = 0;
    m_format = TF_RGB;
}
//---------------------------------------------------------------------------
// CTextureSet Implementation
CTextureSet::~CTextureSet()
{
    DEBUG_ASSERT(m_textures.IsEmpty(), "Destroying an unreleased texture set");
}
//---------------------------------------------------------------------------
void CTextureSet::Release()
{
    for (unsigned i=0; i < m_textures.GetSize(); ++i)
    {
        m_textures[i]->Release();
        delete m_textures[i];
    }
    m_textures.Clear();
}
//---------------------------------------------------------------------------
void CTextureSet::Add(CTexture* texture)
{
    DEBUG_ASSERT(texture, "null texture was given");
    m_textures.Add(texture);
}
//---------------------------------------------------------------------------
void CTextureSet::ScanDirectory(const AString& path)
{
    WIN32_FIND_DATA fd;
    ZeroMemory(&fd, sizeof(fd));
    HANDLE h = FindFirstFile((path + "\\*.bmp").GetText(), &fd);
    if (h != INVALID_HANDLE_VALUE)
    {
        do
        {
            CTexture* texture = new CTexture();
            if (texture->LoadImage(path + "\\" + fd.cFileName))
            {
                Add(texture);
            }
            else
            {
                delete texture;
            }
        }
        while (FindNextFile(h, &fd));
        FindClose(h);
    }
}
//---------------------------------------------------------------------------
CTexture* CTextureSet::FindTexture(const AString& name) const
{
    for (unsigned i=0; i < m_textures.GetSize(); ++i)
    {
        if (name == m_textures[i]->GetName())
        {
            return m_textures[i];
        }
    }
    return NULL;
}
//---------------------------------------------------------------------------
