//---------------------------------------------------------------------------
#ifndef ShapeH
#define ShapeH
//---------------------------------------------------------------------------
#include "DynArray.h"
#include "Segment.h"
#include "Plane.h"
//---------------------------------------------------------------------------
// A 2D shape point
template <typename T> struct TShapePoint
{
    T   X, Y;

    inline TShapePoint(){}
    inline TShapePoint(T x, T y)
        : X(x), Y(y)
    {}
    inline TShapePoint(const TShapePoint& rhs)
        : X(rhs.X), Y(rhs.Y)
    {}
    inline TShapePoint(const TVector<T>& vec)
        : X(vec.X), Y(vec.Y)
    {}
    inline operator TVector<T>() const
    {
        return TVector<T>(X, Y, 0.0, 1.0);
    }
    inline bool operator==(const TShapePoint& rhs) const
    {
        return X == rhs.X && Y == rhs.Y;
    }
};
//---------------------------------------------------------------------------
// A 2D shape
template <typename T> class TShape
{
public:
    inline TShape(){}
    inline TShape(const TShape<T>& rhs)
    {
        operator=(rhs);
    }
    // Add a new point to the shape
    inline void Add(const TShapePoint<T>& p)
    {
        m_points.Add(p);
    }
    // Remove the point at the given index
    inline void Remove(unsigned index)
    {
        m_points.RemoveAt(index);
    }
    // Remove all points
    inline void Clear()
    {
        m_points.Clear();
    }
    // Cut a hole in the shape
    inline void CutHole(const ::TShape<T>& hole)
    {
        // Get points
        const TDynArray< TShapePoint<T> >& points = hole.GetPoints();
        // Validate state and input
        DEBUG_ASSERT(m_points.GetSize() > 2, "Invalid shape");
        DEBUG_ASSERT(points.GetSize() > 2, "Invalid hole shape");
        // Find index of the closest vertex to the hole
        AVector p0(points[0]);
        unsigned closestIndex = 0;
        T closestDistance = 0;
        bool found = false;
        for (unsigned i=0; i < m_points.GetSize(); ++i)
        {
            T distance = NVector::TDistance2D(p0, static_cast< TVector<T> >(m_points[i]));
            if (!found || distance < closestDistance)
            {
                found = true;
                closestDistance = distance;
                closestIndex = 0;
            }
        }
        // Create new points array
        DEBUG_ASSERT(found, "failed to find the closest vertex");
        TDynArray< TShapePoint<T> > newPoints;
        for (unsigned i=0; i <= closestIndex; i++)
            newPoints.Add(m_points[i]);
        for (int i=points.GetSignedSize() - 1; i >= 0; i--)
            newPoints.Add(points[i]);
        newPoints.Add(points.Last());
        for (unsigned i=closestIndex; i < m_points.GetSize(); i++)
            newPoints.Add(m_points[i]);
        // Exchange points
        m_points.Exchange(newPoints);
    }
    // Convert the shape to a series of triangles
    inline void ToTriangles(TDynArray< TVector<T> >& tris) const
    {
        DEBUG_ASSERT(m_points.GetSize() > 2, "invalid shape");
        TDynArray< TSegment<T> > edges;
        // Create edges
        edges.Reserve(m_points.GetSize());
        for (unsigned i=0; i < m_points.GetSize() - 1; ++i)
            edges.Add(TSegment<T>(m_points[i], m_points[i + 1]));
        edges.Add(TSegment<T>(m_points.Last(), m_points.First()));
        // Clip ear triangles until there are only three edges left
        while (edges.GetSize() > 3)
        {
            // Find ear
            bool found = false;
            unsigned index = 0;
            for (unsigned i=0; i < edges.GetSize(); ++i)
            {
                const TVector<T> a = edges[i].A;
                const TVector<T> b = edges[(i+1)%edges.GetSize()].B;
                const TVector<T> c = TVector<T>(b.X, b.Y, 1);
                const TPlane<T> p(a, b, c);
                if (NPlane::TSignedDistance(p, edges[i].B) >= 0.0 &&
                    IsEar(edges, a, edges[i].B, b))
                {
                    found = true;
                    index = (i + 1)%edges.GetSize();
                    break;
                }
            }
            if (!found) break;
            unsigned previous = index > 0 ? index - 1 : edges.GetSize() - 1;
            tris.Add(edges[previous].A);
            tris.Add(edges[index].A);
            tris.Add(edges[index].B);
            edges[previous].B = edges[index].B;
            edges.RemoveAt(index);
        }
        // Add last
        tris.Add(edges[0].A);
        tris.Add(edges[1].A);
        tris.Add(edges[2].A);
    }
    // Add the shape to a vector array
    inline void AddToVectorArray(TDynArray< TVector<T> >& va) const
    {
        for (unsigned i=0; i < m_points.GetSize(); ++i)
            va.Add(m_points[i]);
    }
    // Return the points in this shape
    inline const TDynArray< TShapePoint<T> >& GetPoints() const { return m_points; }
    // Assignment operator
    inline TShape<T>& operator=(const TShape<T>& rhs)
    {
        m_points.Clear();
        m_points.Add(rhs.m_points);
    }
    // Comparison operator
    inline bool operator==(const TShape<T>& rhs) const
    {
        return m_points == rhs.m_points;
    }
    inline bool operator!=(const TShape<T>& rhs) const
    {
        return m_points != rhs.m_points;
    }
private:
    TDynArray< TShapePoint<T> > m_points;   // Points that make up the shape

    // Return true if the two edges given is an ear in the given edge array
    inline static bool IsEar(const TDynArray< TSegment<T> >& edges,
        const TVector<T>& a, const TVector<T>& b, const TVector<T>& c)
    {
        // Check if the edges cross the A-C ear edge
        TSegment<T> ac(a, c);
        for (unsigned i=0; i < edges.GetSize(); ++i)
        {
            TVector<T> ip;
            if (NSegment::TSegmentIntersection2D(ac, edges[i], ip))
            {
                if (NVector::TDistanceSquared2D(ip, edges[i].A) > 0.0001 &&
                    NVector::TDistanceSquared2D(ip, edges[i].B) > 0.0001 &&
                    NVector::TDistanceSquared2D(ip, a) > 0.0001 &&
                    NVector::TDistanceSquared2D(ip, c) > 0.0001)
                    return false;
            }
        }
        // Construct a convex space made of the triangle edges
        TPlane<T> p1(a, b, AVector(b.X, b.Y, 1));
        TPlane<T> p2(b, c, AVector(c.X, c.Y, 1));
        TPlane<T> p3(c, a, AVector(a.X, a.Y, 1));
        // Check if any of the edge vertices is inside the triangle space
        for (unsigned i=0; i < edges.GetSize(); ++i)
        {
            if (NPlane::TSignedDistance(p1, edges[i].A) < -0.0001 &&
                NPlane::TSignedDistance(p2, edges[i].A) < -0.0001 &&
                NPlane::TSignedDistance(p3, edges[i].A) < -0.0001)
                return false;
        }
        // It is an ear
        return true;
    }
};
//---------------------------------------------------------------------------
// Aliases for float
typedef ::TShapePoint<float>    AShapePoint;
typedef ::TShape<float>         AShape;
//---------------------------------------------------------------------------
#endif
