//---------------------------------------------------------------------------
#ifndef SegmentH
#define SegmentH
//---------------------------------------------------------------------------
#include "Vector.h"
//---------------------------------------------------------------------------
// A line segment
template <typename T> struct TSegment
{
    TVector<T>  A, B;
    TSegment(){}
    TSegment(T x1, T y1, T z1, T w1, T x2, T y2, T z2, T w2)
        : A(x1, y1, z1, w1), B(x2, y2, z2, w2)
    {}
    TSegment(T x1, T y1, T z1, T x2, T y2, T z2)
        : A(x1, y1, z1), B(x2, y2, z2)
    {}
    TSegment(T x1, T y1, T x2, T y2)
        : A(x1, y1), B(x2, y2)
    {}
    TSegment(const TVector<T>& a, const TVector<T>& b)
        : A(a), B(b)
    {}
    TSegment(const TSegment<T>& rhs)
        : A(rhs.A), B(rhs.B)
    {}

    // Return the length of the line segment
    inline T GetLength2D() const
    {
        return (B - A).GetLength2D();
    }
    inline T GetLength3D() const
    {
        return (B - A).GetLength3D();
    }
    inline T GetLength4D() const
    {
        return (B - A).GetLength4D();
    }
    inline T GetLengthSquared2D() const
    {
        return (B - A).GetLengthSquared2D();
    }
    inline T GetLengthSquared3D() const
    {
        return (B - A).GetLengthSquared3D();
    }
    inline T GetLengthSquared4D() const
    {
        return (B - A).GetLengthSquared4D();
    }

    // Return the direction normal from A to B
    inline TVector<T> GetDirection2D() const
    {
        return (B - A).GetNormalized2D();
    }
    inline TVector<T> GetDirection3D() const
    {
        return (B - A).GetNormalized3D();
    }
    inline TVector<T> GetDirection4D() const
    {
        return (B - A).GetNormalized4D();
    }

    // Return the offset vector from A to B
    inline TVector<T> GetOffset() const
    {
        return B - A;
    }

    // Swap the two points
    inline void SwapPoints()
    {
        T tmp;
        tmp = A.X; A.X = B.X; B.X = tmp;
        tmp = A.Y; A.Y = B.Y; B.Y = tmp;
        tmp = A.Z; A.Z = B.Z; B.Z = tmp;
        tmp = A.W; A.W = B.W; B.W = tmp;
    }

    // Comparison operators
    inline bool operator==(const TSegment<T>& rhs) const
    {
        return A==rhs.A && B==rhs.B;
    }
    inline bool operator!=(const TSegment<T>& rhs) const
    {
        return !(operator==(rhs));
    }
};
//---------------------------------------------------------------------------
// Float line segment
typedef TSegment<float>     ASegment;
//---------------------------------------------------------------------------
// Segment functions
namespace NSegment
{
    // Convert segment types
    template <typename T> inline TSegment<float> ToSingle(const TSegment<T>& src)
    {
        return TSegment<float>(NVector::ToSingle(src.A), NVector::ToSingle(src.B));
    }
    template <typename T> inline TSegment<double> ToDouble(const TSegment<T>& src)
    {
        return TSegment<double>(NVector::ToDouble(src.A), NVector::ToDouble(src.B));
    }

    template <typename T> inline bool TSegmentIntersection2D(const TSegment<T>& s1, const TSegment<T>& s2, TVector<T>& ip)
    {
        T ubt = ((s2.B.Y-s2.A.Y)*(s1.B.X-s1.A.X))-((s2.B.X-s2.A.X)*(s1.B.Y-s1.A.Y));
        if (fabs(ubt) > 0.0001)
        {
            T ua = (((s2.B.X-s2.A.X)*(s1.A.Y-s2.A.Y))-((s2.B.Y-s2.A.Y)*(s1.A.X-s2.A.X)))/ubt;
            if (ua >= 0.0 && ua <= 1.0)
            {
                T ub = (((s1.B.X-s1.A.X)*(s1.A.Y-s2.A.Y))-((s1.B.Y-s1.A.Y)*(s1.A.X-s2.A.X)))/ubt;
                if (ub >= 0.0 && ub <= 1.0)
                {
                    ip = s1.A + s1.GetOffset()*ua;
                    return true;
                }
            }
        }
        return false;
    }
    template <typename T> inline bool TSegmentIntersection2D(const TSegment<T>& s1, const TSegment<T>& s2)
    {
        T ubt = ((s2.B.Y-s2.A.Y)*(s1.B.X-s1.A.X))-((s2.B.X-s2.A.X)*(s1.B.Y-s1.A.Y));
        if (fabs(ubt) > 0.0001)
        {
            T ua = (((s2.B.X-s2.A.X)*(s1.A.Y-s2.A.Y))-((s2.B.Y-s2.A.Y)*(s1.A.X-s2.A.X)))/ubt;
            if (ua >= 0.0 && ua <= 1.0)
            {
                T ub = (((s1.B.X-s1.A.X)*(s1.A.Y-s2.A.Y))-((s1.B.Y-s1.A.Y)*(s1.A.X-s2.A.X)))/ubt;
                if (ub >= 0.0 && ub <= 1.0)
                    return true;
            }
        }
        return false;
    }
    template <typename T> inline T TSegmentDistance3D(const TSegment<T>& s1, const TSegment<T>& s2)
    {
        TVector<T>  u = s1.B - s1.A;
        TVector<T>  v = s2.B - s2.A;
        TVector<T>  w = s1.A - s2.A;
        double      a = NVector::TDot3D(u, u);      // Use doubles regardless of template type
        double      b = NVector::TDot3D(u, v);      // to avoid imprecision issues with the
        double      c = NVector::TDot3D(v, v);      // squares of very large numbers
        double      d = NVector::TDot3D(u, w);
        double      e = NVector::TDot3D(v, w);
        double      D = a*c - b*b;
        double      sc, sN, sD = D;
        double      tc, tN, tD = D;
        if (D > 0.00001)
        {
            sN = (b*e - c*d);
            tN = (a*e - b*d);
            if (sN < 0.0)
            {
                sN = 0.0;
                tN = e;
                tD = c;
            }
            else if (sN > sD)
            {
                sN = sD;
                tN = e + b;
                tD = c;
            }
        }
        else
        {
            sN = 0.0;
            sD = 1.0;
            tN = e;
            tD = c;
        }
        if (tN < 0.0)
        {
            tN = 0.0;
            if (-d < 0.0) sN = 0.0;
            else if (-d > a) sN = sD;
            else
            {
                sN = -D;
                sD = a;
            }
        }
        else
        {
            tN = tD;
            if ((-d + b) < 0.0) sN = 0;
            else if ((-d + b) > a) sN = sD;
            else
            {
                sN = (-d +  b);
                sD = a;
            }
        }
        sc = (fabs(sN) < 0.00001 ? 0.0 : sN / sD);
        tc = (fabs(tN) < 0.00001 ? 0.0 : tN / tD);
        return TVector<T>(w + (u*sc) - (v*tc)).GetLength3D();
    }
    template <typename T> inline T TSegmentDistance3D(const TSegment<T>& s1, const TSegment<T>& s2, AVector& p1, AVector& p2)
    {
        TVector<T>  u = s1.B - s1.A;
        TVector<T>  v = s2.B - s2.A;
        TVector<T>  w = s1.A - s2.A;
        double      a = NVector::TDot3D(u, u);      // Use doubles regardless of template type
        double      b = NVector::TDot3D(u, v);      // to avoid imprecision issues with the
        double      c = NVector::TDot3D(v, v);      // squares of very large numbers
        double      d = NVector::TDot3D(u, w);
        double      e = NVector::TDot3D(v, w);
        double      D = a*c - b*b;
        double      sc, sN, sD = D;
        double      tc, tN, tD = D;
        if (D > 0.00001)
        {
            sN = (b*e - c*d);
            tN = (a*e - b*d);
            if (sN < 0.0)
            {
                sN = 0.0;
                tN = e;
                tD = c;
            }
            else if (sN > sD)
            {
                sN = sD;
                tN = e + b;
                tD = c;
            }
        }
        else
        {
            sN = 0.0;
            sD = 1.0;
            tN = e;
            tD = c;
        }
        if (tN < 0.0)
        {
            tN = 0.0;
            if (-d < 0.0) sN = 0.0;
            else if (-d > a) sN = sD;
            else
            {
                sN = -D;
                sD = a;
            }
        }
        else if (tN > tD)
        {
            tN = tD;
            if ((-d + b) < 0.0) sN = 0;
            else if ((-d + b) > a) sN = sD;
            else
            {
                sN = (-d + b);
                sD = a;
            }
        }
        sc = (fabs(sN) < 0.00001 ? 0.0 : sN / sD);
        tc = (fabs(tN) < 0.00001 ? 0.0 : tN / tD);
        p1 = s1.A + u*sc;
        p2 = s2.A + v*tc;
        return TVector<T>(w + (u*sc) - (v*tc)).GetLength3D();
    }
}
//---------------------------------------------------------------------------
// Segment hash function
namespace NHash
{
    template <typename T> inline unsigned Func(const TSegment<T>& seg)
    {
        return NHash::Raw((const char*)&seg, sizeof(seg));
    }
}
//---------------------------------------------------------------------------
#endif
