//---------------------------------------------------------------------------
#ifndef RayH
#define RayH
//---------------------------------------------------------------------------
#include "Segment.h"
#include "Plane.h"
#include "Box.h"
void AddThePoint(double x, double y, double z);
//---------------------------------------------------------------------------
// Ray
template <typename T> struct TRay
{
    TVector<T>      O;      // Origin
    TVector<T>      D;      // Direction
    
    inline TRay() {}
    inline TRay(const TRay<T>& src)
    {
        operator=(src);
    }
    inline TRay(const TVector<T>& o, const TVector<T>& d)
        : O(o), D(d)
    {}
    inline TRay(T ox, T oy, T dx, T dy)
        : O(ox, oy), D(dx, dy)
    {}
    inline TRay(T ox, T oy, T oz, T dx, T dy, T dz)
        : O(ox, oy, oz), D(dx, dy, dz)
    {}
    inline TRay(T ox, T oy, T oz, T ow, T dx, T dy, T dz, T dw)
        : O(ox, oy, oz, ow), D(dx, dy, dz, dw)
    {}

    // Set the ray values from the given segment
    inline void FromSegment2D(const TSegment<T>& seg)
    {
        O = seg.A;
        D = seg.B;
        D -= O;
        D.Normalize2D();
    }
    inline void FromSegment2D(const TVector<T>& a, TVector<T>& b)
    {
        O = a;
        D = b;
        D -= O;
        D.Normalize2D();
    }
    inline void FromSegment2D(T ax, T ay, T bx, T by)
    {
        O.X = ax;
        O.Y = ay;
        D.X = bx;
        D.Y = by;
        D -= O;
        D.Normalize2D();
    }
    inline void FromSegment3D(const TSegment<T>& seg)
    {
        O = seg.A;
        D = seg.B;
        D -= O;
        D.Normalize3D();
    }
    inline void FromSegment3D(const TVector<T>& a, const TVector<T>& b)
    {
        O = a;
        D = b;
        D -= O;
        D.Normalize3D();
    }
    inline void FromSegment3D(T ax, T ay, T az, T bx, T by, T bz)
    {
        O.X = ax;
        O.Y = ay;
        O.Z = az;
        D.X = bx;
        D.Y = by;
        D.Z = bz;
        D -= O;
        D.Normalize3D();
    }
    inline void FromSegment4D(const TSegment<T>& seg)
    {
        O = seg.A;
        D = seg.B;
        D -= O;
        D.Normalize4D();
    }
    inline void FromSegment4D(const TVector<T>& a, TVector<T>& b)
    {
        O = a;
        D = b;
        D -= O;
        D.Normalize4D();
    }
    inline void FromSegment4D(T ax, T ay, T az, T aw, T bx, T by, T bz, T bw)
    {
        O.X = ax;
        O.Y = ay;
        O.Z = az;
        O.W = aw;
        D.X = bx;
        D.Y = by;
        D.Z = bz;
        D.W = bw;
        D -= O;
        D.Normalize4D();
    }
    
    // Assignment
    inline TRay<T>& operator=(const TRay<T>& rhs)
    {
        if (this != &rhs)
        {
            O = rhs.O;
            D = rhs.D;
        }
        return *this;
    }

    // Equality operator
    inline bool operator==(const TRay<T>& rhs) const
    {
        return O == rhs.O && D == rhs.D;
    }
    inline bool operator!=(const TRay<T>& rhs) const
    {
        return O != rhs.O || D != rhs.D;
    }
};
//---------------------------------------------------------------------------
// Float ray
typedef TRay<float>         ARay;
// Ray array type
typedef TDynArray<ARay>     ARayArray;
//---------------------------------------------------------------------------
// Ray functions
namespace NRay
{
    // Utility functions to create segment objects directly
    template <typename T> inline TRay<T> TFromSegment2D(const TSegment<T>& seg)
    {
        TRay<T> r;
        r.FromSegment2D(seg);
        return r;
    }
    template <typename T> inline TRay<T> TFromSegment2D(const TVector<T>& a, const TVector<T>& b)
    {
        TRay<T> r;
        r.FromSegment2D(a, b);
        return r;
    }
    template <typename T> inline TRay<T> TFromSegment2D(T ax, T ay, T bx, T by)
    {
        TRay<T> r;
        r.FromSegment2D(ax, ay, bx, by);
        return r;
    }
    template <typename T> inline TRay<T> TFromSegment3D(const TSegment<T>& seg)
    {
        TRay<T> r;
        r.FromSegment3D(seg);
        return r;
    }
    template <typename T> inline TRay<T> TFromSegment3D(const TVector<T>& a, const TVector<T>& b)
    {
        TRay<T> r;
        r.FromSegment3D(a, b);
        return r;
    }
    template <typename T> inline TRay<T> TFromSegment3D(T ax, T ay, T az, T bx, T by, T bz)
    {
        TRay<T> r;
        r.FromSegment3D(ax, ay, az, bx, by, bz);
        return r;
    }
    template <typename T> inline TRay<T> TFromSegment4D(const TSegment<T>& seg)
    {
        TRay<T> r;
        r.FromSegment4D(seg);
        return r;
    }
    template <typename T> inline TRay<T> TFromSegment4D(const TVector<T>& a, const TVector<T>& b)
    {
        TRay<T> r;
        r.FromSegment4D(a, b);
        return r;
    }
    template <typename T> inline TRay<T> TFromSegment4D(T ax, T ay, T az, T aw, T bx, T by, T bz, T bw)
    {
        TRay<T> r;
        r.FromSegment4D(ax, ay, az, aw, bx, by, bz, bw);
        return r;
    }
    // Ray-sphere intersection
    template <typename T> inline bool TSphereHit2D(const TRay<T>& ray, const TVector<T>& c, T r, TVector<T>& ip, const T epsilon=0.00001f)
    {
        TVector<T> SR(ray.O - c);
        T B = NVector::TDot2D(SR, ray.D);
        T C = NVector::TDot2D(SR, SR) - r*r;
        T DD = B*B - C;
        if (DD > epsilon)
        {
            T E = (T)sqrt(DD);
            T t0 = -B-E;
            T t1 = -B+E;
            T t = TMin(t0, t1);
            if (t < (T)0.0) return false;
            ip = ray.O + ray.D*t;
            return true;
        }
        return false;
    }
    template <typename T> inline bool TSphereHit3D(const TRay<T>& ray, const TVector<T>& c, T r, TVector<T>& ip, const T epsilon=0.00001f)
    {
        TVector<double> SR(NVector::ToDouble(ray.O) - NVector::ToDouble(c));
        double B = NVector::TDot3D(SR, NVector::ToDouble(ray.D));
        double C = NVector::TDot3D(SR, SR) - (double)r*(double)r;
        double DD = B*B - C;
        if (DD > 0.0)
        {
            double E = sqrt(DD);
            double t0 = -B-E;
            double t1 = -B+E;
            double t = TMin(t0, t1);
            if (t < 0.0) return false;
            ip = TVector<T>((T)((double)ray.O.X + (double)ray.D.X*t),
                            (T)((double)ray.O.Y + (double)ray.D.Y*t),
                            (T)((double)ray.O.Z + (double)ray.D.Z*t));
            return true;
        }
        return false;
    }
    template <typename T> inline bool TSphereHit4D(const TRay<T>& ray, const TVector<T>& c, T r, TVector<T>& ip, const T epsilon=0.00001f)
    {
        TVector<T> SR(ray.O - c);
        T B = NVector::TDot4D(SR, ray.D);
        T C = NVector::TDot4D(SR, SR) - r*r;
        T DD = B*B - C;
        if (DD > epsilon)
        {
            T E = (T)sqrt(DD);
            T t0 = -B-E;
            T t1 = -B+E;
            T t = TMin(t0, t1);
            if (t < (T)0.0) return false;
            ip = ray.O + ray.D*t;
            return true;
        }
        return false;
    }
    // Ray-plane intersection
    template <typename T> inline bool TPlaneHit3D(const TRay<T>& ray, const TPlane<T>& plane, TVector<T>& ip)
    {
        T VD = NVector::TDot3D(plane, ray.D);
        if (VD == (T)0.0) return false;
        TVector<T> W(ray.O.X - plane.X*plane.W,
                     ray.O.Y - plane.Y*plane.W,
                     ray.O.Z - plane.Z*plane.W);
        T VN = -NVector::TDot3D(plane, W);
        T t(VN/VD);
        if (t < (T)0.0) return false;
        ip = ray.O + ray.D*t;
        return true;
    }
    // Ray-triangle intersection
    template <typename T> inline bool TTriangleHit3D(const TRay<T>& ray, const TVector<T>& a, const TVector<T>& b, const TVector<T>& c, TVector<T>& ip)
    {
        TVector<T> AB(b - a), AC(c - a);
        TPlane<T> P;
        P = NVector::TCross3D(AB, AC);
        P.Normalize3D();
        P.W = NVector::TDot3D(a, P);
        if (!TPlaneHit3D(ray, P, ip)) return false;
        T UU = NVector::TDot3D(AB, AB);
        T UV = NVector::TDot3D(AB, AC);
        T VV = NVector::TDot3D(AC, AC);
        T DD = UV*UV - UU*VV;
        if (DD == (T)0.0) return false;
        TVector<T> W(ip - a);
        T WU = W.X*AB.X + W.Y*AB.Y + W.Z*AB.Z;
        T WV = W.X*AC.X + W.Y*AC.Y + W.Z*AC.Z;
        T SV = (UV*WV - VV*WU)/DD;
        if (SV < (T)0.0 || SV > (T)1.0) return false;
        T TV = (UV*WU - UU*WV)/DD;
        if (TV < (T)0.0 || (SV + TV) > (T)1.0) return false;
        return true;
    }
    // Ray-box intersection
    template <typename T> inline bool TBoxHit3D(const TRay<T>& ray, const TBox<T>& box, T& t)
    {
        TVector<T> DF(((T)1.0)/ray.D.X, ((T)1.0)/ray.D.Y, ((T)1.0)/ray.D.Z);
        T t1 = (box.Min.X - ray.O.X)*DF.X;
        T t2 = (box.Max.X - ray.O.X)*DF.X;
        T t3 = (box.Min.Y - ray.O.Y)*DF.Y;
        T t4 = (box.Max.Y - ray.O.Y)*DF.Y;
        T t5 = (box.Min.Z - ray.O.Z)*DF.Z;
        T t6 = (box.Max.Z - ray.O.Z)*DF.Z;
        T tmin = TMax(TMax(TMin(t1, t2), TMin(t3, t4)), TMin(t5, t6));
        T tmax = TMin(TMin(TMax(t1, t2), TMax(t3, t4)), TMax(t5, t6));
        if (tmax < (T)0.0)
        {
            t = tmax;
            return false;
        }
        if (tmin > tmax)
        {
            t = tmax;
            return false;
        }
        t = tmin;
        return true;
    }
    template <typename T> inline bool TBoxHit3D(const TRay<T>& ray, const TBox<T>& box, TVector<T>& ip)
    {
        T t;
        bool r = TBoxHit3D(ray, box, t);
        ip = ray.D*t + ray.O;
        return r;
    }
}
#endif

