//---------------------------------------------------------------------------
#ifndef NodesH
#define NodesH
//---------------------------------------------------------------------------
#include "Strings.h"
//---------------------------------------------------------------------------
// Base class for nodes/tree hierarchies
class CNode
{
public:
    // Constructs an empty node
    CNode();
    virtual ~CNode();
    // Sets the name for this node
    virtual void SetName(const AString& name)
    {
        m_name = name;
    }
    // Returns true if the given node can become a child of this node
    virtual bool CanAddChild(CNode* child)
    {
        DEBUG_ASSERT(child, "Null child was given");
        return true;
    }
    // Adds the given node under this node, making it its children.  Returns
    // false if it is not possible to remove the given child
    bool AddChild(CNode* child);
    // Removes the given child from this node
    void RemoveChild(CNode* child);
    // Removes and deletes the given child
    void DeleteChild(CNode* child);
    // Removes all children from this node
    void RemoveChildren();
    // Removes and deletes all children of this node
    void DeleteChildren();
    // Returns true if the given node is a direct descendant of this node
    inline bool IsChild(CNode* node) const
    {
        DEBUG_ASSERT(node, "null node was given");
#ifdef DEBUG
        if (node->m_parent == this)
        {
            DEBUG_ASSERT(m_children.Contains(node), "the given node has its parent set to this but is not contained in the children array");
            return true;
        }
        else
        {
            return false;
        }
#else
        return node->m_parent == this;
#endif
    }
    // Returns true if the given node is an ancestor of the this node or - if
    // the includeThis argument is true - the same as this node
    bool IsAncestor(CNode* node, bool includeThis=true) const;
    // Returns true if the given node is a descendant of the this node or - if
    // the includeThis argument is true - the same as this node
    bool IsDescendant(CNode* node, bool includeThis=true) const;
    // Returns the first direct descendant that has the given name or null
    CNode* FindChild(const AString& name) const;
    // Collects all direct descendants that have the given name - note that
    // this method does not clear the passed array, so that it can be used with
    // multiple calls.  The max argument can be used to limit the number of
    // returned values or set to a negative value to not impose any limit
    void FindChildren(TDynArray<CNode*>& result, const AString& name, int max=-1) const;
    // Returns the descendant under the given path (a path is a dot separated
    // list of names, e.g. foo.bar.baz gives the child baz of the child bar of
    // the child foo of this node)
    CNode* GetChildByPath(const AString& path) const;
    // Calculates and returns the path to this node from the given ancestor or
    // from the root if null is given.  Returns false if any of the involved
    // nodes do not contain a name
    bool GetPath(AString& path, CNode* ancestor=NULL) const;
    // Collects all direct descendants of this node - note that this method
    // does not clear the passed array, so that it can be used with multiple
    // calls.  The max argument can be used to limit the number of returned
    // values or set to a negative value to not impose any limit.  If the
    // includeThis argument is set to true, this method will also add this
    // node to the result
    void GetAllDescendants(TDynArray<CNode*>& result, bool includeThis=false, int max=-1) const;
    // Returns the number of children in this node
    inline unsigned GetChildCount() const { return m_children.GetSize(); }
    // Returns the index-th child in this node
    inline CNode* GetChild(unsigned index) const { return m_children[index]; }
    // Returns the children of this node
    inline const TDynArray<CNode*>& GetChildren() const { return m_children; }
    // Returns the parent of this node
    inline CNode* GetParent() const { return m_parent; }
    // Returns true if this is a root node
    inline bool IsRoot() const { return !m_parent; }
    // Returns true if this is a leaf node
    inline bool IsLeaf() const { return m_children.IsEmpty(); }
    // Returns true if this node has children
    inline bool HasChildren() const { return m_children.HasItems(); }
    // Returns the name of this node
    inline const AString& GetName() const { return m_name; }
    // Returns true if this node has a name
    inline bool HasName() const { return !m_name.IsEmpty(); }
protected:
    // Called whenever a child is added to this node
    virtual void OnChildAdded(CNode* child){ (void)child; }
    // Called whenever a child is removed from this node (note that this is
    // also called whenever the children are removed and for each child when
    // the node itself is destroyed)
    virtual void OnChildRemoved(CNode* child){ (void)child; };
private:
    TDynArray<CNode*>   m_children; // Node's children
    CNode*              m_parent;   // Node's parent
    AString             m_name;     // Node's name
};
//---------------------------------------------------------------------------
#endif
