//---------------------------------------------------------------------------
#ifndef MatrixH
#define MatrixH
//---------------------------------------------------------------------------
#include "Vector.h"
//---------------------------------------------------------------------------
// 4x4 matrix
template <typename T> struct TMatrix
{
    union {
        T   M[4][4];
        T   A[16];
    };

    inline TMatrix() {}
    inline explicit TMatrix(const T a[])
    {
        memcpy(A, a, sizeof(T)*16);
    }
    inline TMatrix(const T a[], const T b[], const T c[], const T d[])
    {
        memcpy(M[0], a, sizeof(T)*4);
        memcpy(M[1], b, sizeof(T)*4);
        memcpy(M[2], c, sizeof(T)*4);
        memcpy(M[3], d, sizeof(T)*4);
    }
    inline TMatrix(const TVector<T>& a, const TVector<T>& b, const TVector<T>& c, const TVector<T>& d)
    {
        memcpy(M[0], &a, sizeof(TVector<T>));
        memcpy(M[1], &b, sizeof(TVector<T>));
        memcpy(M[2], &c, sizeof(TVector<T>));
        memcpy(M[3], &d, sizeof(TVector<T>));
    }

    // Transform a 3D vector
    inline TVector<T> Transform3D(const TVector<T>& v) const
    {
        return TVector<T>(
            v.X * M[0][0] + v.Y * M[1][0] + v.Z*M[2][0] + M[3][0],
            v.X * M[0][1] + v.Y * M[1][1] + v.Z*M[2][1] + M[3][1],
            v.X * M[0][2] + v.Y * M[1][2] + v.Z*M[2][2] + M[3][2]);
    }

    // Transform a 4D vector
    inline TVector<T> Transform4D(const TVector<T>& v) const
    {
        return TVector<T>(
            v.X * M[0][0] + v.Y * M[1][0] + v.Z*M[2][0] + v.W*M[3][0],
            v.X * M[0][1] + v.Y * M[1][1] + v.Z*M[2][1] + v.W*M[3][1],
            v.X * M[0][2] + v.Y * M[1][2] + v.Z*M[2][2] + v.W*M[3][2],
            v.X * M[0][3] + v.Y * M[1][3] + v.Z*M[2][3] + v.W*M[3][3]);
    }

    // Transform a 3D vector (usually normal) ignoring the translation
    // part of this matrix (as if the vector's W was 0)
    inline TVector<T> TransformNormal3D(const TVector<T>& v) const
    {
        return TVector<T>(
            v.X * M[0][0] + v.Y * M[1][0] + v.Z*M[2][0],
            v.X * M[0][1] + v.Y * M[1][1] + v.Z*M[2][1],
            v.X * M[0][2] + v.Y * M[1][2] + v.Z*M[2][2]);
    }

    // Return the matrix transposed
    inline TMatrix<T> Transposed() const
    {
        return TMatrix<T>(
            TVector<T>(M[0][0], M[1][0], M[2][0], M[3][0]),
            TVector<T>(M[0][1], M[1][1], M[2][1], M[3][1]),
            TVector<T>(M[0][2], M[1][2], M[2][2], M[3][2]),
            TVector<T>(M[0][3], M[1][3], M[2][3], M[3][3]));
    }

    // Transpose the matrix
    inline void Transpose()
    {
        *this = Transposed();
    }

    // Return the determinant
    inline T Determinant() const
    {
        return M[3][0]*M[2][1]*M[1][2]*M[0][3] - M[2][0]*M[3][1]*M[1][2]*M[0][3] -
               M[3][0]*M[1][1]*M[2][2]*M[0][3] + M[1][0]*M[3][1]*M[2][2]*M[0][3] +
               M[2][0]*M[1][1]*M[3][2]*M[0][3] - M[1][0]*M[2][1]*M[3][2]*M[0][3] -
               M[3][0]*M[2][1]*M[0][2]*M[1][3] + M[2][0]*M[3][1]*M[0][2]*M[1][3] +
               M[3][0]*M[0][1]*M[2][2]*M[1][3] - M[0][0]*M[3][1]*M[2][2]*M[1][3] -
               M[2][0]*M[0][1]*M[3][2]*M[1][3] + M[0][0]*M[2][1]*M[3][2]*M[1][3] +
               M[3][0]*M[1][1]*M[0][2]*M[2][3] - M[1][0]*M[3][1]*M[0][2]*M[2][3] -
               M[3][0]*M[0][1]*M[1][2]*M[2][3] + M[0][0]*M[3][1]*M[1][2]*M[2][3] +
               M[1][0]*M[0][1]*M[3][2]*M[2][3] - M[0][0]*M[1][1]*M[3][2]*M[2][3] -
               M[2][0]*M[1][1]*M[0][2]*M[3][3] + M[1][0]*M[2][1]*M[0][2]*M[3][3] +
               M[2][0]*M[0][1]*M[1][2]*M[3][3] - M[0][0]*M[2][1]*M[1][2]*M[3][3] -
               M[1][0]*M[0][1]*M[2][2]*M[3][3] + M[0][0]*M[1][1]*M[2][2]*M[3][3];
    }

    // Return this matrix inverted
    inline TMatrix<T> GetInverted() const
    {
        T inv[16], det;
        TMatrix<T> r;
        int i;

        inv[0] = A[5]  * A[10] * A[15] -
                 A[5]  * A[11] * A[14] -
                 A[9]  * A[6]  * A[15] +
                 A[9]  * A[7]  * A[14] +
                 A[13] * A[6]  * A[11] -
                 A[13] * A[7]  * A[10];

        inv[4] = -A[4]  * A[10] * A[15] +
                  A[4]  * A[11] * A[14] +
                  A[8]  * A[6]  * A[15] -
                  A[8]  * A[7]  * A[14] -
                  A[12] * A[6]  * A[11] +
                  A[12] * A[7]  * A[10];

        inv[8] = A[4]  * A[9] * A[15] -
                 A[4]  * A[11] * A[13] -
                 A[8]  * A[5] * A[15] +
                 A[8]  * A[7] * A[13] +
                 A[12] * A[5] * A[11] -
                 A[12] * A[7] * A[9];

        inv[12] = -A[4]  * A[9] * A[14] +
                   A[4]  * A[10] * A[13] +
                   A[8]  * A[5] * A[14] -
                   A[8]  * A[6] * A[13] -
                   A[12] * A[5] * A[10] +
                   A[12] * A[6] * A[9];

        inv[1] = -A[1]  * A[10] * A[15] +
                  A[1]  * A[11] * A[14] +
                  A[9]  * A[2] * A[15] -
                  A[9]  * A[3] * A[14] -
                  A[13] * A[2] * A[11] +
                  A[13] * A[3] * A[10];

        inv[5] = A[0]  * A[10] * A[15] -
                 A[0]  * A[11] * A[14] -
                 A[8]  * A[2] * A[15] +
                 A[8]  * A[3] * A[14] +
                 A[12] * A[2] * A[11] -
                 A[12] * A[3] * A[10];

        inv[9] = -A[0]  * A[9] * A[15] +
                  A[0]  * A[11] * A[13] +
                  A[8]  * A[1] * A[15] -
                  A[8]  * A[3] * A[13] -
                  A[12] * A[1] * A[11] +
                  A[12] * A[3] * A[9];

        inv[13] = A[0]  * A[9] * A[14] -
                  A[0]  * A[10] * A[13] -
                  A[8]  * A[1] * A[14] +
                  A[8]  * A[2] * A[13] +
                  A[12] * A[1] * A[10] -
                  A[12] * A[2] * A[9];

        inv[2] = A[1]  * A[6] * A[15] -
                 A[1]  * A[7] * A[14] -
                 A[5]  * A[2] * A[15] +
                 A[5]  * A[3] * A[14] +
                 A[13] * A[2] * A[7] -
                 A[13] * A[3] * A[6];

        inv[6] = -A[0]  * A[6] * A[15] +
                  A[0]  * A[7] * A[14] +
                  A[4]  * A[2] * A[15] -
                  A[4]  * A[3] * A[14] -
                  A[12] * A[2] * A[7] +
                  A[12] * A[3] * A[6];

        inv[10] = A[0]  * A[5] * A[15] -
                  A[0]  * A[7] * A[13] -
                  A[4]  * A[1] * A[15] +
                  A[4]  * A[3] * A[13] +
                  A[12] * A[1] * A[7] -
                  A[12] * A[3] * A[5];

        inv[14] = -A[0]  * A[5] * A[14] +
                   A[0]  * A[6] * A[13] +
                   A[4]  * A[1] * A[14] -
                   A[4]  * A[2] * A[13] -
                   A[12] * A[1] * A[6] +
                   A[12] * A[2] * A[5];

        inv[3] = -A[1] * A[6] * A[11] +
                  A[1] * A[7] * A[10] +
                  A[5] * A[2] * A[11] -
                  A[5] * A[3] * A[10] -
                  A[9] * A[2] * A[7] +
                  A[9] * A[3] * A[6];

        inv[7] = A[0] * A[6] * A[11] -
                 A[0] * A[7] * A[10] -
                 A[4] * A[2] * A[11] +
                 A[4] * A[3] * A[10] +
                 A[8] * A[2] * A[7] -
                 A[8] * A[3] * A[6];

        inv[11] = -A[0] * A[5] * A[11] +
                   A[0] * A[7] * A[9] +
                   A[4] * A[1] * A[11] -
                   A[4] * A[3] * A[9] -
                   A[8] * A[1] * A[7] +
                   A[8] * A[3] * A[5];

        inv[15] = A[0] * A[5] * A[10] -
                  A[0] * A[6] * A[9] -
                  A[4] * A[1] * A[10] +
                  A[4] * A[2] * A[9] +
                  A[8] * A[1] * A[6] -
                  A[8] * A[2] * A[5];

        det = 1.0/(A[0] * inv[0] + A[1] * inv[4] + A[2] * inv[8] + A[3] * inv[12]);

        for (i = 0; i < 16; i++)
            r.A[i] = inv[i] * det;

        return r;
    }

    // Invert this matrix
    inline void Invert()
    {
        *this = GetInverted();
    }

    // Allows index access
    inline TVector<T> operator[](unsigned index) const
    {
        DEBUG_ASSERT(index <= 3, "index for operator[] must be 0 to 3");
        return *reinterpret_cast<const TVector<T>*>(M[index]);
    }
    inline TVector<T>& operator[](unsigned index)
    {
        DEBUG_ASSERT(index <= 3, "index for operator[] must be 0 to 3");
        return *reinterpret_cast<TVector<T>*>(M[index]);
    }

    // Allows the matrix to be used directly as an array of floats
    inline operator const T*() const
    {
        return A;
    }
    inline operator T*()
    {
        return A;
    }

    // Comparison
    inline bool operator==(const TMatrix<T>& rhs) const
    {
        return A[0] == rhs.A[0] && A[1] == rhs.A[1] && A[2] == rhs.A[2] &&
               A[3] == rhs.A[3] && A[4] == rhs.A[4] && A[5] == rhs.A[5] &&
               A[6] == rhs.A[6] && A[7] == rhs.A[7] && A[8] == rhs.A[8] &&
               A[9] == rhs.A[9] && A[10] == rhs.A[10] && A[11] == rhs.A[11] &&
               A[12] == rhs.A[12] && A[13] == rhs.A[13] && A[14] == rhs.A[14] &&
               A[15] == rhs.A[15];
    }
    inline bool operator!=(const TMatrix<T>& rhs) const
    {
        return !(operator==(rhs));
    }
};
//---------------------------------------------------------------------------
// Float matrix
typedef TMatrix<float>  AMatrix;
//---------------------------------------------------------------------------
// Matrix functions
namespace NMatrix
{
    // Identity matrix
    extern const AMatrix    GIdentity;

    // Convert matrix types
    template <typename T> inline TMatrix<float> ToSingle(const TMatrix<T>& src)
    {
        float m[16];
        for (i=0; i < 16; i++) m[i] = src.M[i];
        return TMatrix<float>(m);
    }
    template <typename T> inline TMatrix<double> ToDouble(const TMatrix<T>& src)
    {
        double m[16];
        for (i=0; i < 16; i++) m[i] = src.M[i];
        return TMatrix<double>(m);
    }

    // Returns an identity matrix
    inline AMatrix GetIdentity()
    {
        return AMatrix(
            AVector(1.0f, 0.0f, 0.0f, 0.0f),
            AVector(0.0f, 1.0f, 0.0f, 0.0f),
            AVector(0.0f, 0.0f, 1.0f, 0.0f),
            AVector(0.0f, 0.0f, 0.0f, 1.0f));
    }

    // Returns a scale matrix
    inline AMatrix GetScale(const AVector& xyz)
    {
        return AMatrix(
            AVector(xyz.X, 0.0f, 0.0f, 0.0f),
            AVector(0.0f, xyz.Y, 0.0f, 0.0f),
            AVector(0.0f, 0.0f, xyz.Z, 0.0f),
            AVector(0.0f, 0.0f, 0.0f, xyz.W));
    }

    // Returns a perspective projection matrix
    inline AMatrix GetPerspective(float fov, float aspect, float znear, float zfar)
    {
        float zdelta = znear - zfar;
        float cot = cos(fov*0.5f)/tan(fov*0.5f);
        return AMatrix(
            AVector(cot/aspect, 0.0f, 0.0f, 0.0f),
            AVector(0.0f, cot, 0.0f, 0.0f),
            AVector(0.0f, 0.0f, (zfar + znear)/zdelta, -1.0f),
            AVector(0.0f, 0.0f, 2.0f*znear*zfar/zdelta, 0.0f));
    }
    
    // Returns an orthographic projection matrix
    inline AMatrix GetOrthographic(float left, float right, float top, float bottom, float znear, float zfar)
    {
        return AMatrix(
            AVector(2.0f/(right - left), 0.0f, 0.0f, 0.0f),
            AVector(0.0f, 2.0f/(top - bottom), 0.0f, 0.0f),
            AVector(0.0f, 0.0f, -2.0f/(zfar - znear), 0.0f),
            AVector(-(right + left)/(right - left),
                    -(top + bottom)/(top - bottom),
                    -(zfar + znear)/(zfar - znear),
                    1.0f));
    }

    // Returns a translation matrix
    inline AMatrix GetTranslation(const AVector& offset)
    {
        return AMatrix(
            AVector(1.0f, 0.0f, 0.0f, 0.0f),
            AVector(0.0f, 1.0f, 0.0f, 0.0f),
            AVector(0.0f, 0.0f, 1.0f, 0.0f),
            AVector(offset.X, offset.Y, offset.Z, 1.0f));
    }

    // Returns a rotation matrix using the given angle and axis normal
    inline AMatrix GetRotation(float angle, const AVector& axis)
    {
        const float s = (float)sin(angle);
        const float c = (float)cos(angle);
        const float oc = 1.0f - c;
        const float xx = axis.X * axis.X;
        const float yy = axis.Y * axis.Y;
        const float zz = axis.Z * axis.Z;
        const float xy = axis.X * axis.Y;
        const float yz = axis.Y * axis.Z;
        const float zx = axis.Z * axis.X;
        const float xs = axis.X * s;
        const float ys = axis.Y * s;
        const float zs = axis.Z * s;
        return AMatrix(
            AVector((oc*xx) + c, (oc*xy) + zs, (oc*zx) - ys, 0.0f),
            AVector((oc*xy) - zs, (oc*yy) + c, (oc*yz) + xs, 0.0f),
            AVector((oc*zx) + ys, (oc*yz) - xs, (oc*zz) + c, 0.0f),
            AVector(0.0f, 0.0f, 0.0f, 1.0f));
    }

    // Returns a matrix that points towards the given direction
    inline AMatrix GetDirection(const AVector& dir)
    {
        if (fabs(dir.Y) > 0.8)
        {
            const AVector back(0, 0, -1);
            AVector xAxis = NVector::TCross3D(back, dir);
            AVector yAxis = NVector::TCross3D(dir, xAxis);
            return AMatrix(xAxis, yAxis, dir, AVector(0, 0, 0, 1));
        }
        else
        {
            const AVector up(0, 1, 0);
            AVector xAxis = NVector::TCross3D(up, dir);
            AVector yAxis = NVector::TCross3D(dir, xAxis);
            return AMatrix(xAxis, yAxis, dir, AVector(0, 0, 0, 1));
        }
    }

    // Transforms two matrices
    inline AMatrix Transform(const AMatrix& a, const AMatrix& b)
    {
        AMatrix r;
        for (int i = 0; i < 4; ++i)
        {
            const AVector& v = a[i];
            r.M[i][0] = v.X*b.M[0][0] + v.Y*b.M[1][0] + v.Z*b.M[2][0] + v.W*b.M[3][0];
            r.M[i][1] = v.X*b.M[0][1] + v.Y*b.M[1][1] + v.Z*b.M[2][1] + v.W*b.M[3][1];
            r.M[i][2] = v.X*b.M[0][2] + v.Y*b.M[1][2] + v.Z*b.M[2][2] + v.W*b.M[3][2];
            r.M[i][3] = v.X*b.M[0][3] + v.Y*b.M[1][3] + v.Z*b.M[2][3] + v.W*b.M[3][3];
        }
        return r;
    }
    
    // Returns a look-at matrix
    inline AMatrix GetLookAt(const AVector& pos, const AVector& target, const AVector& up)
    {
        AVector f((target - pos).GetNormalized3D());
        AVector s(NVector::TCross3D(f, up).GetNormalized3D());
        AVector u(NVector::TCross3D(s, f));
        AMatrix tmp1(AVector(s.X, u.X, -f.X, 0.0f),
                     AVector(s.Y, u.Y, -f.Y, 0.0f),
                     AVector(s.Z, u.Z, -f.Z, 0.0f),
                     AVector(0, 0, 0, 1.0f));
        return Transform(GetTranslation(-pos), tmp1);
    }
}
//---------------------------------------------------------------------------
// Matrix hash function
namespace NHash
{
    template <typename T> inline unsigned Func(const TMatrix<T>& matrix)
    {
        return NHash::Raw((const char*)&matrix, sizeof(matrix));
    }
}
//---------------------------------------------------------------------------
#endif

