//---------------------------------------------------------------------------
#include <ctype.h>
#include "UtilsUnit.h"
#include "KeyValueTree.h"
//---------------------------------------------------------------------------
static int GetHexNibble(int v)
{
    DEBUG_ASSERT(v >= 0 && v < 16, "invalid hex nibble value");
    if (v < 10) return '0' + v;
    return 'A' + (v - 10);
}
//---------------------------------------------------------------------------
static int ParseHexNibble(int v)
{
    if (v >= '0' && v <= '9') return v - '0';
    if (v >= 'A' && v <= 'F') return v - 'A' + 10;
    if (v >= 'a' && v <= 'f') return v - 'a' + 10;
    DEBUG_FAIL("invalid character");
    return 0;
}
//---------------------------------------------------------------------------
CKVNode::CKVNode(const AString& name, const AString& value)
    : m_name(name)
    , m_value(value)
    , m_parent(NULL)
{
}
//---------------------------------------------------------------------------
CKVNode::~CKVNode()
{
    // Remove from parent
    if (m_parent)
    {
        m_parent->m_children.Remove(this);
    }
    // Delete children
    DeleteAll();
}
//---------------------------------------------------------------------------
CKVNode* CKVNode::DirectChildByName(const AString& name, bool createMissing)
{
    for (unsigned i=0; i < m_children.GetSize(); ++i)
        if (name == m_children[i]->m_name)
            return m_children[i];
    if (createMissing)
    {
        CKVNode* child = new CKVNode(name);
        m_children.Add(child);
        child->m_parent = this;
        return child;
    }
    return NULL;
}
//---------------------------------------------------------------------------
CKVNode* CKVNode::ChildByPath(const AString& path, bool createMissing)
{
    CKVNode* node = this;
    unsigned start = 0;
    while (true)
    {
        // Check if there is a dot separator
        unsigned dotPos = 0;
        if (path.FindFirstCharacter('.', dotPos, start))
        {
            // Find child
            node = node->DirectChildByName(path.GetSubstring(start, dotPos), createMissing);
            if (!node) return NULL;
            // Advance the search
            start = dotPos + 1;
        }
        else // no dot, reached the deepest level 
        {
            node = node->DirectChildByName(path.GetSubstring(start, path.GetLength()), createMissing);
            return const_cast<CKVNode*>(node);
        }
    }
}
//---------------------------------------------------------------------------
CKVNode* CKVNode::ChildByNameAndValue(const AString& name, const AString& value)
{
    // Check this node
    if (name == m_name && value == m_value) return this;
    // Check children
    for (unsigned i=0; i < m_children.GetSize(); ++i)
    {
        CKVNode* child = m_children[i]->ChildByNameAndValue(name, value);
        if (child) return child;
    }
    return NULL;
}
//---------------------------------------------------------------------------
CKVNode* CKVNode::ChildByGrandchild(const AString& grandchildName, bool recursive)
{
    // Check this node
    CKVNode* child = DirectChildByName(grandchildName);
    if (child) return this;
    // Check children
    if (recursive)
    {
        for (unsigned i=0; i < m_children.GetSize(); ++i)
        {
            child = m_children[i]->ChildByGrandchild(grandchildName, true);
            if (child) return child;
        }
    }
    return NULL;
}
//---------------------------------------------------------------------------
CKVNode* CKVNode::ChildByGrandchildAndValue(const AString& grandchildName, const AString& value, bool recursive)
{
    // Check this node
    CKVNode* child = DirectChildByName(grandchildName, false);
    if (child && child->GetValue() == value) return this;
    // Check children
    if (recursive)
    {
        for (unsigned i=0; i < m_children.GetSize(); ++i)
        {
            child = m_children[i]->ChildByGrandchildAndValue(grandchildName, value, true);
            if (child) return child;
        }
    }
    return NULL;
}
//---------------------------------------------------------------------------
CKVNode* CKVNode::CreateChild(const AString& name, const AString& value)
{
    CKVNode* child = new CKVNode(name, value);
    m_children.Add(child);
    child->m_parent = this;
    return child;
}
//---------------------------------------------------------------------------
void CKVNode::SetChildValue(const AString& path, const AString& value)
{
    CKVNode* child = ChildByPath(path);
    DEBUG_ASSERT(child, "somehow retrieving the child failed even though we asked all missing children to be created");
    child->SetValue(value);
}
//---------------------------------------------------------------------------
bool CKVNode::GetChildValue(const AString& path, AString& value) const
{
    CKVNode* child = const_cast<CKVNode*>(this)->ChildByPath(path, false);
    if (!child) return false;
    value = child->GetValue();
    return true;
}
//---------------------------------------------------------------------------
bool CKVNode::DeleteChildValue(const AString& path)
{
    CKVNode* child = ChildByPath(path, false);
    if (!child) return false;
    child->m_parent->m_children.Remove(child);
    child->m_parent = NULL;
    delete child;
    return true;
}
//---------------------------------------------------------------------------
void CKVNode::DeleteAll()
{
    for (unsigned i=0; i < m_children.GetSize(); i++)
    {
        m_children[i]->m_parent = NULL;
        delete m_children[i];
    }
    m_children.Clear();
}
//---------------------------------------------------------------------------
void CKVNode::Export(AString& target) const
{
    AStringBuilder builder;
    ExportNode(builder, this, 0);
    target.Clear();
    builder.MoveTo(target);
}
//---------------------------------------------------------------------------
bool CKVNode::Import(const AString& source, bool discardExisting)
{
    unsigned head = 0;
    if (discardExisting)
    {
        m_name.Clear();
        m_value.Clear();
        DeleteAll();
    }
    return ImportNode(source, this, head);
}
//---------------------------------------------------------------------------
bool CKVNode::WriteToFile(const AString& path) const
{
    AString code;
    Export(code);
    return NUtilities::WriteStringToFile(path, code);
}
//---------------------------------------------------------------------------
bool CKVNode::LoadFromFile(const AString& path, bool discardExisting)
{
    AString code;
    if (!NUtilities::ReadStringFromFile(path, code)) return false;
    return Import(code, discardExisting);
}
//---------------------------------------------------------------------------
void CKVNode::ExportNode(AStringBuilder& target, const CKVNode* node, unsigned depth)
{
    DEBUG_ASSERT(node, "null node was given");
    // Export the node's name
    if (!node->GetName().IsEmpty() || !node->GetValue().IsEmpty())
    {
        target += NString::TRepeat(' ', depth);
        if (node->GetValue().IsEmpty())
        {
            EncodeString(node->GetName(), target);
            // In case the value is empty and there are no chilren, add an
            // empty string to avoid the ambiguity of having an unnamed node
            // with children right after a named node without a value
            if (!node->HasChildren()) target += "=\"\"";
            target += "\r\n";
        }
        else
        {
            EncodeString(node->GetName(), target);
            target += '=';
            EncodeString(node->GetValue(), target);
            target += "\r\n";
        }
    }
    // Export the node's values
    if (node->HasChildren())
    {
        target += NString::TRepeat(' ', depth) + "{\r\n";
        const AKVNodeArray& children = node->GetChildren();
        for (unsigned i=0; i < children.GetSize(); ++i)
        {
            ExportNode(target, children[i], depth + 2);
        }
        target += NString::TRepeat(' ', depth) + "}\r\n";
    }
}
//---------------------------------------------------------------------------
bool CKVNode::ImportNode(const AString& source, CKVNode* node, unsigned& head)
{
    // Skip spaces
    while (head < source.GetLength() && isspace(source[head])) ++head;
    // Scan the node's name
    node->m_name = DecodeString(source, head);
    // Skip spaces
    while (head < source.GetLength() && isspace(source[head])) ++head;
    // Scan the node's value
    if (head < source.GetLength() && source[head] == '=')
    {
        ++head;
        // Skip spaces
        while (head < source.GetLength() && isspace(source[head])) ++head;
        // Scan the value
        node->m_value = DecodeString(source, head);
    } else node->m_value.Clear();
    // Skip spaces
    while (head < source.GetLength() && isspace(source[head])) ++head;
    // Check for subnodes
    if (head < source.GetLength() && source[head] == '{')
    {
        ++head;
        while (head < source.GetLength())
        {
            unsigned saveHead = head;
            // Skip spaces
            while (head < source.GetLength() && isspace(source[head])) ++head;
            // End of subnodes
            if (head < source.GetLength() && source[head] == '}')
            {
                ++head;
                break;
            }
            // Create subnode
            CKVNode* child = node->CreateChild();
            if (!ImportNode(source, child, head)) return false;
            // If we didn't advance the head, something was wrong
            if (head == saveHead) return false;
        }
    }
    return true;
}
//---------------------------------------------------------------------------
void CKVNode::EncodeString(const AString& str, AStringBuilder& target)
{
    // Check for special characters
    bool needQuotes = false;
    bool specialFound = false;
    for (unsigned i=0; i < str.GetLength() && !specialFound; ++i)
    {
        if (str[i] == '.' || str[i] == '"' || str[i] == '=' || str[i] < 32)
            specialFound = true;
        if (!((str[i] >= 'a' && str[i] <= 'z') ||
              (str[i] >= 'A' && str[i] <= 'Z') ||
              (str[i] >= '0' && str[i] <= '9') ||
               str[i] == '_')) needQuotes = true;
    }
    // Fast case - no specials, use the string as-is
    if (!specialFound)
    {
        if (needQuotes) target += '"';
        target += str;
        if (needQuotes) target += '"';
        return;
    }
    // Slow case - encode the string
    if (needQuotes) target += '"';
    for (unsigned i=0; i < str.GetLength(); ++i)
    {
        if (str[i] == '.' || str[i] == '"' || str[i] == '=' || str[i] == '\\')
        {
            target += '\\';
            target += str[i];
        }
        else if (str[i] < 32)
        {
            target += "\\x";
            target += (char)GetHexNibble((str[i] >> 4)&0x0F);
            target += (char)GetHexNibble(str[i] & 0x0F);
        }
        else target += str[i];
    }
    if (needQuotes) target += '"';
}
//---------------------------------------------------------------------------
AString CKVNode::DecodeString(const AString& str, size_t& head)
{
    // Slow case - decode the string
    AString result;
    if (head < str.GetLength() && str[head] == '"')
    {
        ++head;
        while (head < str.GetLength())
        {
            // Closing quote
            if (str[head] == '"')
            {
                ++head;
                return result;
            }
            // Escape
            if (str[head] == '\\')
            {
                ++head;
                if (head == str.GetLength()) return result + "\\";
                if (str[head] == 'x' && head + 2 < str.GetLength())
                {
                    int nibble1 = ParseHexNibble(str[++head]);
                    int nibble2 = ParseHexNibble(str[++head]);
                    result += (char)((nibble1 << 4)|nibble2);
                    ++head;
                }
                else result += str[head++];
            }
            else result += str[head++];
        }
    }
    else // no quotes
    {
        while (head < str.GetLength() && (isalnum(str[head]) || str[head] == '_'))
        {
            result += str[head++];
        }
    }
    return result;
}
//---------------------------------------------------------------------------


