//---------------------------------------------------------------------------
#ifndef DynArrayH
#define DynArrayH
//---------------------------------------------------------------------------
#include "DebugAssert.h"
//---------------------------------------------------------------------------
// Dynamic array template
template <typename T> class TDynArray
{
public:
    inline TDynArray() : m_items(0), m_size(0), m_capacity(0) {}

    // Copy constructor
    inline TDynArray(const TDynArray<T>& src)
    {
        m_capacity = m_size = src.m_size;
        m_items = new T[m_size];
        for (unsigned i=0; i < m_size; ++i) m_items[i] = src.m_items[i];
    }

    // Destructor
    inline ~TDynArray()
    {
        delete[] m_items;
    }

    // Clear the array
    inline void Clear()
    {
        delete[] m_items;
        m_items = 0;
        m_capacity = m_size = 0;
    }

    // Reset the array by removing all items but not reseting the capacity
    inline void Reset()
    {
        m_size = 0;
    }

    // Set the array's capacity
    inline void SetCapacity(unsigned cap)
    {
        DEBUG_ASSERT(cap >= m_size, "cannot shrink the capacity to less than the size");
        if (cap != m_capacity)
        {
            m_capacity = cap;
            T* newItems = new T[cap];
            for (unsigned i=0; i < m_size; ++i) newItems[i] = m_items[i];
            delete[] m_items;
            m_items = newItems;
        }
    }

    // Reserve capacity for additional items
    inline void Reserve(unsigned count)
    {
        if (m_capacity < m_size + count)
        {
            SetCapacity(m_size + count);
        }
    }

    // Set the capacity to match the array's size
    inline void FitCapacity()
    {
        SetCapacity(m_size);
    }

    // Set the array's size
    inline void SetSize(unsigned size)
    {
        // Grow the capacity if necessary
        if (size >= m_capacity) SetCapacity(size);
        m_size = size;
    }

    // Add a new item
    inline T& Add(const T& item)
    {
        // Grow the capacity if necessary
        if (m_size >= m_capacity) Reserve(64);
        m_items[m_size++] = item;
        return Last();
    }

    // Add the items from the given array
    inline void Add(const TDynArray<T>& src)
    {
        // Grow the capacity if necessary
        if (m_size + src.m_size >= m_capacity) Reserve(src.m_size + 64);
        for (unsigned i=0; i < src.m_size; ++i) Add(src[i]);
    }

    // Add the item if it isn't already part of the array
    inline void AddUnique(const T& item)
    {
        if (!Contains(item)) Add(item);
    }

    // Add the items from the given array that are not part of this array
    inline void AddUnique(const TDynArray<T>& src)
    {
        for (unsigned i=0; i < src.m_size; ++i) AddUnique(src.m_items[i]);
    }

    // Add a one or more default items and return a reference to it
    inline T& Grow(unsigned count=1)
    {
        DEBUG_ASSERT(count > 0, "you must grow the array by at least one item");
        for (; count > 0; --count) Add(T());
        return m_items[m_size - 1];
    }

    // Remove item at the given index
    inline void RemoveAt(unsigned index)
    {
        DEBUG_ASSERT(index < m_size, "index is outside the dynamic array range");
        for (unsigned i=index; i < m_size - 1; ++i)
            m_items[i] = m_items[i + 1];
        --m_size;
    }

    // Remove given item, returns false if the item doesn't exist
    inline bool Remove(const T& item)
    {
        unsigned index;
        if (Find(item, index))
        {
            RemoveAt(index);
            return true;
        }
        return false;
    }

    // Remove item at the given index by swapping it with the last item and shrinking
    inline void RemoveViaSwapAt(unsigned index)
    {
        DEBUG_ASSERT(index < m_size, "index is outside the dynamic array range");
        if (index != m_size - 1) Swap(index, m_size - 1);
        --m_size;
    }

    // Remove given item by swapping it with the last item and shrinking
    inline bool RemoveViaSwap(const T& item)
    {
        unsigned index;
        if (Find(item, index))
        {
            if (index != m_size - 1) Swap(index, m_size - 1);
            --m_size;
            return true;
        }
        return false;
    }

    // Find the index of the given item, returns false if the item doesn't exist
    inline bool Find(const T& item, unsigned& index) const
    {
        for (unsigned i=0; i < m_size; ++i)
        {
            if (m_items[i] == item)
            {
                index = i;
                return true;
            }
        }
        index = (unsigned)-1;
        return false;
    }

    // Return true if the array contains the given item
    inline bool Contains(const T& item) const
    {
        for (unsigned i=0; i < m_size; ++i)
        {
            if (m_items[i] == item)
            {
                return true;
            }
        }
        return false;
    }

    // Swap the items at the given indices
    inline void Swap(unsigned index1, unsigned index2)
    {
        DEBUG_ASSERT(index1 < m_size, "first index is outside the dynamic array range");
        DEBUG_ASSERT(index2 < m_size, "second index is outside the dynamic array range");
        T tmp = m_items[index1];
        m_items[index1] = m_items[index2];
        m_items[index2] = tmp;
    }

    // Sort the array using the given comparison function
    inline void Sort(int (*cmp)(const T& a, const T& b))
    {
        if (m_size > 1) Quicksort(0, m_size - 1, cmp); 
    }

    // Exchange the contents of the given array with the contents of this array
    inline void Exchange(TDynArray<T>& other)
    {
        T* myItems = m_items;
        unsigned mySize = m_size;
        unsigned myCapacity = m_capacity;
        m_items = other.m_items;
        m_size = other.m_size;
        m_capacity = other.m_capacity;
        other.m_items = myItems;
        other.m_size = mySize;
        other.m_capacity = myCapacity;
    }

    // Return the last added item
    inline const T& Last() const
    {
        DEBUG_ASSERT(m_size > 0, "the array is empty");
        return m_items[m_size - 1];
    }
    inline T& Last()
    {
        DEBUG_ASSERT(m_size > 0, "the array is empty");
        return m_items[m_size - 1];
    }

    // Return the first item
    inline const T& First() const
    {
        DEBUG_ASSERT(m_size > 0, "the array is empty");
        return m_items[0];
    }
    inline T& First()
    {
        DEBUG_ASSERT(m_size > 0, "the array is empty");
        return m_items[0];
    }

    // Return the last item and remove it from the array ("pop" it)
    inline T Pop()
    {
        DEBUG_ASSERT(m_size > 0, "the array is empty");
        T item = Last();
        RemoveAt(m_size - 1);
        return item;
    }

    // Return true if the array is empty
    inline bool IsEmpty() const { return m_size == 0; }

    // Return true if the array is not empty
    inline bool HasItems() const { return m_size > 0; }

    // Index access
    inline const T& operator[](unsigned i) const
    {
        DEBUG_ASSERT(i < m_size, "invalid array index");
        return m_items[i];
    }
    inline T& operator[](unsigned i)
    {
        DEBUG_ASSERT(i < m_size, "invalid array index");
        return m_items[i];
    }

    // Assignment operator
    inline TDynArray<T>& operator=(const TDynArray<T>& src)
    {
        if (&src != this)
        {
            delete[] m_items;
            m_capacity = m_size = src.m_size;
            m_items = new T[m_size];
            for (unsigned i=0; i < m_size; ++i) m_items[i] = src.m_items[i];
        }
        return *this;
    }

    // Equality operator
    inline bool operator==(const TDynArray<T>& rhs) const
    {
        if (m_size == rhs.m_size)
        {
            for (unsigned i=0; i < m_size; ++i)
            {
                if (!(m_items[i] == rhs.m_items[i]))
                {
                    return false;
                }
            }
            return true;
        }
        else
        {
            return false;
        }
    }
    inline bool operator!=(const TDynArray<T>& rhs) const
    {
        return !(operator==(rhs));
    }

    // Return the item count in this array
    inline unsigned GetSize() const { return m_size; }

    // Return the item count as a signed integer
    inline int GetSignedSize() const { return (int)m_size; }

    // Return the currentcapacity of this array
    inline unsigned GetCapacity() const { return m_capacity; }

    // Return the items as a typed pointer
    inline const T* GetData() const { return m_items; }
    inline T* GetData() { return m_items; }

private:
    T*          m_items;
    unsigned    m_size;
    unsigned    m_capacity;

    // Quicksort
    void Quicksort(int lo, int hi, int (*cmp)(const T& a, const T& b))
    {
        if (lo < hi)
        {
            int p = QSPartition(lo, hi, cmp);
            Quicksort(lo, p - 1, cmp);
            Quicksort(p + 1, hi, cmp);
        }
    }
    // Quicksort partition
    int QSPartition(int lo, int hi, int (*cmp)(const T& a, const T& b))
    {
        const T& pivot = m_items[hi];
        int i = lo - 1;
        for (int j=lo; j < hi; ++j)
            if (cmp(m_items[j], pivot) < 0) Swap(++i, j);
        Swap((unsigned)(i + 1), (unsigned)hi);
        return i + 1;   
    }
};
//---------------------------------------------------------------------------
// Helper array functions
namespace NDynArray
{
    // TDynArray::Sort comparison function that uses the < and == operators
    // to sort in an ascending order
    template <typename T> int TAscending(const T& a, const T& b)
    {
        return a < b ? -1 : (a == b ? 0 : 1);
    }
    // TDynArray::Sort comparison function that uses the < and == operators
    // to sort in an descending order
    template <typename T> int TDescending(const T& a, const T& b)
    {
        return a < b ? 1 : (a == b ? 0 : -1);
    }
}
//---------------------------------------------------------------------------
#endif
