//---------------------------------------------------------------------------
#include "Documents.h"
//---------------------------------------------------------------------------
// CDocView implementation
CDocView::CDocView(CDocument* document)
    : m_document(document)
{
}
//---------------------------------------------------------------------------
CDocView::~CDocView()
{
    // If we're still registered with the document inform it about
    // our destruction
    if (m_document)
    {
        m_document->OnDocViewDestroyed(this);
    }
}
//---------------------------------------------------------------------------
bool CDocView::IsLastView() const
{
    DEBUG_ASSERT(m_document, "No document associated with the view");
    return m_document->GetViews().GetSize() == 1 &&
           m_document->GetViews()[0] == this;
}
//---------------------------------------------------------------------------
void CDocView::OnModified(bool)
{
    // Does nothing, subclasses should probably do something like adding
    // an asterisk in a window's titlebar or something...
}
//---------------------------------------------------------------------------
void CDocView::OnInvalidated(const void*)
{
    // Just refresh the view, subclasses can override this method to do
    // something more advanced using the "details" parameter
    Refresh();
}
//---------------------------------------------------------------------------
// CDocument implementation
CDocument::CDocument()
    : m_manager(NULL)
    , m_modified(false)
{
}
//---------------------------------------------------------------------------
CDocument::~CDocument()
{
    // Destroy any still registered views
    DestroyAllViews();
    // If the document was destroyed while still being registered,
    // unregister it from the manager
    if (m_manager)
    {
        m_manager->Unregister(this);
    }
}
//---------------------------------------------------------------------------
bool CDocument::CanModify() const
{
    // We shouldn't allow modifications without a manager being present
    // to avoid any UI elements that listen for modifications to miss the
    // flag change
    return m_manager != NULL;
}
//---------------------------------------------------------------------------
bool CDocument::MarkModified()
{
    if (m_modified) return true;
    if (!CanModify()) return false;
    m_modified = true;
    OnModified(true);
    for (unsigned i=0; i < m_views.GetSize(); ++i)
    {
        m_views[i]->OnModified(true);
    }
    if (m_manager) m_manager->DocumentModified(this, m_modified);
    return true;
}
//---------------------------------------------------------------------------
void CDocument::ClearModified()
{
    if (!m_modified) return;
    m_modified = false;
    OnModified(false);
    for (unsigned i=0; i < m_views.GetSize(); ++i)
    {
        m_views[i]->OnModified(false);
    }
    if (m_manager) m_manager->DocumentModified(this, m_modified);    
}
//---------------------------------------------------------------------------
CDocView* CDocument::NewView(int type)
{
    CDocView* view = CreateView(type);
    if (view)
    {
        m_views.Add(view);
        OnViewAdded(view);
        if (m_manager)
        {
            m_manager->ViewCreated(this, view);
        }
    }
    return view;
}
//---------------------------------------------------------------------------
void CDocument::DestroyAllViews()
{
    for (int i=m_views.GetSignedSize() - 1; i >= 0; --i)
    {
        m_views[i]->m_document = NULL;
        OnViewRemoved(m_views[i]);
        if (m_manager)
        {
            m_manager->ViewDestroyed(this, m_views[i]);
        }
        delete m_views[i];
    }
    m_views.Clear();
}
//---------------------------------------------------------------------------
void CDocument::RefreshViews()
{
    for (unsigned i=0; i < m_views.GetSize(); ++i)
    {
        m_views[i]->Refresh();
    }
}
//---------------------------------------------------------------------------
void CDocument::InvalidateViews(const void* detail)
{
    for (unsigned i=0; i < m_views.GetSize(); ++i)
    {
        m_views[i]->OnInvalidated(detail);
    }
}
//---------------------------------------------------------------------------
void CDocument::OnDocViewDestroyed(CDocView* view)
{
    m_views.Remove(view);
    OnViewRemoved(view);
    if (m_manager)
    {
        m_manager->ViewDestroyed(this, view);
    }
}
//---------------------------------------------------------------------------
// CDocumentManager implementation
void CDocumentManager::Register(CDocument* doc)
{
    DEBUG_ASSERT(doc, "Missing document");
    DEBUG_ASSERT(!m_docs.Contains(doc), "The document already has been registered");
    DEBUG_ASSERT(!doc->m_manager, "The document already has a manager");
    m_docs.Add(doc);
    doc->m_manager = this;
    for (unsigned i=0; i < m_listeners.GetSize(); ++i)
    {
        m_listeners[i]->OnDocumentRegistered(doc);
    }
}
//---------------------------------------------------------------------------
void CDocumentManager::Unregister(CDocument* doc)
{
    DEBUG_ASSERT(doc, "Missing document");
    DEBUG_ASSERT(m_docs.Contains(doc), "The document is not registered");
    DEBUG_ASSERT(doc->m_manager, "The document does not have a manager");
    DEBUG_ASSERT(doc->m_manager == this, "The document's manager is not this manager");
    m_docs.Remove(doc);
    doc->m_manager = NULL;
    for (unsigned i=0; i < m_listeners.GetSize(); ++i)
    {
        m_listeners[i]->OnDocumentUnregistered(doc);
    }
}
//---------------------------------------------------------------------------
void CDocumentManager::AddListener(IDocumentListener* listener)
{
    DEBUG_ASSERT(listener, "Missing listener");
    DEBUG_ASSERT(!m_listeners.Contains(listener), "The listener is already in the manager");
    m_listeners.Add(listener);
}
//---------------------------------------------------------------------------
void CDocumentManager::RemoveListener(IDocumentListener* listener)
{
    DEBUG_ASSERT(listener, "Missing listener");
    DEBUG_ASSERT(m_listeners.Contains(listener), "The listener is not in the manager");
    m_listeners.Remove(listener);
}
//---------------------------------------------------------------------------
void CDocumentManager::DocumentModified(CDocument* doc, bool modified)
{
    for (unsigned i=0; i < m_listeners.GetSize(); ++i)
    {
        m_listeners[i]->OnDocumentModified(doc, modified);
    }
}
//---------------------------------------------------------------------------
void CDocumentManager::ViewCreated(CDocument* doc, CDocView* view)
{
    for (unsigned i=0; i < m_listeners.GetSize(); ++i)
    {
        m_listeners[i]->OnViewCreated(doc, view);
    }
}
//---------------------------------------------------------------------------
void CDocumentManager::ViewDestroyed(CDocument* doc, CDocView* view)
{
    // Notify the listeners
    for (unsigned i=0; i < m_listeners.GetSize(); ++i)
    {
        m_listeners[i]->OnViewDestroyed(doc, view);
    }
    // Destroy the document if it has no views anymore
    if (doc->GetViews().IsEmpty())
    {
        delete doc;
    }
}
//---------------------------------------------------------------------------

