//---------------------------------------------------------------------------
#include <string.h>
#include "Colors.h"
//---------------------------------------------------------------------------
// Clamps the given integer to 0..255 for color use
static unsigned char ColorClamp(int v)
{
    if (v < 0) return 0;
    if (v > 255) return 255;
    return static_cast<unsigned char>(v);
}
//---------------------------------------------------------------------------
SColor NColor::Mix(const SColor& target, const SColor& source, unsigned alpha)
{
    alpha = (alpha*source.Alpha)/255;
    return SColor(static_cast<unsigned char>((source.Red*alpha)/255 + (target.Red*(255 - alpha)/255)),
                  static_cast<unsigned char>((source.Green*alpha)/255 + (target.Green*(255 - alpha)/255)),
                  static_cast<unsigned char>((source.Blue*alpha)/255 + (target.Blue*(255 - alpha)/255)),
                  static_cast<unsigned char>(source.Alpha + (target.Alpha*(255 - target.Alpha))/255));
}
//---------------------------------------------------------------------------
// NBlender functions
FBlender NBlender::ByMode(EBlendMode mode)
{
    switch (mode)
    {
    case BM_REPLACE:    return Replace;
    case BM_MIX:        return Mix;
    case BM_ADD:        return Add;
    case BM_SUBTRACT:   return Subtract;
    case BM_MULTIPLY:   return Multiply;
    case BM_COLORIZE:   return Colorize;
    case BM_HUE:        return Hue;
    case BM_SATURATION: return Saturation;
    case BM_VALUE:      return Value;
    default:
        DEBUG_FAIL("Invalid blending mode");
        return NULL;
    }
}
//---------------------------------------------------------------------------
void NBlender::Replace(SColor* target, const SColor* source, size_t count, unsigned)
{
    memcpy(target, source, count*sizeof(SColor));
}
//---------------------------------------------------------------------------
void NBlender::Mix(SColor* target, const SColor* source, size_t count, unsigned opacity)
{
    for (size_t i=count; i > 0; --i,++target,++source)
    {
        *target = NColor::Mix(*target, *source, opacity);
    }
}
//---------------------------------------------------------------------------
void NBlender::Add(SColor* target, const SColor* source, size_t count, unsigned opacity)
{
    SColor tmp;
    for (size_t i=count; i > 0; --i,++target,++source)
    {
        tmp.Red = ColorClamp(source->Red*opacity/255 + target->Red);
        tmp.Green = ColorClamp(source->Red*opacity/255 + target->Green);
        tmp.Blue = ColorClamp(source->Red*opacity/255 + target->Blue);
        tmp.Alpha = target->Alpha;
        *target = NColor::Mix(*target, tmp, opacity*source->Alpha/255);
    }
}
//---------------------------------------------------------------------------
void NBlender::Subtract(SColor* target, const SColor* source, size_t count, unsigned opacity)
{
    SColor tmp;
    for (size_t i=count; i > 0; --i,++target,++source)
    {
        tmp.Red = ColorClamp(target->Red - source->Red*opacity/255);
        tmp.Green = ColorClamp(target->Green - source->Green*opacity/255);
        tmp.Blue = ColorClamp(target->Blue - source->Blue*opacity/255);
        tmp.Alpha = target->Alpha;
        *target = NColor::Mix(*target, tmp, opacity*source->Alpha/255);
    }
}
//---------------------------------------------------------------------------
void NBlender::Multiply(SColor* target, const SColor* source, size_t count, unsigned opacity)
{
    SColor tmp;
    for (size_t i=count; i > 0; --i,++target,++source)
    {
        tmp.Red = ColorClamp(target->Red * source->Red*opacity/255/255);
        tmp.Green = ColorClamp(target->Green * source->Green*opacity/255/255);
        tmp.Blue = ColorClamp(target->Blue * source->Blue*opacity/255/255);
        tmp.Alpha = target->Alpha;
        *target = NColor::Mix(*target, tmp, opacity*source->Alpha/255);
    }
}
//---------------------------------------------------------------------------
void NBlender::Colorize(SColor* target, const SColor* source, size_t count, unsigned opacity)
{
    float sh, ss, sv, th, ts, tv;
    SColor tmp;
    for (size_t i=count; i > 0; --i,++target,++source)
    {
        NColor::ToHSV(*source, sh, ss, sv);
        NColor::ToHSV(*target, th, ts, tv);
        tmp = NColor::FromHSV(sh, ss, tv);
        tmp.Alpha = target->Alpha;
        *target = NColor::Mix(*target, tmp, opacity*source->Alpha/255);
    }
}
//---------------------------------------------------------------------------
void NBlender::Hue(SColor* target, const SColor* source, size_t count, unsigned opacity)
{
    float sh, ss, sv, th, ts, tv;
    SColor tmp;
    for (size_t i=count; i > 0; --i,++target,++source)
    {
        NColor::ToHSV(*source, sh, ss, sv);
        NColor::ToHSV(*target, th, ts, tv);
        tmp = NColor::FromHSV(sh, ts, tv);
        tmp.Alpha = target->Alpha;
        *target = NColor::Mix(*target, tmp, opacity*source->Alpha/255);
    }
}
//---------------------------------------------------------------------------
void NBlender::Saturation(SColor* target, const SColor* source, size_t count, unsigned opacity)
{
    float sh, ss, sv, th, ts, tv;
    SColor tmp;
    for (size_t i=count; i > 0; --i,++target,++source)
    {
        NColor::ToHSV(*source, sh, ss, sv);
        NColor::ToHSV(*target, th, ts, tv);
        tmp = NColor::FromHSV(th, ss, tv);
        tmp.Alpha = target->Alpha;
        *target = NColor::Mix(*target, tmp, opacity*source->Alpha/255);
    }
}
//---------------------------------------------------------------------------
void NBlender::Value(SColor* target, const SColor* source, size_t count, unsigned opacity)
{
    float sh, ss, sv, th, ts, tv;
    SColor tmp;
    for (size_t i=count; i > 0; --i,++target,++source)
    {
        NColor::ToHSV(*source, sh, ss, sv);
        NColor::ToHSV(*target, th, ts, tv);
        tmp = NColor::FromHSV(th, ts, sv);
        tmp.Alpha = target->Alpha;
        *target = NColor::Mix(*target, tmp, opacity*source->Alpha/255);
    }
}
//---------------------------------------------------------------------------

