//---------------------------------------------------------------------------
#include "Cameras.h"
//---------------------------------------------------------------------------
CCamera::CCamera()
    : m_position(0, 0, 0)
    , m_direction(0, 0, -1)
    , m_aspect(1)
    , m_zNear(0.1)
    , m_zFar(100)
    , m_FOV(55)
    , m_orthoScale(1)
    , m_mode(CM_PERSPECTIVE)
{
    RecalcDeriveds();
}
//---------------------------------------------------------------------------
void CCamera::MoveTowards(const AVector& dir)
{
    m_position += dir;
    RecalcDeriveds();
}
//---------------------------------------------------------------------------
void CCamera::MoveTowardsInView(const AVector& dir)
{
    MoveTowards(m_view.GetInverted().TransformNormal3D(dir));
}
//---------------------------------------------------------------------------
void CCamera::MoveForward(float speed)
{
    MoveTowards(m_direction*speed);
}
//---------------------------------------------------------------------------
void CCamera::RotateHorizontally(float angle)
{
    m_direction = NMatrix::GetRotation(angle, AVector(0, 1, 0)).Transform3D(m_direction);
    RecalcDeriveds();
}
//---------------------------------------------------------------------------
void CCamera::RotateVertically(float angle)
{
    m_direction = NMatrix::GetRotation(angle,
        AVector(m_view.M[0][0], m_view.M[1][0], m_view[2][0])).Transform3D(m_direction);
    RecalcDeriveds();
}
//---------------------------------------------------------------------------
void CCamera::RotateAroundPoint(const AVector& point, float hangle, float vangle)
{
    AVector pointToCamera(m_position - point);
    NMatrix::GetRotation(vangle, AVector(1, 0, 0)).Transform3D(pointToCamera);
    NMatrix::GetRotation(hangle, AVector(0, 1, 0)).Transform3D(pointToCamera);
    m_position = point + pointToCamera;
    RecalcDeriveds(); 
}
//---------------------------------------------------------------------------
void CCamera::SetPosition(const AVector& pos)
{
    m_position = pos;
    RecalcDeriveds();
}
//---------------------------------------------------------------------------
void CCamera::SetDirection(const AVector& dir)
{
    m_direction = dir;
    RecalcDeriveds();
}
//---------------------------------------------------------------------------
void CCamera::SetPositionAndDirection(const AVector& pos, const AVector& dir)
{
    m_position = pos;
    m_direction = dir;
    RecalcDeriveds();
}
//---------------------------------------------------------------------------
void CCamera::SetAspect(float aspect)
{
    DEBUG_ASSERT(aspect > 0.0f, "Invalid aspect ratio");
    m_aspect = aspect;
    RecalcDeriveds();
}
//---------------------------------------------------------------------------
void CCamera::SetClipPlanes(float znear, float zfar)
{
    DEBUG_ASSERT(znear > 0, "Invalid near clip plane");
    DEBUG_ASSERT(zfar > 0, "Invalid far clip plane");
    DEBUG_ASSERT(znear < zfar, "Near clip plane is further than far clip plane");
    m_zNear = znear;
    m_zFar = zfar;
    RecalcDeriveds();
}
//---------------------------------------------------------------------------
void CCamera::SetFOV(float fov)
{
    DEBUG_ASSERT(fov > 0 && fov < 180, "Invalid vertical field of view");
    m_FOV = fov;
    RecalcDeriveds();
}
//---------------------------------------------------------------------------
void CCamera::SetOrthoScale(float orthoScale)
{
    DEBUG_ASSERT(orthoScale >= 0.0f, "Invalid orthographic scale");
    m_orthoScale = orthoScale;
    RecalcDeriveds();
}
//---------------------------------------------------------------------------
void CCamera::SetMode(ECameraMode mode)
{
    m_mode = mode;
    RecalcDeriveds();
}
//---------------------------------------------------------------------------
void CCamera::Setup(ECameraMode mode, float fov, float aspect, float znear, float zfar, float orthoScale)
{
    DEBUG_ASSERT(aspect > 0.0f, "Invalid aspect ratio");
    DEBUG_ASSERT(znear > 0, "Invalid near clip plane");
    DEBUG_ASSERT(zfar > 0, "Invalid far clip plane");
    DEBUG_ASSERT(znear < zfar, "Near clip plane is further than far clip plane");
    DEBUG_ASSERT(fov > 0 && fov < 180, "Invalid vertical field of view");
    DEBUG_ASSERT(orthoScale >= 0.0f, "Invalid orthographic scale");
    m_mode = mode;
    m_aspect = aspect;
    m_zNear = znear;
    m_zFar = zfar;
    m_FOV = fov;
    m_orthoScale = orthoScale;
    RecalcDeriveds();
}
//---------------------------------------------------------------------------
AVector CCamera::WorldToScreen(const AVector& v, int width, int height) const
{
    DEBUG_ASSERT(width > 0 && height > 0, "Invalid screen dimensions");
    AVector r(v);
    r.W = 1.0f;
    r = NMatrix::Transform(m_view, m_projection).Transform4D(r);
    r.X = (r.X/r.W + 1.0f)*0.5f*(float)width;
    r.Y = (1.0f - r.Y/r.W)*0.5f*(float)height;
    r.Z = (r.Z/r.W + 1.0f)*0.5f;
    r.W = 1.0f;
    return r;
}
//---------------------------------------------------------------------------
AVector CCamera::ScreenToWorld(const AVector& v, int width, int height) const
{
    DEBUG_ASSERT(width > 0 && height > 0, "Invalid screen dimensions");
    AVector r(v.X/(float)width*2.0f - 1.0f,
              ((float)height - v.Y)/(float)height*2.0f - 1.0f,
              v.Z/2.0f-1.0f,
              1.0f);
    r = m_unprojection.Transform4D(r);
    r.X /= r.W;
    r.Y /= r.W;
    r.Z /= r.W;
    r.W = 1.0f;
    return r;
}
//---------------------------------------------------------------------------
ARay CCamera::RayAtScreenCoords(int x, int y, int width, int height)
{
    DEBUG_ASSERT(width > 0 && height > 0, "Invalid screen dimensions");
    return NRay::TFromSegment3D(ScreenToWorld(AVector(x, y, 0.1), width, height),
                                ScreenToWorld(AVector(x, y, 0.9), width, height));
}
//---------------------------------------------------------------------------
void CCamera::RecalcDeriveds()
{
    // Projection
    switch (m_mode) {
    case CM_PERSPECTIVE:
        m_projection = NMatrix::GetPerspective(m_FOV*PI/180.0, m_aspect, m_zNear, m_zFar);
        break;
    case CM_ORTHOGRAPHIC:
        m_projection = NMatrix::GetOrthographic(-m_orthoScale*0.5f*m_aspect, m_orthoScale*0.5f*m_aspect, m_orthoScale*0.5f, -m_orthoScale*0.5f, -m_zFar, m_zFar);
        break;
    default:
        DEBUG_FAIL("Invalid camera mode");
    }
    // View and direction vectors
    AVector target(m_position + m_direction.GetNormalized3D());
    AVector up(fabs(m_direction.Y) > 0.999f ? AVector(0, 0, 1) : AVector(0, 1, 0));
    m_view = NMatrix::GetLookAt(m_position, target, up);
    m_right = NVector::TCross3D(m_direction, up).GetNormalized3D();
    m_up = NVector::TCross3D(m_right, m_direction).GetNormalized3D();
    // Unprojection
    m_unprojection = NMatrix::Transform(m_view, m_projection).GetInverted();
}
//---------------------------------------------------------------------------
