//---------------------------------------------------------------------------
#ifndef BoxH
#define BoxH
//---------------------------------------------------------------------------
#include "Vector.h"
//---------------------------------------------------------------------------
// Axis aligned 3D box
template <typename T> struct TBox
{
    TVector<T>  Min;    // Minimum values
    TVector<T>  Max;    // Maximum values

    // Undefined box
    inline TBox() {}

    // Box with the given extents
    inline TBox(const TVector<T>& min, const TVector<T>& max)
        : Min(min)
        , Max(max)
    {
        DEBUG_ASSERT(max.X >= min.X, "box min/max X coordinate is inverted");
        DEBUG_ASSERT(max.Y >= min.Y, "box min/max Y coordinate is inverted");
        DEBUG_ASSERT(max.Z >= min.Z, "box min/max Z coordinate is inverted");
    }

    // Returns the box's size
    inline TVector<T> GetSize() const
    {
        DEBUG_ASSERT(Max.X >= Min.X, "box min/max X coordinate is inverted");
        DEBUG_ASSERT(Max.Y >= Min.Y, "box min/max Y coordinate is inverted");
        DEBUG_ASSERT(Max.Z >= Min.Z, "box min/max Z coordinate is inverted");
        return Max - Min;
    }

    // Returns the box's width (X size)
    inline T GetWidth() const
    {
        DEBUG_ASSERT(Max.X >= Min.X, "box min/max X coordinate is inverted");
        return Max.X - Min.X;
    }

    // Returns the box's depth (Y size)
    inline T GetDepth() const
    {
        DEBUG_ASSERT(Max.Y >= Min.Y, "box min/max Y coordinate is inverted");
        return Max.Y - Min.Y;
    }

    // Returns the box's height (Z size)
    inline T GetHeight() const
    {
        DEBUG_ASSERT(Max.Z >= Min.Z, "box min/max Z coordinate is inverted");
        return Max.Z - Min.Z;
    }

    // Returns true if the point is inside the box
    inline bool Inside(const TVector<T>& point) const
    {
        DEBUG_ASSERT(Max.X >= Min.X, "box min/max X coordinate is inverted");
        DEBUG_ASSERT(Max.Y >= Min.Y, "box min/max Y coordinate is inverted");
        DEBUG_ASSERT(Max.Z >= Min.Z, "box min/max Z coordinate is inverted");
        return point.X >= Min.X && point.Y >= Min.Y && point.Z >= Min.Z &&
               point.X <= Max.X && point.Y <= Max.Y && point.Z <= Max.Z;
    }

    // Extend the box to cover the point
    inline void Include(const TVector<T>& point)
    {
        DEBUG_ASSERT(Max.X >= Min.X, "box min/max X coordinate is inverted");
        DEBUG_ASSERT(Max.Y >= Min.Y, "box min/max Y coordinate is inverted");
        DEBUG_ASSERT(Max.Z >= Min.Z, "box min/max Z coordinate is inverted");
        Min.X = TMin<T>(Min.X, point.X);
        Min.Y = TMin<T>(Min.Y, point.Y);
        Min.Z = TMin<T>(Min.Z, point.Z);
        Max.X = TMax<T>(Max.X, point.X);
        Max.Y = TMax<T>(Max.Y, point.Y);
        Max.Z = TMax<T>(Max.Z, point.Z);
    }

    // Extend the box to cover the given box
    inline void Include(const TBox<T>& box)
    {
        DEBUG_ASSERT(Max.X >= Min.X, "this box min/max X coordinate is inverted");
        DEBUG_ASSERT(Max.Y >= Min.Y, "this box min/max Y coordinate is inverted");
        DEBUG_ASSERT(Max.Z >= Min.Z, "this box min/max Z coordinate is inverted");
        DEBUG_ASSERT(box.Max.X >= box.Min.X, "given box min/max X coordinate is inverted");
        DEBUG_ASSERT(box.Max.Y >= box.Min.Y, "given box min/max Y coordinate is inverted");
        DEBUG_ASSERT(box.Max.Z >= box.Min.Z, "given box min/max Z coordinate is inverted");
        Min.X = TMin(Min.X, box.Min.X);
        Min.Y = TMin(Min.Y, box.Min.Y);
        Min.Z = TMin(Min.Z, box.Min.Z);
        Max.X = TMax(Max.X, box.Max.X);
        Max.Y = TMax(Max.Y, box.Max.Y);
        Max.Z = TMax(Max.Z, box.Max.Z);
    }

    // Returns the box's center
    inline TVector<T> GetCenter2D() const
    {
        return TVector<T>((Max.X + Min.X)*0.5f, (Max.Y + Min.Y)*0.5f);
    }
    inline TVector<T> GetCenter3D() const
    {
        return TVector<T>((Max.X + Min.X)*0.5f, (Max.Y + Min.Y)*0.5f, (Max.Z + Min.Z)*0.5f);
    }

    // Comparison
    inline bool operator==(const TBox<T>& rhs) const
    {
        return Min==rhs.Min && Max==rhs.Max;
    }
    inline bool operator!=(const TBox<T>& rhs) const
    {
        return Min!=rhs.Min || Max!=rhs.Max;
    }
};
//---------------------------------------------------------------------------
// Float bounding box
typedef TBox<float> ABox;
//---------------------------------------------------------------------------
// Box functions
namespace NBox
{
    // Convert box types
    template <typename T> inline TBox<float> ToSingle(const TBox<T>& src)
    {
        return TBox<float>(NVector::ToSingle(src.Min), NVector::ToSingle(src.Max));
    }
    template <typename T> inline TBox<double> ToDouble(const TBox<T>& src)
    {
        return TBox<double>(NVector::ToDouble(src.Min), NVector::ToDouble(src.Max));
    }

    // Returns the bounding box of a series of 3D points, returns false if the box cannot be calculated
    inline bool CalcPointsBoundingBox(const AVector::TAVectorArray& points, ABox& box)
    {
        // Make sure we have points
        if (points.IsEmpty()) return false;

        // Include all points
        box.Min = box.Max = points[0];
        for (unsigned i = 1; i < points.GetSize(); ++i)
        {
            box.Include(points[i]);
        }

        return true;
    }
#if 0
    // Returns the bounding box of a series of 3D points, assumes XYZ order
    template <typename T> inline bool TCalcPointsBoundingBox(const T* points, unsigned count, TBox<T>& box)
    {
        DEBUG_ASSERT(points, "null pointer for points was given");

        // Make sure we have points
        if (count < 1) return false;

        // Include all points
        box.Min = box.Max = TVector<T>(points[0], points[1], points[2]);
        for (unsigned i = 0; i < count; ++i)
        {
            box.Include(TVector<T>(points[0], points[1], points[2]));
            points += 3;
        }

        return true;
    }
#endif 0
}
//---------------------------------------------------------------------------
// Box hash function
namespace NHash
{
    template <typename T> inline unsigned Func(const TBox<T>& box)
    {
        return NHash::Raw((const char*)&box, sizeof(box));
    }
}
#endif
